<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\TripCost;
use App\Models\Trip;
use App\Models\Vehicle;
use App\Models\Driver;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CostReportController extends Controller
{
    /**
     * Display cost report with filters
     */
    public function index(Request $request)
    {
        $query = TripCost::with(['trip', 'trip.vehicle', 'trip.driver']);

        // Apply filters
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }
        if ($request->filled('trip_id')) {
            $query->where('trip_id', $request->trip_id);
        }
        if ($request->filled('vehicle_id')) {
            $query->whereHas('trip', function ($q) use ($request) {
                $q->where('vehicle_id', $request->vehicle_id);
            });
        }
        if ($request->filled('driver_id')) {
            $query->whereHas('trip', function ($q) use ($request) {
                $q->where('driver_id', $request->driver_id);
            });
        }

        // Get costs
        $costs = $query->orderBy('created_at', 'desc')->paginate(20)->withQueryString();

        // Calculate summary statistics
        $summaryQuery = TripCost::query();
        if ($request->filled('date_from')) {
            $summaryQuery->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $summaryQuery->whereDate('created_at', '<=', $request->date_to);
        }
        if ($request->filled('trip_id')) {
            $summaryQuery->where('trip_id', $request->trip_id);
        }
        if ($request->filled('vehicle_id')) {
            $summaryQuery->whereHas('trip', function ($q) use ($request) {
                $q->where('vehicle_id', $request->vehicle_id);
            });
        }
        if ($request->filled('driver_id')) {
            $summaryQuery->whereHas('trip', function ($q) use ($request) {
                $q->where('driver_id', $request->driver_id);
            });
        }

        $summary = [
            'total_cost' => $summaryQuery->sum('total_cost'),
            'total_base_cost' => $summaryQuery->sum('base_cost'),
            'total_toll_cost' => $summaryQuery->sum('toll_cost'),
            'total_fuel_cost' => $summaryQuery->sum('fuel_cost'),
            'total_driver_allowance' => $summaryQuery->sum('driver_allowance'),
            'total_other_costs' => $summaryQuery->sum('other_costs'),
            'total_records' => $summaryQuery->count(),
            'avg_cost' => $summaryQuery->avg('total_cost') ?? 0,
        ];

        // Get data for filters
        $trips = Trip::orderBy('trip_number', 'desc')->limit(100)->get();
        $vehicles = Vehicle::where('status', 'active')->orderBy('vehicle_number')->get();
        $drivers = Driver::where('status', 'active')->orderBy('name')->get();

        // Cost types (categories) - for filter dropdown compatibility
        $costTypes = collect(['base_cost', 'toll_cost', 'fuel_cost', 'driver_allowance', 'other_costs']);

        // Cost breakdown by type
        $costBreakdown = [
            ['type' => 'Base Cost', 'total' => $summary['total_base_cost'], 'color' => '#3b82f6'],
            ['type' => 'Toll Cost', 'total' => $summary['total_toll_cost'], 'color' => '#ef4444'],
            ['type' => 'Fuel Cost', 'total' => $summary['total_fuel_cost'], 'color' => '#f59e0b'],
            ['type' => 'Driver Allowance', 'total' => $summary['total_driver_allowance'], 'color' => '#10b981'],
            ['type' => 'Other Costs', 'total' => $summary['total_other_costs'], 'color' => '#8b5cf6'],
        ];

        // Alias for backward compatibility with view
        $byCostType = collect($costBreakdown)->map(function ($item) use ($summary) {
            return (object) [
                'cost_type' => $item['type'],
                'total_amount' => $item['total'],
                'count' => $summary['total_records'],
            ];
        });

        // By trip summary (top 10)
        $byTrip = TripCost::query()
            ->select('trip_id', DB::raw('SUM(total_cost) as total_amount'), DB::raw('COUNT(*) as count'))
            ->when($request->filled('date_from'), fn($q) => $q->whereDate('created_at', '>=', $request->date_from))
            ->when($request->filled('date_to'), fn($q) => $q->whereDate('created_at', '<=', $request->date_to))
            ->when($request->filled('vehicle_id'), fn($q) => $q->whereHas('trip', function ($sq) use ($request) {
                $sq->where('vehicle_id', $request->vehicle_id);
            }))
            ->when($request->filled('driver_id'), fn($q) => $q->whereHas('trip', function ($sq) use ($request) {
                $sq->where('driver_id', $request->driver_id);
            }))
            ->groupBy('trip_id')
            ->orderByDesc('total_amount')
            ->with('trip')
            ->limit(10)
            ->get();

        // Monthly trend (last 6 months)
        $monthlyTrend = TripCost::query()
            ->select(
                DB::raw('DATE_FORMAT(created_at, "%Y-%m") as month'),
                DB::raw('SUM(total_cost) as total_amount'),
                DB::raw('SUM(base_cost) as base_total'),
                DB::raw('SUM(toll_cost) as toll_total'),
                DB::raw('SUM(fuel_cost) as fuel_total'),
                DB::raw('SUM(driver_allowance) as allowance_total'),
                DB::raw('SUM(other_costs) as other_total'),
                DB::raw('COUNT(*) as count')
            )
            ->where('created_at', '>=', now()->subMonths(6))
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        return view('admin.reports.cost.index', compact(
            'costs',
            'summary',
            'trips',
            'vehicles',
            'drivers',
            'costTypes',
            'costBreakdown',
            'byCostType',
            'byTrip',
            'monthlyTrend'
        ));
    }

    /**
     * Export cost report
     */
    public function export(Request $request)
    {
        $format = $request->get('format', 'csv');

        $query = TripCost::with(['trip', 'trip.vehicle', 'trip.driver']);

        // Apply filters
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }
        if ($request->filled('trip_id')) {
            $query->where('trip_id', $request->trip_id);
        }
        if ($request->filled('vehicle_id')) {
            $query->whereHas('trip', function ($q) use ($request) {
                $q->where('vehicle_id', $request->vehicle_id);
            });
        }
        if ($request->filled('driver_id')) {
            $query->whereHas('trip', function ($q) use ($request) {
                $q->where('driver_id', $request->driver_id);
            });
        }

        $costs = $query->orderBy('created_at', 'desc')->get();

        // Calculate summary
        $summary = [
            'total_cost' => $costs->sum('total_cost'),
            'total_base_cost' => $costs->sum('base_cost'),
            'total_toll_cost' => $costs->sum('toll_cost'),
            'total_fuel_cost' => $costs->sum('fuel_cost'),
            'total_driver_allowance' => $costs->sum('driver_allowance'),
            'total_other_costs' => $costs->sum('other_costs'),
            'total_records' => $costs->count(),
            'avg_cost' => $costs->avg('total_cost') ?? 0,
        ];

        // Cost breakdown
        $costBreakdown = [
            ['type' => 'Base Cost', 'total' => $summary['total_base_cost']],
            ['type' => 'Toll Cost', 'total' => $summary['total_toll_cost']],
            ['type' => 'Fuel Cost', 'total' => $summary['total_fuel_cost']],
            ['type' => 'Driver Allowance', 'total' => $summary['total_driver_allowance']],
            ['type' => 'Other Costs', 'total' => $summary['total_other_costs']],
        ];

        $filename = 'cost_report_' . date('Y-m-d_His');

        switch ($format) {
            case 'pdf':
                return $this->exportPdf($costs, $summary, $costBreakdown, $filename, $request);
            case 'html':
                return $this->exportHtml($costs, $summary, $costBreakdown, $filename, $request);
            case 'excel':
            case 'csv':
            default:
                return $this->exportCsv($costs, $summary, $costBreakdown, $filename);
        }
    }

    /**
     * Export as CSV
     */
    private function exportCsv($costs, $summary, $costBreakdown, $filename)
    {
        $output = fopen('php://temp', 'r+');

        // BOM for Excel
        fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

        // Header
        fputcsv($output, ['COST ANALYSIS REPORT']);
        fputcsv($output, ['Generated: ' . now()->format('Y-m-d H:i:s')]);
        fputcsv($output, []);

        // Summary
        fputcsv($output, ['=== SUMMARY ===']);
        fputcsv($output, ['Total Cost ($)', number_format($summary['total_cost'], 2)]);
        fputcsv($output, ['Average Cost ($)', number_format($summary['avg_cost'], 2)]);
        fputcsv($output, ['Total Records', $summary['total_records']]);
        fputcsv($output, []);

        // By Cost Type
        fputcsv($output, ['=== COST BREAKDOWN ===']);
        fputcsv($output, ['Cost Type', 'Total Amount ($)']);
        foreach ($costBreakdown as $item) {
            fputcsv($output, [
                $item['type'],
                number_format($item['total'], 2),
            ]);
        }
        fputcsv($output, []);

        // Data
        fputcsv($output, ['=== COST DETAILS ===']);
        fputcsv($output, [
            'Date',
            'Trip Number',
            'Vehicle',
            'Driver',
            'Base Cost ($)',
            'Toll Cost ($)',
            'Fuel Cost ($)',
            'Driver Allowance ($)',
            'Other Costs ($)',
            'Total Cost ($)',
            'Notes'
        ]);

        foreach ($costs as $cost) {
            fputcsv($output, [
                $cost->created_at->format('Y-m-d'),
                $cost->trip->trip_number ?? '-',
                $cost->trip->vehicle->vehicle_number ?? '-',
                $cost->trip->driver->name ?? '-',
                number_format($cost->base_cost, 2),
                number_format($cost->toll_cost, 2),
                number_format($cost->fuel_cost, 2),
                number_format($cost->driver_allowance, 2),
                number_format($cost->other_costs, 2),
                number_format($cost->total_cost, 2),
                $cost->notes ?? '-',
            ]);
        }

        rewind($output);
        $content = stream_get_contents($output);
        fclose($output);

        return response($content)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '.csv"');
    }

    /**
     * Export as HTML
     */
    private function exportHtml($costs, $summary, $costBreakdown, $filename, $request)
    {
        $html = $this->generateHtml($costs, $summary, $costBreakdown, $request);

        return response($html)
            ->header('Content-Type', 'text/html')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '.html"');
    }

    /**
     * Export as PDF
     */
    private function exportPdf($costs, $summary, $costBreakdown, $filename, $request)
    {
        $html = $this->generateHtml($costs, $summary, $costBreakdown, $request);

        if (class_exists(\Barryvdh\DomPDF\Facade\Pdf::class)) {
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadHTML($html);
            $pdf->setPaper('A4', 'landscape');
            return $pdf->download($filename . '.pdf');
        } elseif (class_exists(\Dompdf\Dompdf::class)) {
            $dompdf = new \Dompdf\Dompdf();
            $dompdf->loadHtml($html);
            $dompdf->setPaper('A4', 'landscape');
            $dompdf->render();

            return response($dompdf->output())
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="' . $filename . '.pdf"');
        }

        return response($html)
            ->header('Content-Type', 'text/html')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '.html"');
    }

    /**
     * Generate HTML content for export
     */
    private function generateHtml($costs, $summary, $costBreakdown, $request)
    {
        $dateRange = '';
        if ($request->filled('date_from') || $request->filled('date_to')) {
            $dateRange = ($request->date_from ?? 'All') . ' to ' . ($request->date_to ?? 'All');
        } else {
            $dateRange = 'All Time';
        }

        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Cost Analysis Report</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 10px; color: #333; padding: 20px; }
        .header { text-align: center; margin-bottom: 20px; border-bottom: 2px solid #dc2626; padding-bottom: 15px; }
        .header h1 { color: #dc2626; font-size: 22px; margin-bottom: 5px; }
        .header .subtitle { color: #6b7280; font-size: 12px; }
        .meta { margin-bottom: 20px; background: #fef2f2; padding: 10px; border-radius: 5px; }
        .summary { margin-bottom: 20px; background: #f3f4f6; padding: 15px; border-radius: 5px; }
        .section-title { font-size: 14px; font-weight: bold; color: #374151; margin-bottom: 10px; border-bottom: 1px solid #e5e7eb; padding-bottom: 5px; }
        table { width: 100%; border-collapse: collapse; }
        th { background: #dc2626; color: white; padding: 6px; text-align: left; font-size: 9px; }
        td { padding: 5px 6px; border-bottom: 1px solid #e5e7eb; font-size: 9px; }
        tr:nth-child(even) { background: #fef2f2; }
        .summary-table td { padding: 10px; text-align: center; background: white; border-radius: 5px; }
        .summary-value { font-size: 16px; font-weight: bold; color: #dc2626; }
        .summary-label { font-size: 8px; color: #6b7280; text-transform: uppercase; }
        .footer { margin-top: 20px; text-align: center; color: #6b7280; font-size: 9px; border-top: 1px solid #e5e7eb; padding-top: 10px; }
        .text-right { text-align: right; }
        .breakdown-table th { background: #7c3aed; }
        .breakdown-table tr:nth-child(even) { background: #f5f3ff; }
    </style>
</head>
<body>
    <div class="header">
        <h1>Cost Analysis Report</h1>
        <div class="subtitle">LogiFlow Fleet Management System</div>
    </div>
    
    <div class="meta">
        <table style="width: 100%;">
            <tr>
                <td style="text-align: center; padding: 5px;">
                    <div style="font-size: 9px; color: #6b7280;">Date Range</div>
                    <div style="font-weight: bold; color: #b91c1c;">' . $dateRange . '</div>
                </td>
                <td style="text-align: center; padding: 5px;">
                    <div style="font-size: 9px; color: #6b7280;">Generated</div>
                    <div style="font-weight: bold; color: #b91c1c;">' . now()->format('M d, Y H:i') . '</div>
                </td>
                <td style="text-align: center; padding: 5px;">
                    <div style="font-size: 9px; color: #6b7280;">Total Records</div>
                    <div style="font-weight: bold; color: #b91c1c;">' . $summary['total_records'] . '</div>
                </td>
            </tr>
        </table>
    </div>

    <div class="summary">
        <table class="summary-table" style="width: 100%;">
            <tr>
                <td>
                    <div class="summary-value">$' . number_format($summary['total_cost'], 2) . '</div>
                    <div class="summary-label">Total Cost</div>
                </td>
                <td>
                    <div class="summary-value">$' . number_format($summary['avg_cost'], 2) . '</div>
                    <div class="summary-label">Average Cost</div>
                </td>
                <td>
                    <div class="summary-value">' . $summary['total_records'] . '</div>
                    <div class="summary-label">Total Entries</div>
                </td>
            </tr>
        </table>
    </div>

    <div class="section-title">Cost Breakdown by Type</div>
    <table class="breakdown-table" style="margin-bottom: 20px;">
        <thead>
            <tr>
                <th>Cost Type</th>
                <th class="text-right">Total Amount</th>
                <th class="text-right">% of Total</th>
            </tr>
        </thead>
        <tbody>';

        foreach ($costBreakdown as $item) {
            $percentage = $summary['total_cost'] > 0
                ? round(($item['total'] / $summary['total_cost']) * 100, 1)
                : 0;
            $html .= '<tr>
                <td>' . $item['type'] . '</td>
                <td class="text-right">$' . number_format($item['total'], 2) . '</td>
                <td class="text-right">' . $percentage . '%</td>
            </tr>';
        }

        $html .= '</tbody>
    </table>

    <div class="section-title">Cost Details</div>
    <table>
        <thead>
            <tr>
                <th>Date</th>
                <th>Trip #</th>
                <th>Vehicle</th>
                <th>Driver</th>
                <th class="text-right">Base</th>
                <th class="text-right">Toll</th>
                <th class="text-right">Fuel</th>
                <th class="text-right">Allowance</th>
                <th class="text-right">Other</th>
                <th class="text-right">Total</th>
            </tr>
        </thead>
        <tbody>';

        foreach ($costs as $cost) {
            $html .= '<tr>
                <td>' . $cost->created_at->format('Y-m-d') . '</td>
                <td>' . htmlspecialchars($cost->trip->trip_number ?? '-') . '</td>
                <td>' . htmlspecialchars($cost->trip->vehicle->vehicle_number ?? '-') . '</td>
                <td>' . htmlspecialchars($cost->trip->driver->name ?? '-') . '</td>
                <td class="text-right">$' . number_format($cost->base_cost, 2) . '</td>
                <td class="text-right">$' . number_format($cost->toll_cost, 2) . '</td>
                <td class="text-right">$' . number_format($cost->fuel_cost, 2) . '</td>
                <td class="text-right">$' . number_format($cost->driver_allowance, 2) . '</td>
                <td class="text-right">$' . number_format($cost->other_costs, 2) . '</td>
                <td class="text-right"><strong>$' . number_format($cost->total_cost, 2) . '</strong></td>
            </tr>';
        }

        $html .= '</tbody>
        <tfoot>
            <tr style="background: #dc2626; color: white; font-weight: bold;">
                <td colspan="4">TOTALS</td>
                <td class="text-right">$' . number_format($summary['total_base_cost'], 2) . '</td>
                <td class="text-right">$' . number_format($summary['total_toll_cost'], 2) . '</td>
                <td class="text-right">$' . number_format($summary['total_fuel_cost'], 2) . '</td>
                <td class="text-right">$' . number_format($summary['total_driver_allowance'], 2) . '</td>
                <td class="text-right">$' . number_format($summary['total_other_costs'], 2) . '</td>
                <td class="text-right">$' . number_format($summary['total_cost'], 2) . '</td>
            </tr>
        </tfoot>
    </table>
    
    <div class="footer">
        <p>Generated by LogiFlow Fleet Management System on ' . now()->format('F d, Y \a\t h:i A') . '</p>
    </div>
</body>
</html>';

        return $html;
    }
}
