<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\FuelLog;
use App\Models\Vehicle;
use App\Models\Driver;
use App\Models\Trip;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class FuelLogController extends Controller
{
    /**
     * Display a listing of fuel logs
     */
    public function index(Request $request)
    {
        $query = FuelLog::with(['vehicle.vehicleType', 'driver', 'creator']);

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('fuel_number', 'like', "%{$search}%")
                    ->orWhere('fuel_station', 'like', "%{$search}%")
                    ->orWhere('receipt_number', 'like', "%{$search}%")
                    ->orWhereHas('vehicle', function ($vq) use ($search) {
                        $vq->where('vehicle_number', 'like', "%{$search}%");
                    });
            });
        }

        // Vehicle filter
        if ($request->filled('vehicle_id')) {
            $query->where('vehicle_id', $request->vehicle_id);
        }

        // Driver filter
        if ($request->filled('driver_id')) {
            $query->where('driver_id', $request->driver_id);
        }

        // Fuel type filter
        if ($request->filled('fuel_type')) {
            $query->where('fuel_type', $request->fuel_type);
        }

        // Payment method filter
        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        // Date range filter
        if ($request->filled('date_from')) {
            $query->where('fill_date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->where('fill_date', '<=', $request->date_to);
        }

        // Get stats
        $stats = [
            'total_logs' => FuelLog::count(),
            'this_month_cost' => FuelLog::thisMonth()->sum('total_cost'),
            'this_month_quantity' => FuelLog::thisMonth()->sum('quantity'),
            'avg_efficiency' => FuelLog::whereNotNull('fuel_efficiency')
                ->where('fuel_efficiency', '>', 0)
                ->avg('fuel_efficiency'),
        ];

        $fuelLogs = $query->orderBy('fill_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(15)
            ->withQueryString();

        $vehicles = Vehicle::where('status', 'active')->orderBy('vehicle_number')->get();
        $drivers = Driver::where('status', 'active')->orderBy('name')->get();

        return view('admin.fuel-logs.index', compact('fuelLogs', 'vehicles', 'drivers', 'stats'));
    }

    /**
     * Show the form for creating a new fuel log
     */
    public function create()
    {
        $vehicles = Vehicle::where('status', 'active')
            ->with('vehicleType')
            ->orderBy('vehicle_number')
            ->get();
        $drivers = Driver::where('status', 'active')->orderBy('name')->get();
        $trips = Trip::whereIn('status', ['assigned', 'in_transit'])
            ->with(['vehicle', 'driver'])
            ->orderBy('created_at', 'desc')
            ->get();

        $suggestedNumber = FuelLog::generateFuelNumber();

        return view('admin.fuel-logs.create', compact('vehicles', 'drivers', 'trips', 'suggestedNumber'));
    }

    /**
     * Store a newly created fuel log
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'fuel_number' => 'required|string|max:50|unique:fuel_logs,fuel_number',
            'vehicle_id' => 'required|exists:vehicles,id',
            'driver_id' => 'nullable|exists:drivers,id',
            'fuel_type' => 'required|in:diesel,petrol,cng,electric',
            'quantity' => 'required|numeric|min:0.01|max:9999.99',
            'price_per_unit' => 'required|numeric|min:0.01|max:999.99',
            'odometer_reading' => 'nullable|numeric|min:0',
            'fill_date' => 'required|date',
            'fill_time' => 'nullable|date_format:H:i',
            'is_full_tank' => 'nullable|boolean',
            'fuel_station' => 'nullable|string|max:255',
            'station_location' => 'nullable|string|max:255',
            'receipt_number' => 'nullable|string|max:100',
            'payment_method' => 'required|in:cash,card,company_account,fuel_card',
            'trip_id' => 'nullable|exists:trips,id',
            'notes' => 'nullable|string|max:1000',
        ], [
            'fuel_number.required' => __('Fuel log number is required'),
            'fuel_number.unique' => __('This fuel log number already exists'),
            'vehicle_id.required' => __('Please select a vehicle'),
            'fuel_type.required' => __('Please select fuel type'),
            'quantity.required' => __('Quantity is required'),
            'quantity.min' => __('Quantity must be greater than 0'),
            'price_per_unit.required' => __('Price per unit is required'),
            'fill_date.required' => __('Fill date is required'),
            'payment_method.required' => __('Please select payment method'),
        ]);

        // Calculate total cost
        $validated['total_cost'] = $validated['quantity'] * $validated['price_per_unit'];
        $validated['is_full_tank'] = $request->has('is_full_tank');
        $validated['created_by'] = Auth::id();

        // Get previous odometer for efficiency calculation
        $previousOdometer = FuelLog::getPreviousOdometer($validated['vehicle_id']);
        if ($previousOdometer && $validated['odometer_reading']) {
            $validated['previous_odometer'] = $previousOdometer;
            $validated['distance_traveled'] = $validated['odometer_reading'] - $previousOdometer;

            // Calculate efficiency if full tank
            if ($validated['is_full_tank'] && $validated['distance_traveled'] > 0) {
                $validated['fuel_efficiency'] = round($validated['distance_traveled'] / $validated['quantity'], 2);
            }
        }

        try {
            $fuelLog = FuelLog::create($validated);

            Log::info("Fuel log created: {$fuelLog->fuel_number} by user " . Auth::id());

            return redirect()->route('admin.fuel-logs.index')
                ->with('success', __('Fuel log created successfully'));
        } catch (\Exception $e) {
            Log::error("Error creating fuel log: " . $e->getMessage());
            return back()->withInput()->with('error', __('Error creating fuel log'));
        }
    }

    /**
     * Display the specified fuel log
     */
    public function show(FuelLog $fuelLog)
    {
        $fuelLog->load(['vehicle.vehicleType', 'driver', 'trip', 'creator']);

        // Get vehicle fuel history summary
        $vehicleStats = [
            'total_cost' => FuelLog::getTotalCostByVehicle($fuelLog->vehicle_id),
            'total_quantity' => FuelLog::getTotalQuantityByVehicle($fuelLog->vehicle_id),
            'avg_efficiency' => FuelLog::getAverageEfficiencyByVehicle($fuelLog->vehicle_id),
            'fill_count' => FuelLog::where('vehicle_id', $fuelLog->vehicle_id)->count(),
        ];

        return view('admin.fuel-logs.show', compact('fuelLog', 'vehicleStats'));
    }

    /**
     * Show the form for editing the specified fuel log
     */
    public function edit(FuelLog $fuelLog)
    {
        $vehicles = Vehicle::where('status', 'active')
            ->with('vehicleType')
            ->orderBy('vehicle_number')
            ->get();
        $drivers = Driver::where('status', 'active')->orderBy('name')->get();
        $trips = Trip::whereIn('status', ['assigned', 'in_transit', 'delivered'])
            ->with(['vehicle', 'driver'])
            ->orderBy('created_at', 'desc')
            ->get();

        return view('admin.fuel-logs.edit', compact('fuelLog', 'vehicles', 'drivers', 'trips'));
    }

    /**
     * Update the specified fuel log
     */
    public function update(Request $request, FuelLog $fuelLog)
    {
        $validated = $request->validate([
            'fuel_number' => 'required|string|max:50|unique:fuel_logs,fuel_number,' . $fuelLog->id,
            'vehicle_id' => 'required|exists:vehicles,id',
            'driver_id' => 'nullable|exists:drivers,id',
            'fuel_type' => 'required|in:diesel,petrol,cng,electric',
            'quantity' => 'required|numeric|min:0.01|max:9999.99',
            'price_per_unit' => 'required|numeric|min:0.01|max:999.99',
            'odometer_reading' => 'nullable|numeric|min:0',
            'fill_date' => 'required|date',
            'fill_time' => 'nullable|date_format:H:i',
            'is_full_tank' => 'nullable|boolean',
            'fuel_station' => 'nullable|string|max:255',
            'station_location' => 'nullable|string|max:255',
            'receipt_number' => 'nullable|string|max:100',
            'payment_method' => 'required|in:cash,card,company_account,fuel_card',
            'trip_id' => 'nullable|exists:trips,id',
            'notes' => 'nullable|string|max:1000',
        ], [
            'fuel_number.required' => __('Fuel log number is required'),
            'fuel_number.unique' => __('This fuel log number already exists'),
            'vehicle_id.required' => __('Please select a vehicle'),
            'fuel_type.required' => __('Please select fuel type'),
            'quantity.required' => __('Quantity is required'),
            'price_per_unit.required' => __('Price per unit is required'),
            'fill_date.required' => __('Fill date is required'),
            'payment_method.required' => __('Please select payment method'),
        ]);

        // Calculate total cost
        $validated['total_cost'] = $validated['quantity'] * $validated['price_per_unit'];
        $validated['is_full_tank'] = $request->has('is_full_tank');

        // Recalculate efficiency if vehicle or odometer changed
        if ($validated['vehicle_id'] != $fuelLog->vehicle_id || $validated['odometer_reading'] != $fuelLog->odometer_reading) {
            $previousOdometer = FuelLog::getPreviousOdometer($validated['vehicle_id'], $fuelLog->id);
            if ($previousOdometer && $validated['odometer_reading']) {
                $validated['previous_odometer'] = $previousOdometer;
                $validated['distance_traveled'] = $validated['odometer_reading'] - $previousOdometer;

                if ($validated['is_full_tank'] && $validated['distance_traveled'] > 0) {
                    $validated['fuel_efficiency'] = round($validated['distance_traveled'] / $validated['quantity'], 2);
                } else {
                    $validated['fuel_efficiency'] = null;
                }
            } else {
                $validated['previous_odometer'] = null;
                $validated['distance_traveled'] = null;
                $validated['fuel_efficiency'] = null;
            }
        }

        try {
            $fuelLog->update($validated);

            Log::info("Fuel log updated: {$fuelLog->fuel_number} by user " . Auth::id());

            return redirect()->route('admin.fuel-logs.index')
                ->with('success', __('Fuel log updated successfully'));
        } catch (\Exception $e) {
            Log::error("Error updating fuel log: " . $e->getMessage());
            return back()->withInput()->with('error', __('Error updating fuel log'));
        }
    }

    /**
     * Remove the specified fuel log
     */
    public function destroy(FuelLog $fuelLog)
    {
        try {
            $fuelNumber = $fuelLog->fuel_number;
            $fuelLog->delete();

            Log::info("Fuel log deleted: {$fuelNumber} by user " . Auth::id());

            return redirect()->route('admin.fuel-logs.index')
                ->with('success', __('Fuel log deleted successfully'));
        } catch (\Exception $e) {
            Log::error("Error deleting fuel log: " . $e->getMessage());
            return back()->with('error', __('Error deleting fuel log'));
        }
    }

    /**
     * Show vehicle fuel history
     */
    public function vehicleHistory(Vehicle $vehicle)
    {
        $fuelLogs = FuelLog::where('vehicle_id', $vehicle->id)
            ->with(['driver', 'creator'])
            ->orderBy('fill_date', 'desc')
            ->orderBy('id', 'desc')
            ->paginate(20);

        // Vehicle fuel stats
        $stats = [
            'total_cost' => FuelLog::getTotalCostByVehicle($vehicle->id),
            'total_quantity' => FuelLog::getTotalQuantityByVehicle($vehicle->id),
            'avg_efficiency' => FuelLog::getAverageEfficiencyByVehicle($vehicle->id),
            'fill_count' => FuelLog::where('vehicle_id', $vehicle->id)->count(),
            'this_month_cost' => FuelLog::where('vehicle_id', $vehicle->id)->thisMonth()->sum('total_cost'),
        ];

        return view('admin.fuel-logs.vehicle-history', compact('vehicle', 'fuelLogs', 'stats'));
    }

    /**
     * Get previous odometer via AJAX
     */
    public function getPreviousOdometer(Request $request)
    {
        $vehicleId = $request->input('vehicle_id');
        $excludeId = $request->input('exclude_id');

        $previousOdometer = FuelLog::getPreviousOdometer($vehicleId, $excludeId);

        return response()->json([
            'previous_odometer' => $previousOdometer,
        ]);
    }
}
