<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\FuelLog;
use App\Models\Vehicle;
use App\Models\Driver;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class FuelReportController extends Controller
{
    /**
     * Display fuel report with filters
     */
    public function index(Request $request)
    {
        $query = FuelLog::with(['vehicle', 'driver']);

        // Apply filters
        if ($request->filled('date_from')) {
            $query->whereDate('fill_date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('fill_date', '<=', $request->date_to);
        }
        if ($request->filled('vehicle_id')) {
            $query->where('vehicle_id', $request->vehicle_id);
        }
        if ($request->filled('driver_id')) {
            $query->where('driver_id', $request->driver_id);
        }
        if ($request->filled('fuel_type')) {
            $query->where('fuel_type', $request->fuel_type);
        }

        // Get fuel logs
        $fuelLogs = $query->orderBy('fill_date', 'desc')->paginate(20)->withQueryString();

        // Calculate summary statistics
        $summaryQuery = FuelLog::query();
        if ($request->filled('date_from')) {
            $summaryQuery->whereDate('fill_date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $summaryQuery->whereDate('fill_date', '<=', $request->date_to);
        }
        if ($request->filled('vehicle_id')) {
            $summaryQuery->where('vehicle_id', $request->vehicle_id);
        }
        if ($request->filled('driver_id')) {
            $summaryQuery->where('driver_id', $request->driver_id);
        }
        if ($request->filled('fuel_type')) {
            $summaryQuery->where('fuel_type', $request->fuel_type);
        }

        $summary = [
            'total_quantity' => $summaryQuery->sum('quantity'),
            'total_cost' => $summaryQuery->sum('total_cost'),
            'total_records' => $summaryQuery->count(),
            'avg_price' => $summaryQuery->avg('price_per_unit') ?? 0,
        ];

        // Get data for filters
        $vehicles = Vehicle::where('status', 'active')->orderBy('vehicle_number')->get();
        $drivers = Driver::where('status', 'active')->orderBy('name')->get();

        // By vehicle summary
        $byVehicle = FuelLog::query()
            ->select('vehicle_id', DB::raw('SUM(quantity) as total_quantity'), DB::raw('SUM(total_cost) as total_cost'), DB::raw('COUNT(*) as fill_count'))
            ->when($request->filled('date_from'), fn($q) => $q->whereDate('fill_date', '>=', $request->date_from))
            ->when($request->filled('date_to'), fn($q) => $q->whereDate('fill_date', '<=', $request->date_to))
            ->when($request->filled('vehicle_id'), fn($q) => $q->where('vehicle_id', $request->vehicle_id))
            ->when($request->filled('driver_id'), fn($q) => $q->where('driver_id', $request->driver_id))
            ->when($request->filled('fuel_type'), fn($q) => $q->where('fuel_type', $request->fuel_type))
            ->groupBy('vehicle_id')
            ->with('vehicle')
            ->get();

        // By fuel type summary
        $byFuelType = FuelLog::query()
            ->select('fuel_type', DB::raw('SUM(quantity) as total_quantity'), DB::raw('SUM(total_cost) as total_cost'), DB::raw('COUNT(*) as fill_count'))
            ->when($request->filled('date_from'), fn($q) => $q->whereDate('fill_date', '>=', $request->date_from))
            ->when($request->filled('date_to'), fn($q) => $q->whereDate('fill_date', '<=', $request->date_to))
            ->when($request->filled('vehicle_id'), fn($q) => $q->where('vehicle_id', $request->vehicle_id))
            ->when($request->filled('driver_id'), fn($q) => $q->where('driver_id', $request->driver_id))
            ->when($request->filled('fuel_type'), fn($q) => $q->where('fuel_type', $request->fuel_type))
            ->groupBy('fuel_type')
            ->get();

        return view('admin.reports.fuel.index', compact(
            'fuelLogs',
            'summary',
            'vehicles',
            'drivers',
            'byVehicle',
            'byFuelType'
        ));
    }

    /**
     * Export fuel report
     */
    public function export(Request $request)
    {
        $format = $request->get('format', 'csv');

        $query = FuelLog::with(['vehicle', 'driver']);

        // Apply filters
        if ($request->filled('date_from')) {
            $query->whereDate('fill_date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('fill_date', '<=', $request->date_to);
        }
        if ($request->filled('vehicle_id')) {
            $query->where('vehicle_id', $request->vehicle_id);
        }
        if ($request->filled('driver_id')) {
            $query->where('driver_id', $request->driver_id);
        }
        if ($request->filled('fuel_type')) {
            $query->where('fuel_type', $request->fuel_type);
        }

        $fuelLogs = $query->orderBy('fill_date', 'desc')->get();

        // Calculate summary
        $summary = [
            'total_quantity' => $fuelLogs->sum('quantity'),
            'total_cost' => $fuelLogs->sum('total_cost'),
            'total_records' => $fuelLogs->count(),
            'avg_price' => $fuelLogs->avg('price_per_unit') ?? 0,
        ];

        $filename = 'fuel_report_' . date('Y-m-d_His');

        switch ($format) {
            case 'pdf':
                return $this->exportPdf($fuelLogs, $summary, $filename, $request);
            case 'html':
                return $this->exportHtml($fuelLogs, $summary, $filename, $request);
            case 'excel':
            case 'csv':
            default:
                return $this->exportCsv($fuelLogs, $summary, $filename);
        }
    }

    /**
     * Export as CSV
     */
    private function exportCsv($fuelLogs, $summary, $filename)
    {
        $output = fopen('php://temp', 'r+');

        // BOM for Excel
        fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

        // Header
        fputcsv($output, ['FUEL CONSUMPTION REPORT']);
        fputcsv($output, ['Generated: ' . now()->format('Y-m-d H:i:s')]);
        fputcsv($output, []);

        // Summary
        fputcsv($output, ['=== SUMMARY ===']);
        fputcsv($output, ['Total Quantity (L)', number_format($summary['total_quantity'], 2)]);
        fputcsv($output, ['Total Cost ($)', number_format($summary['total_cost'], 2)]);
        fputcsv($output, ['Average Price/Unit ($)', number_format($summary['avg_price'], 2)]);
        fputcsv($output, ['Total Records', $summary['total_records']]);
        fputcsv($output, []);

        // Data
        fputcsv($output, ['=== FUEL LOG DETAILS ===']);
        fputcsv($output, ['Date', 'Vehicle', 'Driver', 'Fuel Type', 'Quantity (L)', 'Price/Unit ($)', 'Total Cost ($)', 'Odometer', 'Station']);

        foreach ($fuelLogs as $log) {
            fputcsv($output, [
                $log->fill_date,
                $log->vehicle->vehicle_number ?? '-',
                $log->driver->name ?? '-',
                ucfirst($log->fuel_type),
                number_format($log->quantity, 2),
                number_format($log->price_per_unit, 2),
                number_format($log->total_cost, 2),
                $log->odometer_reading ?? '-',
                $log->fuel_station ?? '-',
            ]);
        }

        rewind($output);
        $content = stream_get_contents($output);
        fclose($output);

        return response($content)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '.csv"');
    }

    /**
     * Export as HTML
     */
    private function exportHtml($fuelLogs, $summary, $filename, $request)
    {
        $html = $this->generateHtml($fuelLogs, $summary, $request);

        return response($html)
            ->header('Content-Type', 'text/html')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '.html"');
    }

    /**
     * Export as PDF
     */
    private function exportPdf($fuelLogs, $summary, $filename, $request)
    {
        $html = $this->generateHtml($fuelLogs, $summary, $request);

        if (class_exists(\Barryvdh\DomPDF\Facade\Pdf::class)) {
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadHTML($html);
            $pdf->setPaper('A4', 'landscape');
            return $pdf->download($filename . '.pdf');
        } elseif (class_exists(\Dompdf\Dompdf::class)) {
            $dompdf = new \Dompdf\Dompdf();
            $dompdf->loadHtml($html);
            $dompdf->setPaper('A4', 'landscape');
            $dompdf->render();

            return response($dompdf->output())
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="' . $filename . '.pdf"');
        }

        // Fallback to HTML
        return response($html)
            ->header('Content-Type', 'text/html')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '.html"');
    }

    /**
     * Generate HTML content for export
     */
    private function generateHtml($fuelLogs, $summary, $request)
    {
        $dateRange = '';
        if ($request->filled('date_from') || $request->filled('date_to')) {
            $dateRange = ($request->date_from ?? 'All') . ' to ' . ($request->date_to ?? 'All');
        } else {
            $dateRange = 'All Time';
        }

        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Fuel Consumption Report</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 11px; color: #333; padding: 20px; }
        .header { text-align: center; margin-bottom: 20px; border-bottom: 2px solid #f97316; padding-bottom: 15px; }
        .header h1 { color: #f97316; font-size: 22px; margin-bottom: 5px; }
        .header .subtitle { color: #6b7280; font-size: 12px; }
        .meta { margin-bottom: 20px; background: #fff7ed; padding: 10px; border-radius: 5px; }
        .meta-row { display: flex; justify-content: space-between; }
        .meta-item { text-align: center; padding: 5px 15px; }
        .meta-label { font-size: 9px; color: #6b7280; text-transform: uppercase; }
        .meta-value { font-size: 14px; font-weight: bold; color: #ea580c; }
        .summary { display: flex; justify-content: space-around; margin-bottom: 20px; background: #f3f4f6; padding: 15px; border-radius: 5px; }
        .summary-item { text-align: center; }
        .summary-value { font-size: 18px; font-weight: bold; color: #f97316; }
        .summary-label { font-size: 9px; color: #6b7280; text-transform: uppercase; }
        table { width: 100%; border-collapse: collapse; margin-top: 15px; }
        th { background: #f97316; color: white; padding: 8px; text-align: left; font-size: 10px; }
        td { padding: 6px 8px; border-bottom: 1px solid #e5e7eb; font-size: 10px; }
        tr:nth-child(even) { background: #fef3e2; }
        .footer { margin-top: 20px; text-align: center; color: #6b7280; font-size: 9px; border-top: 1px solid #e5e7eb; padding-top: 10px; }
        .text-right { text-align: right; }
    </style>
</head>
<body>
    <div class="header">
        <h1>Fuel Consumption Report</h1>
        <div class="subtitle">LogiFleet Fleet Management System</div>
    </div>
    
    <div class="meta">
        <table style="width: 100%;">
            <tr>
                <td style="text-align: center; padding: 5px;">
                    <div class="meta-label">Date Range</div>
                    <div class="meta-value">' . $dateRange . '</div>
                </td>
                <td style="text-align: center; padding: 5px;">
                    <div class="meta-label">Generated</div>
                    <div class="meta-value">' . now()->format('M d, Y H:i') . '</div>
                </td>
                <td style="text-align: center; padding: 5px;">
                    <div class="meta-label">Total Records</div>
                    <div class="meta-value">' . $summary['total_records'] . '</div>
                </td>
            </tr>
        </table>
    </div>

    <div class="summary">
        <table style="width: 100%;">
            <tr>
                <td style="text-align: center; padding: 10px; background: white; border-radius: 5px; margin: 0 5px;">
                    <div class="summary-value">' . number_format($summary['total_quantity'], 2) . ' L</div>
                    <div class="summary-label">Total Quantity</div>
                </td>
                <td style="text-align: center; padding: 10px; background: white; border-radius: 5px; margin: 0 5px;">
                    <div class="summary-value">$' . number_format($summary['total_cost'], 2) . '</div>
                    <div class="summary-label">Total Cost</div>
                </td>
                <td style="text-align: center; padding: 10px; background: white; border-radius: 5px; margin: 0 5px;">
                    <div class="summary-value">$' . number_format($summary['avg_price'], 2) . '</div>
                    <div class="summary-label">Avg Price/Unit</div>
                </td>
            </tr>
        </table>
    </div>

    <h3 style="margin-bottom: 10px; color: #374151;">Fuel Log Details</h3>
    <table>
        <thead>
            <tr>
                <th>Date</th>
                <th>Vehicle</th>
                <th>Driver</th>
                <th>Fuel Type</th>
                <th class="text-right">Quantity (L)</th>
                <th class="text-right">Price/Unit</th>
                <th class="text-right">Total Cost</th>
                <th>Odometer</th>
                <th>Station</th>
            </tr>
        </thead>
        <tbody>';

        foreach ($fuelLogs as $log) {
            $html .= '<tr>
                <td>' . $log->fill_date . '</td>
                <td>' . htmlspecialchars($log->vehicle->vehicle_number ?? '-') . '</td>
                <td>' . htmlspecialchars($log->driver->name ?? '-') . '</td>
                <td>' . ucfirst($log->fuel_type) . '</td>
                <td class="text-right">' . number_format($log->quantity, 2) . '</td>
                <td class="text-right">$' . number_format($log->price_per_unit, 2) . '</td>
                <td class="text-right">$' . number_format($log->total_cost, 2) . '</td>
                <td>' . ($log->odometer_reading ?? '-') . '</td>
                <td>' . htmlspecialchars($log->fuel_station ?? '-') . '</td>
            </tr>';
        }

        $html .= '</tbody>
    </table>
    
    <div class="footer">
        <p>Generated by LogiFleet Fleet Management System on ' . now()->format('F d, Y \a\t h:i A') . '</p>
    </div>
</body>
</html>';

        return $html;
    }
}
