<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\VehicleMaintenance;
use App\Models\MaintenanceType;
use App\Models\MaintenancePart;
use App\Models\Vehicle;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class MaintenanceController extends Controller
{
    public function index(Request $request)
    {
        $query = VehicleMaintenance::with(['vehicle', 'maintenanceType']);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('maintenance_number', 'like', "%{$search}%")
                    ->orWhere('vendor_name', 'like', "%{$search}%")
                    ->orWhereHas('vehicle', function ($vq) use ($search) {
                        $vq->where('vehicle_number', 'like', "%{$search}%");
                    });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('priority')) {
            $query->where('priority', $request->priority);
        }

        if ($request->filled('vehicle_id')) {
            $query->where('vehicle_id', $request->vehicle_id);
        }

        $maintenances = $query->latest('scheduled_date')->paginate(15);

        $stats = [
            'total' => VehicleMaintenance::count(),
            'scheduled' => VehicleMaintenance::scheduled()->count(),
            'in_progress' => VehicleMaintenance::where('status', 'in_progress')->count(),
            'completed' => VehicleMaintenance::completed()->count(),
            'overdue' => VehicleMaintenance::overdue()->count(),
            'total_cost_this_month' => VehicleMaintenance::completed()
                ->whereMonth('completed_date', now()->month)
                ->whereYear('completed_date', now()->year)
                ->sum(DB::raw('labor_cost + parts_cost')),
        ];

        $vehicles = Vehicle::where('status', 'active')->get();
        $maintenanceTypes = MaintenanceType::active()->get();

        return view('admin.maintenances.index', compact(
            'maintenances',
            'stats',
            'vehicles',
            'maintenanceTypes'
        ));
    }

    public function create(Request $request)
    {
        $vehicles = Vehicle::with('vehicleType')->get();
        $maintenanceTypes = MaintenanceType::active()->get();
        $selectedVehicle = $request->vehicle_id ? Vehicle::find($request->vehicle_id) : null;

        return view('admin.maintenances.create', compact(
            'vehicles',
            'maintenanceTypes',
            'selectedVehicle'
        ));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'vehicle_id' => 'required|exists:vehicles,id',
            'maintenance_type_id' => 'required|exists:maintenance_types,id',
            'status' => 'required|in:scheduled,in_progress,completed,cancelled',
            'priority' => 'required|in:low,medium,high,urgent',
            'scheduled_date' => 'required|date',
            'completed_date' => 'nullable|date|after_or_equal:scheduled_date',
            'odometer_reading' => 'nullable|integer|min:0',
            'next_due_km' => 'nullable|integer|min:0',
            'next_due_date' => 'nullable|date|after:scheduled_date',
            'labor_cost' => 'nullable|numeric|min:0',
            'parts_cost' => 'nullable|numeric|min:0',
            'vendor_name' => 'nullable|string|max:255',
            'vendor_contact' => 'nullable|string|max:100',
            'invoice_number' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'notes' => 'nullable|string',
            'performed_by' => 'nullable|string|max:255',
            'parts' => 'nullable|array',
            'parts.*.part_name' => 'required_with:parts|string|max:255',
            'parts.*.part_number' => 'nullable|string|max:100',
            'parts.*.quantity' => 'required_with:parts|integer|min:1',
            'parts.*.unit_price' => 'required_with:parts|numeric|min:0',
        ]);

        $validated['cost'] = ($validated['labor_cost'] ?? 0) + ($validated['parts_cost'] ?? 0);
        $validated['created_by'] = auth()->id();

        // Auto-calculate next due based on maintenance type
        $maintenanceType = MaintenanceType::find($validated['maintenance_type_id']);

        if ($validated['status'] === 'completed' && $validated['completed_date']) {
            if (!isset($validated['next_due_date']) && $maintenanceType->default_interval_days) {
                $validated['next_due_date'] = \Carbon\Carbon::parse($validated['completed_date'])
                    ->addDays($maintenanceType->default_interval_days);
            }

            if (!isset($validated['next_due_km']) && $maintenanceType->default_interval_km && isset($validated['odometer_reading'])) {
                $validated['next_due_km'] = $validated['odometer_reading'] + $maintenanceType->default_interval_km;
            }
        }

        DB::beginTransaction();
        try {
            $maintenance = VehicleMaintenance::create($validated);

            // Create parts if provided
            if ($request->has('parts') && is_array($request->parts)) {
                foreach ($request->parts as $part) {
                    if (!empty($part['part_name'])) {
                        $maintenance->parts()->create([
                            'part_name' => $part['part_name'],
                            'part_number' => $part['part_number'] ?? null,
                            'quantity' => $part['quantity'],
                            'unit_price' => $part['unit_price'],
                        ]);
                    }
                }

                // Update parts_cost from actual parts
                $totalPartsCost = $maintenance->parts()->sum('total_price');
                $maintenance->update([
                    'parts_cost' => $totalPartsCost,
                    'cost' => ($validated['labor_cost'] ?? 0) + $totalPartsCost,
                ]);
            }

            // Update vehicle status if maintenance is in progress
            if ($validated['status'] === 'in_progress') {
                Vehicle::find($validated['vehicle_id'])->update(['status' => 'maintenance']);
            }

            DB::commit();

            return redirect()->route('admin.maintenances.show', $maintenance)
                ->with('success', __('created_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', __('error_occurred') . ': ' . $e->getMessage())->withInput();
        }
    }

    public function show(VehicleMaintenance $maintenance)
    {
        $maintenance->load([
            'vehicle.vehicleType',
            'maintenanceType',
            'parts',
            'createdBy'
        ]);

        $maintenanceHistory = VehicleMaintenance::where('vehicle_id', $maintenance->vehicle_id)
            ->where('id', '!=', $maintenance->id)
            ->where('status', 'completed')
            ->with('maintenanceType')
            ->latest('completed_date')
            ->take(5)
            ->get();

        return view('admin.maintenances.show', compact('maintenance', 'maintenanceHistory'));
    }

    public function edit(VehicleMaintenance $maintenance)
    {
        $vehicles = Vehicle::with('vehicleType')->get();
        $maintenanceTypes = MaintenanceType::active()->get();
        $maintenance->load('parts');

        return view('admin.maintenances.edit', compact('maintenance', 'vehicles', 'maintenanceTypes'));
    }

    public function update(Request $request, VehicleMaintenance $maintenance)
    {
        $validated = $request->validate([
            'vehicle_id' => 'required|exists:vehicles,id',
            'maintenance_type_id' => 'required|exists:maintenance_types,id',
            'status' => 'required|in:scheduled,in_progress,completed,cancelled',
            'priority' => 'required|in:low,medium,high,urgent',
            'scheduled_date' => 'required|date',
            'completed_date' => 'nullable|date|after_or_equal:scheduled_date',
            'odometer_reading' => 'nullable|integer|min:0',
            'next_due_km' => 'nullable|integer|min:0',
            'next_due_date' => 'nullable|date|after:scheduled_date',
            'labor_cost' => 'nullable|numeric|min:0',
            'parts_cost' => 'nullable|numeric|min:0',
            'vendor_name' => 'nullable|string|max:255',
            'vendor_contact' => 'nullable|string|max:100',
            'invoice_number' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'notes' => 'nullable|string',
            'performed_by' => 'nullable|string|max:255',
        ]);

        $oldStatus = $maintenance->status;
        $oldVehicleId = $maintenance->vehicle_id;

        $validated['cost'] = ($validated['labor_cost'] ?? 0) + ($validated['parts_cost'] ?? 0);

        DB::beginTransaction();
        try {
            $maintenance->update($validated);

            // Handle vehicle status changes
            if ($oldStatus !== $validated['status']) {
                if ($validated['status'] === 'in_progress') {
                    Vehicle::find($validated['vehicle_id'])->update(['status' => 'maintenance']);
                } elseif (in_array($validated['status'], ['completed', 'cancelled']) && $oldStatus === 'in_progress') {
                    Vehicle::find($oldVehicleId)->update(['status' => 'active']);
                }
            }

            DB::commit();

            return redirect()->route('admin.maintenances.show', $maintenance)
                ->with('success', __('updated_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', __('error_occurred') . ': ' . $e->getMessage())->withInput();
        }
    }

    public function destroy(VehicleMaintenance $maintenance)
    {
        if ($maintenance->status === 'in_progress') {
            $maintenance->vehicle->update(['status' => 'active']);
        }

        $maintenance->delete();

        return redirect()->route('admin.maintenances.index')
            ->with('success', __('deleted_successfully'));
    }

    public function dueSoon()
    {
        $maintenances = VehicleMaintenance::with(['vehicle', 'maintenanceType'])
            ->scheduled()
            ->where('scheduled_date', '<=', today()->addDays(7))
            ->orderBy('scheduled_date')
            ->paginate(15);

        return view('admin.maintenances.due-soon', compact('maintenances'));
    }

    public function overdue()
    {
        $maintenances = VehicleMaintenance::with(['vehicle', 'maintenanceType'])
            ->overdue()
            ->orderBy('scheduled_date')
            ->paginate(15);

        return view('admin.maintenances.overdue', compact('maintenances'));
    }

    public function vehicleHistory(Vehicle $vehicle)
    {
        $maintenances = VehicleMaintenance::where('vehicle_id', $vehicle->id)
            ->with('maintenanceType')
            ->latest('scheduled_date')
            ->paginate(15);

        $stats = [
            'total_maintenances' => VehicleMaintenance::where('vehicle_id', $vehicle->id)->count(),
            'total_cost' => VehicleMaintenance::where('vehicle_id', $vehicle->id)
                ->where('status', 'completed')
                ->sum(DB::raw('labor_cost + parts_cost')),
            'scheduled' => VehicleMaintenance::where('vehicle_id', $vehicle->id)
                ->scheduled()
                ->count(),
        ];

        return view('admin.maintenances.vehicle-history', compact('vehicle', 'maintenances', 'stats'));
    }
}
