<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\MaintenanceType;
use Illuminate\Http\Request;

class MaintenanceTypeController extends Controller
{
    public function index(Request $request)
    {
        $query = MaintenanceType::query();

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%")
                ->orWhere('description', 'like', "%{$search}%");
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $maintenanceTypes = $query->withCount('maintenances')
            ->latest()
            ->paginate(15);

        return view('admin.maintenance-types.index', compact('maintenanceTypes'));
    }

    public function create()
    {
        return view('admin.maintenance-types.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:maintenance_types,name',
            'description' => 'nullable|string',
            'default_interval_km' => 'nullable|integer|min:0',
            'default_interval_days' => 'nullable|integer|min:0',
            'estimated_cost' => 'nullable|numeric|min:0',
            'status' => 'required|in:active,inactive',
        ]);

        MaintenanceType::create($validated);

        return redirect()->route('admin.maintenance-types.index')
            ->with('success', __('created_successfully'));
    }

    public function edit(MaintenanceType $maintenanceType)
    {
        return view('admin.maintenance-types.edit', compact('maintenanceType'));
    }

    public function update(Request $request, MaintenanceType $maintenanceType)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:maintenance_types,name,' . $maintenanceType->id,
            'description' => 'nullable|string',
            'default_interval_km' => 'nullable|integer|min:0',
            'default_interval_days' => 'nullable|integer|min:0',
            'estimated_cost' => 'nullable|numeric|min:0',
            'status' => 'required|in:active,inactive',
        ]);

        $maintenanceType->update($validated);

        return redirect()->route('admin.maintenance-types.index')
            ->with('success', __('updated_successfully'));
    }

    public function destroy(MaintenanceType $maintenanceType)
    {
        if ($maintenanceType->maintenances()->exists()) {
            return back()->with('error', __('cannot_delete_maintenance_type_in_use'));
        }

        $maintenanceType->delete();

        return redirect()->route('admin.maintenance-types.index')
            ->with('success', __('deleted_successfully'));
    }
}
