<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Report;
use App\Models\Vehicle;
use App\Models\Driver;
use App\Models\VehicleType;
use App\Models\Trip;
use App\Models\FuelLog;
use App\Models\Delivery;
use App\Models\Tripcost;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ReportController extends Controller
{
    /**
     * Display a listing of reports
     */
    public function index(Request $request)
    {
        $query = Report::with(['creator', 'vehicle', 'driver', 'vehicleType']);

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('report_number', 'like', "%{$search}%")
                    ->orWhere('title', 'like', "%{$search}%");
            });
        }

        // Filter by report type
        if ($request->filled('report_type')) {
            $query->where('report_type', $request->report_type);
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by format
        if ($request->filled('format')) {
            $query->where('format', $request->format);
        }

        // Filter by date range
        if ($request->filled('date_from')) {
            $query->where('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('created_at', '<=', $request->date_to . ' 23:59:59');
        }

        $reports = $query->orderBy('created_at', 'desc')->paginate(15);

        // Stats
        $stats = [
            'total' => Report::count(),
            'completed' => Report::where('status', 'completed')->count(),
            'pending' => Report::whereIn('status', ['draft', 'generating'])->count(),
            'scheduled' => Report::where('is_scheduled', true)->count(),
        ];

        return view('admin.reports.index', compact('reports', 'stats'));
    }

    /**
     * Show the form for creating a new report
     */
    public function create(Request $request)
    {
        $vehicles = Vehicle::where('status', 'active')->orderBy('vehicle_number')->get();
        $drivers = Driver::where('status', 'active')->orderBy('name')->get();
        $vehicleTypes = VehicleType::where('status', 'active')->orderBy('name')->get();

        $suggestedNumber = Report::generateReportNumber();

        // Pre-select report type if provided
        $selectedType = $request->get('type', '');

        return view('admin.reports.create', compact(
            'vehicles',
            'drivers',
            'vehicleTypes',
            'suggestedNumber',
            'selectedType'
        ));
    }

    /**
     * Store a newly created report
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'report_number' => 'required|string|max:50|unique:reports,report_number',
            'title' => 'required|string|max:255',
            'report_type' => 'required|in:fuel_consumption,trip_summary,vehicle_utilization,driver_performance,cost_analysis,delivery_performance,custom',
            'description' => 'nullable|string|max:1000',
            'date_from' => 'nullable|date',
            'date_to' => 'nullable|date|after_or_equal:date_from',
            'vehicle_id' => 'nullable|exists:vehicles,id',
            'driver_id' => 'nullable|exists:drivers,id',
            'vehicle_type_id' => 'nullable|exists:vehicle_types,id',
            'format' => 'required|in:pdf,excel,csv,html',
            'is_scheduled' => 'nullable|boolean',
            'schedule_frequency' => 'nullable|required_if:is_scheduled,1|in:daily,weekly,monthly,quarterly',
        ], [
            'report_number.required' => __('Report number is required'),
            'report_number.unique' => __('This report number already exists'),
            'title.required' => __('Report title is required'),
            'report_type.required' => __('Please select a report type'),
            'date_to.after_or_equal' => __('End date must be after or equal to start date'),
            'format.required' => __('Please select an output format'),
        ]);

        $validated['status'] = 'draft';
        $validated['created_by'] = Auth::id();
        $validated['is_scheduled'] = $request->boolean('is_scheduled');

        // Calculate next run date for scheduled reports
        if ($validated['is_scheduled'] && isset($validated['schedule_frequency'])) {
            $validated['next_run_at'] = $this->calculateNextRunDate($validated['schedule_frequency']);
        }

        $report = Report::create($validated);

        // If generate now was clicked, generate immediately
        if ($request->has('generate_now')) {
            return $this->executeGeneration($report, __('Report created and generated successfully'));
        }

        return redirect()->route('admin.reports.show', $report)
            ->with('success', __('Report created successfully'));
    }

    /**
     * Display the specified report
     */
    public function show(Report $report)
    {
        $report->load(['creator', 'vehicle', 'driver', 'vehicleType']);

        // Get report data if completed
        $reportData = null;
        if ($report->status === 'completed' && $report->summary_data) {
            $reportData = $report->summary_data;
        }

        return view('admin.reports.show', compact('report', 'reportData'));
    }

    /**
     * Show the form for editing the specified report
     */
    public function edit(Report $report)
    {
        // Can only edit draft or failed reports
        if (!in_array($report->status, ['draft', 'failed'])) {
            return redirect()->route('admin.reports.show', $report)
                ->with('error', __('Cannot edit a report that has been generated'));
        }

        $vehicles = Vehicle::where('status', 'active')->orderBy('vehicle_number')->get();
        $drivers = Driver::where('status', 'active')->orderBy('name')->get();
        $vehicleTypes = VehicleType::where('status', 'active')->orderBy('name')->get();

        return view('admin.reports.edit', compact('report', 'vehicles', 'drivers', 'vehicleTypes'));
    }

    /**
     * Update the specified report
     */
    public function update(Request $request, Report $report)
    {
        // Can only edit draft or failed reports
        if (!in_array($report->status, ['draft', 'failed'])) {
            return redirect()->route('admin.reports.show', $report)
                ->with('error', __('Cannot edit a report that has been generated'));
        }

        $validated = $request->validate([
            'report_number' => 'required|string|max:50|unique:reports,report_number,' . $report->id,
            'title' => 'required|string|max:255',
            'report_type' => 'required|in:fuel_consumption,trip_summary,vehicle_utilization,driver_performance,cost_analysis,delivery_performance,custom',
            'description' => 'nullable|string|max:1000',
            'date_from' => 'nullable|date',
            'date_to' => 'nullable|date|after_or_equal:date_from',
            'vehicle_id' => 'nullable|exists:vehicles,id',
            'driver_id' => 'nullable|exists:drivers,id',
            'vehicle_type_id' => 'nullable|exists:vehicle_types,id',
            'format' => 'required|in:pdf,excel,csv,html',
            'is_scheduled' => 'nullable|boolean',
            'schedule_frequency' => 'nullable|required_if:is_scheduled,1|in:daily,weekly,monthly,quarterly',
        ]);

        $validated['is_scheduled'] = $request->boolean('is_scheduled');

        // Update next run date for scheduled reports
        if ($validated['is_scheduled'] && isset($validated['schedule_frequency'])) {
            $validated['next_run_at'] = $this->calculateNextRunDate($validated['schedule_frequency']);
        } else {
            $validated['next_run_at'] = null;
            $validated['schedule_frequency'] = null;
        }

        // Reset status if it was failed
        if ($report->status === 'failed') {
            $validated['status'] = 'draft';
            $validated['error_message'] = null;
        }

        $report->update($validated);

        return redirect()->route('admin.reports.show', $report)
            ->with('success', __('Report updated successfully'));
    }

    /**
     * Remove the specified report
     */
    public function destroy(Report $report)
    {
        // Delete generated file if exists
        if ($report->file_path && file_exists(storage_path('app/' . $report->file_path))) {
            unlink(storage_path('app/' . $report->file_path));
        }

        $report->delete();

        return redirect()->route('admin.reports.index')
            ->with('success', __('Report deleted successfully'));
    }

    /**
     * Generate the report (POST route from show page)
     */
    public function generate(Report $report)
    {
        return $this->executeGeneration($report, __('Report generated successfully'));
    }

    /**
     * Execute report generation logic
     */
    private function executeGeneration(Report $report, string $successMessage)
    {
        try {
            $report->update(['status' => 'generating']);

            // Generate report data based on type
            $data = $this->generateReportData($report);

            // Create the report file
            $fileResult = $this->createReportFile($report, $data);

            // Store summary data and file info
            $report->update([
                'status' => 'completed',
                'summary_data' => $data,
                'file_path' => $fileResult['path'],
                'file_size' => $fileResult['size'],
                'generated_at' => now(),
                'error_message' => null,
            ]);

            return redirect()->route('admin.reports.show', $report)
                ->with('success', $successMessage);
        } catch (\Exception $e) {
            Log::error('Report generation failed', [
                'report_id' => $report->id,
                'error' => $e->getMessage()
            ]);

            $report->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);

            return redirect()->route('admin.reports.show', $report)
                ->with('error', __('Report generation failed: ') . $e->getMessage());
        }
    }

    /**
     * Create the report file based on format
     */
    private function createReportFile(Report $report, array $data): array
    {
        $directory = 'reports/' . date('Y/m');
        $filename = $report->report_number . '_' . time();

        // Ensure directory exists
        $storagePath = storage_path('app/' . $directory);
        if (!is_dir($storagePath)) {
            mkdir($storagePath, 0755, true);
        }

        switch ($report->format) {
            case 'csv':
                return $this->createCsvFile($directory, $filename, $report, $data);
            case 'html':
                return $this->createHtmlFile($directory, $filename, $report, $data);
            case 'pdf':
                return $this->createPdfFile($directory, $filename, $report, $data);
            case 'excel':
                return $this->createCsvFile($directory, $filename, $report, $data);
            default:
                return $this->createCsvFile($directory, $filename, $report, $data);
        }
    }

    /**
     * Create PDF report file
     */
    private function createPdfFile(string $directory, string $filename, Report $report, array $data): array
    {
        $filePath = $directory . '/' . $filename . '.pdf';
        $fullPath = storage_path('app/' . $filePath);

        // Generate HTML content for PDF
        $html = $this->generatePdfHtml($report, $data);

        // Check if dompdf is available
        if (class_exists(\Barryvdh\DomPDF\Facade\Pdf::class)) {
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadHTML($html);
            $pdf->setPaper('A4', 'portrait');
            $pdf->save($fullPath);
        } elseif (class_exists(\Dompdf\Dompdf::class)) {
            $dompdf = new \Dompdf\Dompdf();
            $dompdf->loadHtml($html);
            $dompdf->setPaper('A4', 'portrait');
            $dompdf->render();
            file_put_contents($fullPath, $dompdf->output());
        } else {
            // Fallback: Save as HTML with .pdf extension (browser will handle)
            file_put_contents($fullPath, $html);
        }

        return [
            'path' => $filePath,
            'size' => filesize($fullPath)
        ];
    }

    /**
     * Generate HTML content for PDF
     */
    private function generatePdfHtml(Report $report, array $data): string
    {
        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>' . htmlspecialchars($report->title) . '</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 12px; color: #333; padding: 30px; }
        .header { text-align: center; margin-bottom: 30px; border-bottom: 2px solid #2563eb; padding-bottom: 20px; }
        .header h1 { color: #2563eb; font-size: 24px; margin-bottom: 10px; }
        .header .subtitle { color: #6b7280; font-size: 14px; }
        .meta { margin-bottom: 25px; }
        .meta-grid { display: table; width: 100%; }
        .meta-item { display: table-cell; width: 25%; padding: 10px; background: #f9fafb; border: 1px solid #e5e7eb; }
        .meta-label { font-size: 10px; color: #6b7280; text-transform: uppercase; }
        .meta-value { font-size: 12px; font-weight: bold; margin-top: 3px; }
        .section { margin-bottom: 25px; }
        .section h2 { color: #374151; font-size: 16px; margin-bottom: 15px; padding-bottom: 8px; border-bottom: 1px solid #e5e7eb; }
        .summary-grid { width: 100%; }
        .summary-grid td { width: 25%; padding: 15px; text-align: center; background: #f3f4f6; border: 1px solid #e5e7eb; }
        .summary-value { font-size: 20px; font-weight: bold; color: #2563eb; }
        .summary-label { font-size: 10px; color: #6b7280; text-transform: uppercase; margin-top: 5px; }
        table.data-table { width: 100%; border-collapse: collapse; margin-top: 10px; }
        table.data-table th { background: #2563eb; color: white; padding: 10px; text-align: left; font-size: 11px; }
        table.data-table td { padding: 8px 10px; border-bottom: 1px solid #e5e7eb; font-size: 11px; }
        table.data-table tr:nth-child(even) { background: #f9fafb; }
        .footer { margin-top: 30px; padding-top: 15px; border-top: 1px solid #e5e7eb; text-align: center; color: #6b7280; font-size: 10px; }
        .page-break { page-break-after: always; }
    </style>
</head>
<body>
    <div class="header">
        <h1>' . htmlspecialchars($report->title) . '</h1>
        <div class="subtitle">Report Number: ' . htmlspecialchars($report->report_number) . '</div>
    </div>
    
    <div class="meta">
        <table class="meta-grid" style="width:100%; border-collapse: collapse;">
            <tr>
                <td class="meta-item">
                    <div class="meta-label">Report Type</div>
                    <div class="meta-value">' . ucwords(str_replace('_', ' ', $report->report_type)) . '</div>
                </td>
                <td class="meta-item">
                    <div class="meta-label">Date Range</div>
                    <div class="meta-value">' . ($report->date_from ?? 'All') . ' to ' . ($report->date_to ?? 'All') . '</div>
                </td>
                <td class="meta-item">
                    <div class="meta-label">Generated</div>
                    <div class="meta-value">' . now()->format('M d, Y H:i') . '</div>
                </td>
                <td class="meta-item">
                    <div class="meta-label">Format</div>
                    <div class="meta-value">PDF</div>
                </td>
            </tr>
        </table>
    </div>';

        // Summary Section
        if (isset($data['summary']) && count($data['summary']) > 0) {
            $html .= '<div class="section"><h2>Summary</h2><table class="summary-grid" style="width:100%; border-collapse: collapse;"><tr>';
            $count = 0;
            foreach ($data['summary'] as $key => $value) {
                if ($count > 0 && $count % 4 == 0) {
                    $html .= '</tr><tr>';
                }
                $label = ucwords(str_replace('_', ' ', $key));
                $html .= '<td><div class="summary-value">' . htmlspecialchars($value) . '</div><div class="summary-label">' . htmlspecialchars($label) . '</div></td>';
                $count++;
            }
            // Fill empty cells if needed
            while ($count % 4 != 0) {
                $html .= '<td></td>';
                $count++;
            }
            $html .= '</tr></table></div>';
        }

        // Data Tables
        $html .= $this->generatePdfDataTables($data, $report->report_type);

        $html .= '<div class="footer">
            <p>Generated by LogiFleet Fleet Management System</p>
            <p>Report generated on ' . now()->format('F d, Y \a\t h:i A') . '</p>
        </div>
</body>
</html>';

        return $html;
    }

    /**
     * Generate PDF data tables based on report type
     */
    private function generatePdfDataTables(array $data, string $reportType): string
    {
        $html = '';

        switch ($reportType) {
            case 'fuel_consumption':
                if (isset($data['by_vehicle']) && count($data['by_vehicle']) > 0) {
                    $html .= '<div class="section"><h2>Fuel Consumption by Vehicle</h2>';
                    $html .= '<table class="data-table"><thead><tr><th>Vehicle</th><th>Total Quantity</th><th>Total Cost</th><th>Avg Efficiency</th><th>Fill Count</th></tr></thead><tbody>';
                    foreach ($data['by_vehicle'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars($row['vehicle'] ?? '-') . '</td><td>' . number_format($row['total_quantity'] ?? 0, 2) . ' L</td><td>$' . number_format($row['total_cost'] ?? 0, 2) . '</td><td>' . ($row['avg_efficiency'] ?? '-') . '</td><td>' . ($row['fill_count'] ?? 0) . '</td></tr>';
                    }
                    $html .= '</tbody></table></div>';
                }
                if (isset($data['by_fuel_type']) && count($data['by_fuel_type']) > 0) {
                    $html .= '<div class="section"><h2>By Fuel Type</h2>';
                    $html .= '<table class="data-table"><thead><tr><th>Fuel Type</th><th>Total Quantity</th><th>Total Cost</th></tr></thead><tbody>';
                    foreach ($data['by_fuel_type'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars(ucfirst($row['type'] ?? '-')) . '</td><td>' . number_format($row['total_quantity'] ?? 0, 2) . ' L</td><td>$' . number_format($row['total_cost'] ?? 0, 2) . '</td></tr>';
                    }
                    $html .= '</tbody></table></div>';
                }
                break;

            case 'trip_summary':
                if (isset($data['by_status']) && count($data['by_status']) > 0) {
                    $html .= '<div class="section"><h2>Trips by Status</h2>';
                    $html .= '<table class="data-table"><thead><tr><th>Status</th><th>Count</th></tr></thead><tbody>';
                    foreach ($data['by_status'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars(ucfirst($row['status'] ?? '-')) . '</td><td>' . ($row['count'] ?? 0) . '</td></tr>';
                    }
                    $html .= '</tbody></table></div>';
                }
                if (isset($data['by_vehicle']) && count($data['by_vehicle']) > 0) {
                    $html .= '<div class="section"><h2>Trips by Vehicle</h2>';
                    $html .= '<table class="data-table"><thead><tr><th>Vehicle</th><th>Total Trips</th><th>Completed</th></tr></thead><tbody>';
                    foreach ($data['by_vehicle'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars($row['vehicle'] ?? '-') . '</td><td>' . ($row['trip_count'] ?? 0) . '</td><td>' . ($row['completed'] ?? 0) . '</td></tr>';
                    }
                    $html .= '</tbody></table></div>';
                }
                break;

            case 'vehicle_utilization':
                if (isset($data['vehicles']) && count($data['vehicles']) > 0) {
                    $html .= '<div class="section"><h2>Vehicle Utilization Details</h2>';
                    $html .= '<table class="data-table"><thead><tr><th>Vehicle Number</th><th>Type</th><th>Status</th><th>Total Trips</th><th>Completed</th><th>Utilization Rate</th></tr></thead><tbody>';
                    foreach ($data['vehicles'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars($row['vehicle_number'] ?? '-') . '</td><td>' . htmlspecialchars($row['vehicle_type'] ?? '-') . '</td><td>' . htmlspecialchars(ucfirst($row['status'] ?? '-')) . '</td><td>' . ($row['total_trips'] ?? 0) . '</td><td>' . ($row['completed_trips'] ?? 0) . '</td><td>' . ($row['utilization_rate'] ?? 0) . '%</td></tr>';
                    }
                    $html .= '</tbody></table></div>';
                }
                break;

            case 'driver_performance':
                if (isset($data['drivers']) && count($data['drivers']) > 0) {
                    $html .= '<div class="section"><h2>Driver Performance Details</h2>';
                    $html .= '<table class="data-table"><thead><tr><th>Driver Name</th><th>Mobile</th><th>Status</th><th>Total Trips</th><th>Completed</th><th>Rate</th><th>Deliveries</th></tr></thead><tbody>';
                    foreach ($data['drivers'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars($row['name'] ?? '-') . '</td><td>' . htmlspecialchars($row['mobile'] ?? '-') . '</td><td>' . htmlspecialchars(ucfirst($row['status'] ?? '-')) . '</td><td>' . ($row['total_trips'] ?? 0) . '</td><td>' . ($row['completed_trips'] ?? 0) . '</td><td>' . ($row['completion_rate'] ?? 0) . '%</td><td>' . ($row['total_deliveries'] ?? 0) . '</td></tr>';
                    }
                    $html .= '</tbody></table></div>';
                }
                break;

            case 'cost_analysis':
                if (isset($data['by_cost_type']) && count($data['by_cost_type']) > 0) {
                    $html .= '<div class="section"><h2>Costs by Type</h2>';
                    $html .= '<table class="data-table"><thead><tr><th>Cost Type</th><th>Total Amount</th><th>Count</th></tr></thead><tbody>';
                    foreach ($data['by_cost_type'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars(ucfirst($row['type'] ?? '-')) . '</td><td>$' . number_format($row['total'] ?? 0, 2) . '</td><td>' . ($row['count'] ?? 0) . '</td></tr>';
                    }
                    $html .= '</tbody></table></div>';
                }
                break;

            case 'delivery_performance':
                if (isset($data['by_status']) && count($data['by_status']) > 0) {
                    $html .= '<div class="section"><h2>Deliveries by Status</h2>';
                    $html .= '<table class="data-table"><thead><tr><th>Status</th><th>Count</th></tr></thead><tbody>';
                    foreach ($data['by_status'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars(ucfirst($row['status'] ?? '-')) . '</td><td>' . ($row['count'] ?? 0) . '</td></tr>';
                    }
                    $html .= '</tbody></table></div>';
                }
                break;
        }

        return $html;
    }

    /**
     * Create CSV report file
     */
    private function createCsvFile(string $directory, string $filename, Report $report, array $data): array
    {
        $filePath = $directory . '/' . $filename . '.csv';
        $fullPath = storage_path('app/' . $filePath);

        $handle = fopen($fullPath, 'w');

        // Add BOM for Excel compatibility
        fprintf($handle, chr(0xEF) . chr(0xBB) . chr(0xBF));

        // Report Header
        fputcsv($handle, ['Report: ' . $report->title]);
        fputcsv($handle, ['Type: ' . ucwords(str_replace('_', ' ', $report->report_type))]);
        fputcsv($handle, ['Generated: ' . now()->format('Y-m-d H:i:s')]);
        fputcsv($handle, ['Date Range: ' . ($report->date_from ?? 'All') . ' to ' . ($report->date_to ?? 'All')]);
        fputcsv($handle, []); // Empty row

        // Summary Section
        if (isset($data['summary'])) {
            fputcsv($handle, ['=== SUMMARY ===']);
            foreach ($data['summary'] as $key => $value) {
                fputcsv($handle, [ucwords(str_replace('_', ' ', $key)), $value]);
            }
            fputcsv($handle, []);
        }

        // Data Sections
        $this->writeCsvDataSections($handle, $data, $report->report_type);

        fclose($handle);

        return [
            'path' => $filePath,
            'size' => filesize($fullPath)
        ];
    }

    /**
     * Write CSV data sections based on report type
     */
    private function writeCsvDataSections($handle, array $data, string $reportType): void
    {
        switch ($reportType) {
            case 'fuel_consumption':
                if (isset($data['by_vehicle']) && count($data['by_vehicle']) > 0) {
                    fputcsv($handle, ['=== BY VEHICLE ===']);
                    fputcsv($handle, ['Vehicle', 'Total Quantity', 'Total Cost', 'Avg Efficiency', 'Fill Count']);
                    foreach ($data['by_vehicle'] as $row) {
                        fputcsv($handle, [
                            $row['vehicle'] ?? '-',
                            $row['total_quantity'] ?? 0,
                            $row['total_cost'] ?? 0,
                            $row['avg_efficiency'] ?? '-',
                            $row['fill_count'] ?? 0
                        ]);
                    }
                }
                break;

            case 'trip_summary':
                if (isset($data['by_status']) && count($data['by_status']) > 0) {
                    fputcsv($handle, ['=== BY STATUS ===']);
                    fputcsv($handle, ['Status', 'Count']);
                    foreach ($data['by_status'] as $row) {
                        fputcsv($handle, [$row['status'] ?? '-', $row['count'] ?? 0]);
                    }
                }
                if (isset($data['by_vehicle']) && count($data['by_vehicle']) > 0) {
                    fputcsv($handle, []);
                    fputcsv($handle, ['=== BY VEHICLE ===']);
                    fputcsv($handle, ['Vehicle', 'Trip Count', 'Completed']);
                    foreach ($data['by_vehicle'] as $row) {
                        fputcsv($handle, [
                            $row['vehicle'] ?? '-',
                            $row['trip_count'] ?? 0,
                            $row['completed'] ?? 0
                        ]);
                    }
                }
                break;

            case 'vehicle_utilization':
                if (isset($data['vehicles']) && count($data['vehicles']) > 0) {
                    fputcsv($handle, ['=== VEHICLE UTILIZATION ===']);
                    fputcsv($handle, ['Vehicle Number', 'Vehicle Type', 'Status', 'Total Trips', 'Completed', 'Utilization Rate']);
                    foreach ($data['vehicles'] as $row) {
                        fputcsv($handle, [
                            $row['vehicle_number'] ?? '-',
                            $row['vehicle_type'] ?? '-',
                            $row['status'] ?? '-',
                            $row['total_trips'] ?? 0,
                            $row['completed_trips'] ?? 0,
                            ($row['utilization_rate'] ?? 0) . '%'
                        ]);
                    }
                }
                break;

            case 'driver_performance':
                if (isset($data['drivers']) && count($data['drivers']) > 0) {
                    fputcsv($handle, ['=== DRIVER PERFORMANCE ===']);
                    fputcsv($handle, ['Name', 'Mobile', 'Status', 'Total Trips', 'Completed', 'Completion Rate', 'Total Deliveries', 'Successful']);
                    foreach ($data['drivers'] as $row) {
                        fputcsv($handle, [
                            $row['name'] ?? '-',
                            $row['mobile'] ?? '-',
                            $row['status'] ?? '-',
                            $row['total_trips'] ?? 0,
                            $row['completed_trips'] ?? 0,
                            ($row['completion_rate'] ?? 0) . '%',
                            $row['total_deliveries'] ?? 0,
                            $row['successful_deliveries'] ?? 0
                        ]);
                    }
                }
                break;

            case 'cost_analysis':
                if (isset($data['by_cost_type']) && count($data['by_cost_type']) > 0) {
                    fputcsv($handle, ['=== BY COST TYPE ===']);
                    fputcsv($handle, ['Cost Type', 'Total Amount', 'Count']);
                    foreach ($data['by_cost_type'] as $row) {
                        fputcsv($handle, [
                            $row['type'] ?? '-',
                            $row['total'] ?? 0,
                            $row['count'] ?? 0
                        ]);
                    }
                }
                break;

            case 'delivery_performance':
                if (isset($data['by_status']) && count($data['by_status']) > 0) {
                    fputcsv($handle, ['=== BY STATUS ===']);
                    fputcsv($handle, ['Status', 'Count']);
                    foreach ($data['by_status'] as $row) {
                        fputcsv($handle, [$row['status'] ?? '-', $row['count'] ?? 0]);
                    }
                }
                break;
        }
    }

    /**
     * Create HTML report file
     */
    private function createHtmlFile(string $directory, string $filename, Report $report, array $data): array
    {
        $filePath = $directory . '/' . $filename . '.html';
        $fullPath = storage_path('app/' . $filePath);

        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>' . htmlspecialchars($report->title) . '</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 40px; color: #333; }
        h1 { color: #2563eb; border-bottom: 2px solid #2563eb; padding-bottom: 10px; }
        h2 { color: #374151; margin-top: 30px; }
        .meta { color: #6b7280; margin-bottom: 20px; }
        .summary { background: #f3f4f6; padding: 20px; border-radius: 8px; margin: 20px 0; }
        .summary-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; }
        .summary-item { background: white; padding: 15px; border-radius: 6px; text-align: center; }
        .summary-value { font-size: 24px; font-weight: bold; color: #2563eb; }
        .summary-label { font-size: 12px; color: #6b7280; }
        table { width: 100%; border-collapse: collapse; margin: 20px 0; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #e5e7eb; }
        th { background: #f9fafb; font-weight: 600; }
        tr:hover { background: #f9fafb; }
        .footer { margin-top: 40px; padding-top: 20px; border-top: 1px solid #e5e7eb; color: #6b7280; font-size: 12px; }
    </style>
</head>
<body>
    <h1>' . htmlspecialchars($report->title) . '</h1>
    <div class="meta">
        <p><strong>Report Number:</strong> ' . htmlspecialchars($report->report_number) . '</p>
        <p><strong>Type:</strong> ' . ucwords(str_replace('_', ' ', $report->report_type)) . '</p>
        <p><strong>Date Range:</strong> ' . ($report->date_from ?? 'All') . ' to ' . ($report->date_to ?? 'All') . '</p>
        <p><strong>Generated:</strong> ' . now()->format('F d, Y h:i A') . '</p>
    </div>';

        // Summary Section
        if (isset($data['summary'])) {
            $html .= '<div class="summary"><h2>Summary</h2><div class="summary-grid">';
            foreach ($data['summary'] as $key => $value) {
                $label = ucwords(str_replace('_', ' ', $key));
                $html .= '<div class="summary-item"><div class="summary-value">' . htmlspecialchars($value) . '</div><div class="summary-label">' . htmlspecialchars($label) . '</div></div>';
            }
            $html .= '</div></div>';
        }

        // Data Tables
        $html .= $this->generateHtmlDataTables($data, $report->report_type);

        $html .= '<div class="footer">Generated by LogiFleet Fleet Management System</div></body></html>';

        file_put_contents($fullPath, $html);

        return [
            'path' => $filePath,
            'size' => filesize($fullPath)
        ];
    }

    /**
     * Generate HTML data tables based on report type
     */
    private function generateHtmlDataTables(array $data, string $reportType): string
    {
        $html = '';

        switch ($reportType) {
            case 'fuel_consumption':
                if (isset($data['by_vehicle']) && count($data['by_vehicle']) > 0) {
                    $html .= '<h2>By Vehicle</h2><table><thead><tr><th>Vehicle</th><th>Total Quantity</th><th>Total Cost</th><th>Avg Efficiency</th><th>Fill Count</th></tr></thead><tbody>';
                    foreach ($data['by_vehicle'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars($row['vehicle'] ?? '-') . '</td><td>' . ($row['total_quantity'] ?? 0) . '</td><td>$' . number_format($row['total_cost'] ?? 0, 2) . '</td><td>' . ($row['avg_efficiency'] ?? '-') . '</td><td>' . ($row['fill_count'] ?? 0) . '</td></tr>';
                    }
                    $html .= '</tbody></table>';
                }
                break;

            case 'trip_summary':
            case 'delivery_performance':
                if (isset($data['by_status']) && count($data['by_status']) > 0) {
                    $html .= '<h2>By Status</h2><table><thead><tr><th>Status</th><th>Count</th></tr></thead><tbody>';
                    foreach ($data['by_status'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars(ucfirst($row['status'] ?? '-')) . '</td><td>' . ($row['count'] ?? 0) . '</td></tr>';
                    }
                    $html .= '</tbody></table>';
                }
                break;

            case 'vehicle_utilization':
                if (isset($data['vehicles']) && count($data['vehicles']) > 0) {
                    $html .= '<h2>Vehicle Utilization</h2><table><thead><tr><th>Vehicle</th><th>Type</th><th>Status</th><th>Total Trips</th><th>Completed</th><th>Utilization</th></tr></thead><tbody>';
                    foreach ($data['vehicles'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars($row['vehicle_number'] ?? '-') . '</td><td>' . htmlspecialchars($row['vehicle_type'] ?? '-') . '</td><td>' . htmlspecialchars($row['status'] ?? '-') . '</td><td>' . ($row['total_trips'] ?? 0) . '</td><td>' . ($row['completed_trips'] ?? 0) . '</td><td>' . ($row['utilization_rate'] ?? 0) . '%</td></tr>';
                    }
                    $html .= '</tbody></table>';
                }
                break;

            case 'driver_performance':
                if (isset($data['drivers']) && count($data['drivers']) > 0) {
                    $html .= '<h2>Driver Performance</h2><table><thead><tr><th>Name</th><th>Mobile</th><th>Status</th><th>Trips</th><th>Completed</th><th>Rate</th></tr></thead><tbody>';
                    foreach ($data['drivers'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars($row['name'] ?? '-') . '</td><td>' . htmlspecialchars($row['mobile'] ?? '-') . '</td><td>' . htmlspecialchars($row['status'] ?? '-') . '</td><td>' . ($row['total_trips'] ?? 0) . '</td><td>' . ($row['completed_trips'] ?? 0) . '</td><td>' . ($row['completion_rate'] ?? 0) . '%</td></tr>';
                    }
                    $html .= '</tbody></table>';
                }
                break;

            case 'cost_analysis':
                if (isset($data['by_cost_type']) && count($data['by_cost_type']) > 0) {
                    $html .= '<h2>By Cost Type</h2><table><thead><tr><th>Cost Type</th><th>Total Amount</th><th>Count</th></tr></thead><tbody>';
                    foreach ($data['by_cost_type'] as $row) {
                        $html .= '<tr><td>' . htmlspecialchars(ucfirst($row['type'] ?? '-')) . '</td><td>$' . number_format($row['total'] ?? 0, 2) . '</td><td>' . ($row['count'] ?? 0) . '</td></tr>';
                    }
                    $html .= '</tbody></table>';
                }
                break;
        }

        return $html;
    }

    /**
     * Download the generated report
     */
    public function download(Report $report)
    {
        // Check if report is completed
        if ($report->status !== 'completed') {
            return redirect()->route('admin.reports.show', $report)
                ->with('error', __('Report has not been generated yet'));
        }

        // If file exists, serve it
        if ($report->file_path) {
            $filePath = storage_path('app/' . $report->file_path);

            if (file_exists($filePath)) {
                $extension = pathinfo($filePath, PATHINFO_EXTENSION);
                $downloadName = $report->report_number . '.' . $extension;

                // Set proper content type for different formats
                $contentTypes = [
                    'pdf' => 'application/pdf',
                    'csv' => 'text/csv',
                    'html' => 'text/html',
                    'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                ];

                $contentType = $contentTypes[$extension] ?? 'application/octet-stream';

                return response()->download($filePath, $downloadName, [
                    'Content-Type' => $contentType
                ]);
            }
        }

        // If no file but has summary data, generate on-the-fly based on format
        if ($report->summary_data) {
            return $this->generateDownloadOnTheFly($report);
        }

        return redirect()->route('admin.reports.show', $report)
            ->with('error', __('Report file not found. Please regenerate the report.'));
    }

    /**
     * Generate download file on-the-fly from summary data
     */
    private function generateDownloadOnTheFly(Report $report)
    {
        $data = $report->summary_data;
        $filename = $report->report_number;

        // Handle PDF format
        if ($report->format === 'pdf') {
            return $this->generatePdfOnTheFly($report, $data);
        }

        // Handle HTML format
        if ($report->format === 'html') {
            return $this->generateHtmlOnTheFly($report, $data);
        }

        // Default to CSV
        $output = fopen('php://temp', 'r+');

        // Add BOM for Excel
        fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

        // Header
        fputcsv($output, ['Report: ' . $report->title]);
        fputcsv($output, ['Type: ' . ucwords(str_replace('_', ' ', $report->report_type))]);
        fputcsv($output, ['Generated: ' . now()->format('Y-m-d H:i:s')]);
        fputcsv($output, []);

        // Summary
        if (isset($data['summary'])) {
            fputcsv($output, ['=== SUMMARY ===']);
            foreach ($data['summary'] as $key => $value) {
                fputcsv($output, [ucwords(str_replace('_', ' ', $key)), $value]);
            }
            fputcsv($output, []);
        }

        // Data sections
        $this->writeCsvDataSections($output, $data, $report->report_type);

        rewind($output);
        $content = stream_get_contents($output);
        fclose($output);

        return response($content)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '.csv"');
    }

    /**
     * Generate PDF on-the-fly
     */
    private function generatePdfOnTheFly(Report $report, array $data)
    {
        $html = $this->generatePdfHtml($report, $data);
        $filename = $report->report_number . '.pdf';

        // Try to use dompdf if available
        if (class_exists(\Barryvdh\DomPDF\Facade\Pdf::class)) {
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadHTML($html);
            $pdf->setPaper('A4', 'portrait');
            return $pdf->download($filename);
        } elseif (class_exists(\Dompdf\Dompdf::class)) {
            $dompdf = new \Dompdf\Dompdf();
            $dompdf->loadHtml($html);
            $dompdf->setPaper('A4', 'portrait');
            $dompdf->render();

            return response($dompdf->output())
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
        }

        // Fallback: Serve HTML with PDF extension (browser may handle it)
        return response($html)
            ->header('Content-Type', 'text/html')
            ->header('Content-Disposition', 'attachment; filename="' . $report->report_number . '.html"');
    }

    /**
     * Generate HTML on-the-fly
     */
    private function generateHtmlOnTheFly(Report $report, array $data)
    {
        $html = $this->generatePdfHtml($report, $data);
        $filename = $report->report_number . '.html';

        return response($html)
            ->header('Content-Type', 'text/html')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }

    /**
     * Generate report data based on type
     */
    private function generateReportData(Report $report): array
    {
        switch ($report->report_type) {
            case 'fuel_consumption':
                return $this->generateFuelConsumptionData($report);
            case 'trip_summary':
                return $this->generateTripSummaryData($report);
            case 'vehicle_utilization':
                return $this->generateVehicleUtilizationData($report);
            case 'driver_performance':
                return $this->generateDriverPerformanceData($report);
            case 'cost_analysis':
                return $this->generateCostAnalysisData($report);
            case 'delivery_performance':
                return $this->generateDeliveryPerformanceData($report);
            default:
                return $this->generateCustomData($report);
        }
    }

    /**
     * Generate fuel consumption report data
     */
    private function generateFuelConsumptionData(Report $report): array
    {
        $query = FuelLog::query();

        if ($report->date_from) {
            $query->where('fill_date', '>=', $report->date_from);
        }
        if ($report->date_to) {
            $query->where('fill_date', '<=', $report->date_to);
        }
        if ($report->vehicle_id) {
            $query->where('vehicle_id', $report->vehicle_id);
        }
        if ($report->driver_id) {
            $query->where('driver_id', $report->driver_id);
        }

        $fuelLogs = $query->with(['vehicle', 'driver'])->get();

        $byVehicle = $fuelLogs->groupBy('vehicle_id')->map(function ($logs) {
            return [
                'vehicle' => $logs->first()->vehicle->vehicle_number ?? 'Unknown',
                'total_quantity' => $logs->sum('quantity'),
                'total_cost' => $logs->sum('total_cost'),
                'avg_efficiency' => $logs->whereNotNull('fuel_efficiency')->avg('fuel_efficiency'),
                'fill_count' => $logs->count(),
            ];
        })->values()->toArray();

        $byFuelType = $fuelLogs->groupBy('fuel_type')->map(function ($logs, $type) {
            return [
                'type' => ucfirst($type),
                'total_quantity' => $logs->sum('quantity'),
                'total_cost' => $logs->sum('total_cost'),
            ];
        })->values()->toArray();

        return [
            'summary' => [
                'total_quantity' => $fuelLogs->sum('quantity'),
                'total_cost' => $fuelLogs->sum('total_cost'),
                'avg_price_per_unit' => $fuelLogs->avg('price_per_unit'),
                'avg_efficiency' => $fuelLogs->whereNotNull('fuel_efficiency')->avg('fuel_efficiency'),
                'total_records' => $fuelLogs->count(),
            ],
            'by_vehicle' => $byVehicle,
            'by_fuel_type' => $byFuelType,
        ];
    }

    /**
     * Generate trip summary report data
     */
    private function generateTripSummaryData(Report $report): array
    {
        $query = Trip::query();

        if ($report->date_from) {
            $query->where('created_at', '>=', $report->date_from);
        }
        if ($report->date_to) {
            $query->where('created_at', '<=', $report->date_to->endOfDay());
        }
        if ($report->vehicle_id) {
            $query->where('vehicle_id', $report->vehicle_id);
        }
        if ($report->driver_id) {
            $query->where('driver_id', $report->driver_id);
        }

        $trips = $query->with(['vehicle', 'driver', 'costs'])->get();

        $byStatus = $trips->groupBy('status')->map(function ($trips, $status) {
            return [
                'status' => ucfirst(str_replace('_', ' ', $status)),
                'count' => $trips->count(),
            ];
        })->values()->toArray();

        $byVehicle = $trips->groupBy('vehicle_id')->map(function ($trips) {
            return [
                'vehicle' => $trips->first()->vehicle->vehicle_number ?? 'Unknown',
                'trip_count' => $trips->count(),
                'completed' => $trips->where('status', 'delivered')->count(),
            ];
        })->values()->toArray();

        return [
            'summary' => [
                'total_trips' => $trips->count(),
                'completed' => $trips->where('status', 'delivered')->count(),
                'in_transit' => $trips->where('status', 'in_transit')->count(),
                'pending' => $trips->whereIn('status', ['assigned', 'pending'])->count(),
                'cancelled' => $trips->where('status', 'cancelled')->count(),
            ],
            'by_status' => $byStatus,
            'by_vehicle' => $byVehicle,
        ];
    }

    /**
     * Generate vehicle utilization report data
     */
    private function generateVehicleUtilizationData(Report $report): array
    {
        $query = Vehicle::with(['vehicleType', 'trips']);

        if ($report->vehicle_type_id) {
            $query->where('vehicle_type_id', $report->vehicle_type_id);
        }
        if ($report->vehicle_id) {
            $query->where('id', $report->vehicle_id);
        }

        $vehicles = $query->get();

        $utilizationData = $vehicles->map(function ($vehicle) use ($report) {
            $tripsQuery = $vehicle->trips();

            if ($report->date_from) {
                $tripsQuery->where('created_at', '>=', $report->date_from);
            }
            if ($report->date_to) {
                $tripsQuery->where('created_at', '<=', $report->date_to->endOfDay());
            }

            $trips = $tripsQuery->get();

            return [
                'vehicle_number' => $vehicle->vehicle_number,
                'vehicle_type' => $vehicle->vehicleType->name ?? '-',
                'status' => $vehicle->status,
                'total_trips' => $trips->count(),
                'completed_trips' => $trips->where('status', 'delivered')->count(),
                'utilization_rate' => $trips->count() > 0 ?
                    round(($trips->where('status', 'delivered')->count() / $trips->count()) * 100, 1) : 0,
            ];
        })->toArray();

        return [
            'summary' => [
                'total_vehicles' => $vehicles->count(),
                'active_vehicles' => $vehicles->where('status', 'active')->count(),
                'avg_utilization' => collect($utilizationData)->avg('utilization_rate'),
            ],
            'vehicles' => $utilizationData,
        ];
    }

    /**
     * Generate driver performance report data
     */
    private function generateDriverPerformanceData(Report $report): array
    {
        $query = Driver::with(['trips', 'deliveries']);

        if ($report->driver_id) {
            $query->where('id', $report->driver_id);
        }

        $drivers = $query->get();

        $performanceData = $drivers->map(function ($driver) use ($report) {
            $tripsQuery = $driver->trips();
            $deliveriesQuery = $driver->deliveries();

            if ($report->date_from) {
                $tripsQuery->where('created_at', '>=', $report->date_from);
                $deliveriesQuery->where('created_at', '>=', $report->date_from);
            }
            if ($report->date_to) {
                $tripsQuery->where('created_at', '<=', $report->date_to->endOfDay());
                $deliveriesQuery->where('created_at', '<=', $report->date_to->endOfDay());
            }

            $trips = $tripsQuery->get();
            $deliveries = $deliveriesQuery->get();

            $completedTrips = $trips->where('status', 'delivered')->count();
            $totalTrips = $trips->count();

            return [
                'name' => $driver->name,
                'mobile' => $driver->mobile,
                'status' => $driver->status,
                'total_trips' => $totalTrips,
                'completed_trips' => $completedTrips,
                'completion_rate' => $totalTrips > 0 ? round(($completedTrips / $totalTrips) * 100, 1) : 0,
                'total_deliveries' => $deliveries->count(),
                'successful_deliveries' => $deliveries->where('delivery_status', 'completed')->count(),
            ];
        })->toArray();

        return [
            'summary' => [
                'total_drivers' => $drivers->count(),
                'active_drivers' => $drivers->where('status', 'active')->count(),
                'avg_completion_rate' => collect($performanceData)->avg('completion_rate'),
            ],
            'drivers' => $performanceData,
        ];
    }

    /**
     * Generate cost analysis report data
     */
    private function generateCostAnalysisData(Report $report): array
    {
        $costsQuery = Tripcost::query();
        $fuelQuery = FuelLog::query();

        if ($report->date_from) {
            $costsQuery->where('created_at', '>=', $report->date_from);
            $fuelQuery->where('fill_date', '>=', $report->date_from);
        }
        if ($report->date_to) {
            $costsQuery->where('created_at', '<=', $report->date_to->endOfDay());
            $fuelQuery->where('fill_date', '<=', $report->date_to);
        }
        if ($report->vehicle_id) {
            $costsQuery->whereHas('trip', function ($q) use ($report) {
                $q->where('vehicle_id', $report->vehicle_id);
            });
            $fuelQuery->where('vehicle_id', $report->vehicle_id);
        }

        $costs = $costsQuery->with('trip')->get();
        $fuelCosts = $fuelQuery->sum('total_cost');

        $byCostType = $costs->groupBy('cost_type')->map(function ($items, $type) {
            return [
                'type' => ucfirst(str_replace('_', ' ', $type)),
                'total' => $items->sum('amount'),
                'count' => $items->count(),
            ];
        })->values()->toArray();

        return [
            'summary' => [
                'total_trip_costs' => $costs->sum('amount'),
                'total_fuel_costs' => $fuelCosts,
                'grand_total' => $costs->sum('amount') + $fuelCosts,
                'cost_records' => $costs->count(),
            ],
            'by_cost_type' => $byCostType,
            'fuel_costs' => $fuelCosts,
        ];
    }

    /**
     * Generate delivery performance report data
     */
    private function generateDeliveryPerformanceData(Report $report): array
    {
        $query = Delivery::with(['trip.vehicle', 'trip.driver']);

        if ($report->date_from) {
            $query->where('delivered_at', '>=', $report->date_from);
        }
        if ($report->date_to) {
            $query->where('delivered_at', '<=', $report->date_to->endOfDay());
        }
        if ($report->driver_id) {
            $query->whereHas('trip', function ($q) use ($report) {
                $q->where('driver_id', $report->driver_id);
            });
        }
        if ($report->vehicle_id) {
            $query->whereHas('trip', function ($q) use ($report) {
                $q->where('vehicle_id', $report->vehicle_id);
            });
        }

        $deliveries = $query->get();

        $byStatus = $deliveries->groupBy('delivery_status')->map(function ($items, $status) {
            return [
                'status' => ucfirst($status),
                'count' => $items->count(),
            ];
        })->values()->toArray();

        return [
            'summary' => [
                'total_deliveries' => $deliveries->count(),
                'completed' => $deliveries->where('delivery_status', 'completed')->count(),
                'partial' => $deliveries->where('delivery_status', 'partial')->count(),
                'failed' => $deliveries->where('delivery_status', 'failed')->count(),
                'pending' => $deliveries->where('delivery_status', 'pending')->count(),
                'success_rate' => $deliveries->count() > 0 ?
                    round(($deliveries->where('delivery_status', 'completed')->count() / $deliveries->count()) * 100, 1) : 0,
            ],
            'by_status' => $byStatus,
        ];
    }

    /**
     * Generate custom report data
     */
    private function generateCustomData(Report $report): array
    {
        return [
            'summary' => [
                'message' => 'Custom report data',
                'generated_at' => now()->toDateTimeString(),
            ],
        ];
    }

    /**
     * Calculate next run date for scheduled reports
     */
    private function calculateNextRunDate(string $frequency): Carbon
    {
        switch ($frequency) {
            case 'daily':
                return Carbon::tomorrow()->setTime(6, 0);
            case 'weekly':
                return Carbon::now()->next(Carbon::MONDAY)->setTime(6, 0);
            case 'monthly':
                return Carbon::now()->addMonth()->startOfMonth()->setTime(6, 0);
            case 'quarterly':
                return Carbon::now()->addQuarter()->startOfQuarter()->setTime(6, 0);
            default:
                return Carbon::tomorrow()->setTime(6, 0);
        }
    }
}
