<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Trip;
use App\Models\TripLocation;
use App\Models\Shipment;
use App\Models\Vehicle;
use App\Models\Driver;
use App\Models\TripFile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class TripController extends Controller
{
    public function index(Request $request)
    {
        $query = Trip::with(['vehicle', 'driver', 'locations']);

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('trip_number', 'like', "%{$search}%")
                    ->orWhere('shipment_reference', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $trips = $query->latest()->paginate(15);

        return view('admin.trips.index', compact('trips'));
    }

    public function create()
    {
        // Get pending shipments for optional assignment
        $shipments = Shipment::where('status', 'pending')
            ->with('vehicleType')
            ->orderBy('created_at', 'desc')
            ->get();

        // Get active vehicles
        $vehicles = Vehicle::where('status', 'active')
            ->with('vehicleType')
            ->orderBy('vehicle_number')
            ->get();

        $allVehicles = $vehicles;

        // Get only AVAILABLE drivers (active status, not on trip)
        $drivers = Driver::where('status', 'active')
            ->with('ownVehicle')
            ->orderBy('name')
            ->get();

        $allDrivers = $drivers;

        // Create a map of vehicle IDs to their owner driver IDs
        $vehicleOwners = Driver::where('driver_type', 'own_vehicle')
            ->whereNotNull('own_vehicle_id')
            ->pluck('id', 'own_vehicle_id')
            ->toArray();

        // Create a map of driver IDs to their own vehicle IDs
        $driverOwnVehicles = Driver::where('driver_type', 'own_vehicle')
            ->whereNotNull('own_vehicle_id')
            ->pluck('own_vehicle_id', 'id')
            ->toArray();

        // Vehicle availability
        $vehiclesInUse = Trip::whereIn('status', ['assigned', 'in_transit'])
            ->pluck('vehicle_id')
            ->toArray();

        $vehicleTrips = Trip::whereIn('status', ['assigned', 'in_transit'])
            ->get()
            ->pluck('trip_number', 'vehicle_id')
            ->toArray();

        // Driver availability
        $driversOnTrip = Trip::whereIn('status', ['assigned', 'in_transit'])
            ->pluck('driver_id')
            ->toArray();

        $driverTrips = Trip::whereIn('status', ['assigned', 'in_transit'])
            ->get()
            ->pluck('trip_number', 'driver_id')
            ->toArray();

        return view('admin.trips.create', compact(
            'shipments',
            'vehicles',
            'drivers',
            'allVehicles',
            'allDrivers',
            'vehicleOwners',
            'driverOwnVehicles',
            'vehiclesInUse',
            'vehicleTrips',
            'driversOnTrip',
            'driverTrips'
        ));
    }

    public function store(Request $request)
    {
        // Validation with lat/lng fields and multiple locations
        $validated = $request->validate([
            'vehicle_id' => 'required|exists:vehicles,id',
            'driver_id' => 'required|exists:drivers,id',
            'shipment_id' => 'nullable|exists:shipments,id',
            'shipment_reference' => 'nullable|string|max:100',
            'has_multiple_locations' => 'required|boolean',
            'trip_instructions' => 'nullable|string',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',

            // Pickup location (always required)
            'pickup_location' => 'required|string|max:500',
            'pickup_lat' => 'nullable|numeric|between:-90,90',
            'pickup_lng' => 'nullable|numeric|between:-180,180',

            // Single drop location (required if has_multiple_locations = 0)
            'drop_location' => 'required_if:has_multiple_locations,0|nullable|string|max:500',
            'drop_lat' => 'nullable|numeric|between:-90,90',
            'drop_lng' => 'nullable|numeric|between:-180,180',

            // Multiple drop locations (required if has_multiple_locations = 1)
            'locations' => 'required_if:has_multiple_locations,1|nullable|array|min:1',
            'locations.*.address' => 'required_with:locations|string|max:500',
            'locations.*.lat' => 'nullable|numeric|between:-90,90',
            'locations.*.lng' => 'nullable|numeric|between:-180,180',
            'locations.*.location_name' => 'nullable|string|max:255',
            'locations.*.contact_name' => 'nullable|string|max:255',
            'locations.*.contact_phone' => 'nullable|string|max:50',
            'locations.*.notes' => 'nullable|string|max:1000',
        ]);

        $driver = Driver::find($validated['driver_id']);

        // Vehicle availability check
        $vehicleInUse = Trip::whereIn('status', ['assigned', 'in_transit'])
            ->where('vehicle_id', $validated['vehicle_id'])
            ->first();

        if ($vehicleInUse) {
            return back()
                ->withErrors(['vehicle_id' => __('This vehicle is already in use on trip: ' . $vehicleInUse->trip_number)])
                ->withInput();
        }

        // Check if driver has own_vehicle
        if ($driver->driver_type === 'own_vehicle' && $driver->own_vehicle_id) {
            if ($driver->own_vehicle_id != $validated['vehicle_id']) {
                return back()
                    ->withErrors(['driver_id' => __('This driver can only be assigned to their own vehicle.')])
                    ->withInput();
            }
        }

        // Check if selected vehicle belongs to another own-vehicle driver
        $vehicleOwner = Driver::where('driver_type', 'own_vehicle')
            ->where('own_vehicle_id', $validated['vehicle_id'])
            ->first();

        if ($vehicleOwner && $vehicleOwner->id != $validated['driver_id']) {
            return back()
                ->withErrors(['vehicle_id' => __('This vehicle is owned by another driver.')])
                ->withInput();
        }

        // Check if driver is available
        if (!$driver->isAvailable()) {
            return back()
                ->withErrors(['driver_id' => __('This driver is not available.')])
                ->withInput();
        }

        DB::beginTransaction();
        try {
            // Generate trip number
            $lastTrip = Trip::latest('id')->first();
            $tripNumber = 'TRIP-' . str_pad(($lastTrip ? $lastTrip->id + 1 : 1), 6, '0', STR_PAD_LEFT);

            // Create trip
            $trip = Trip::create([
                'trip_number' => $tripNumber,
                'vehicle_id' => $validated['vehicle_id'],
                'driver_id' => $validated['driver_id'],
                'shipment_id' => $validated['shipment_id'] ?? null,
                'shipment_reference' => $validated['shipment_reference'] ?? null,
                'has_multiple_locations' => $validated['has_multiple_locations'],
                'pickup_location' => $validated['pickup_location'],
                'pickup_lat' => $validated['pickup_lat'] ?? null,
                'pickup_lng' => $validated['pickup_lng'] ?? null,

                // Single drop location (only if not multiple)
                'drop_location' => !$validated['has_multiple_locations']
                    ? ($validated['drop_location'] ?? null)
                    : null,
                'drop_lat' => !$validated['has_multiple_locations']
                    ? ($validated['drop_lat'] ?? null)
                    : null,
                'drop_lng' => !$validated['has_multiple_locations']
                    ? ($validated['drop_lng'] ?? null)
                    : null,

                'trip_instructions' => $validated['trip_instructions'],
                'status' => 'assigned',
                'start_date' => $validated['start_date'] ?? null,
                'end_date' => $validated['end_date'] ?? null,
                'created_by' => Auth::id(),
            ]);

            // Create multiple drop locations if has_multiple_locations = 1
            if ($validated['has_multiple_locations'] && !empty($validated['locations'])) {
                $sequence = 1;
                foreach ($validated['locations'] as $locationData) {
                    if (!empty($locationData['address'])) {
                        TripLocation::create([
                            'trip_id' => $trip->id,
                            'sequence' => $sequence++,
                            'location_name' => $locationData['location_name'] ?? null,
                            'address' => $locationData['address'],
                            'lat' => $locationData['lat'] ?? null,
                            'lng' => $locationData['lng'] ?? null,
                            'contact_name' => $locationData['contact_name'] ?? null,
                            'contact_phone' => $locationData['contact_phone'] ?? null,
                            'notes' => $locationData['notes'] ?? null,
                            'status' => 'pending',
                        ]);
                    }
                }
                Log::info("Created {$sequence} drop locations for trip {$tripNumber}");
            }

            // Update shipment if provided
            if ($validated['shipment_id']) {
                $shipment = Shipment::find($validated['shipment_id']);
                if ($shipment) {
                    $shipment->update([
                        'vehicle_id' => $validated['vehicle_id'],
                        'driver_id' => $validated['driver_id'],
                        'status' => 'assigned',
                    ]);
                }
            }

            // Set driver status to on_trip
            $driver->update(['status' => 'on_trip']);

            DB::commit();

            return redirect()->route('admin.trips.index')
                ->with('success', __('created_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Trip creation failed: ' . $e->getMessage());
            return back()->with('error', __('error') . ': ' . $e->getMessage())
                ->withInput();
        }
    }

    public function show(Trip $trip)
    {
        $trip->load(['vehicle.vehicleType', 'driver', 'shipment', 'cost', 'files', 'creator', 'deliveries', 'locations']);
        return view('admin.trips.show', compact('trip'));
    }

    public function edit(Trip $trip)
    {
        $trip->load('locations');

        // Get pending shipments + current shipment
        $shipments = Shipment::where(function ($q) use ($trip) {
            $q->where('status', 'pending')
                ->orWhere('id', $trip->shipment_id);
        })
            ->with('vehicleType')
            ->orderBy('created_at', 'desc')
            ->get();

        // Get active vehicles
        $vehicles = Vehicle::where('status', 'active')
            ->with('vehicleType')
            ->orderBy('vehicle_number')
            ->get();

        $allVehicles = $vehicles;

        // Get active drivers
        $drivers = Driver::where('status', 'active')
            ->orWhere('id', $trip->driver_id)
            ->with('ownVehicle')
            ->orderBy('name')
            ->get();

        $allDrivers = $drivers;

        // Maps for vehicle-driver relationships
        $vehicleOwners = Driver::where('driver_type', 'own_vehicle')
            ->whereNotNull('own_vehicle_id')
            ->pluck('id', 'own_vehicle_id')
            ->toArray();

        $driverOwnVehicles = Driver::where('driver_type', 'own_vehicle')
            ->whereNotNull('own_vehicle_id')
            ->pluck('own_vehicle_id', 'id')
            ->toArray();

        // Check availability (exclude current trip)
        $vehiclesInUse = Trip::whereIn('status', ['assigned', 'in_transit'])
            ->where('id', '!=', $trip->id)
            ->pluck('vehicle_id')
            ->toArray();

        $vehicleTrips = Trip::whereIn('status', ['assigned', 'in_transit'])
            ->where('id', '!=', $trip->id)
            ->get()
            ->pluck('trip_number', 'vehicle_id')
            ->toArray();

        $driversOnTrip = Trip::whereIn('status', ['assigned', 'in_transit'])
            ->where('id', '!=', $trip->id)
            ->pluck('driver_id')
            ->toArray();

        $driverTrips = Trip::whereIn('status', ['assigned', 'in_transit'])
            ->where('id', '!=', $trip->id)
            ->get()
            ->pluck('trip_number', 'driver_id')
            ->toArray();

        return view('admin.trips.edit', compact(
            'trip',
            'shipments',
            'vehicles',
            'drivers',
            'allVehicles',
            'allDrivers',
            'vehicleOwners',
            'driverOwnVehicles',
            'vehiclesInUse',
            'vehicleTrips',
            'driversOnTrip',
            'driverTrips'
        ));
    }

    public function update(Request $request, Trip $trip)
    {
        // Validation
        $validated = $request->validate([
            'vehicle_id' => 'required|exists:vehicles,id',
            'driver_id' => 'required|exists:drivers,id',
            'shipment_id' => 'nullable|exists:shipments,id',
            'shipment_reference' => 'nullable|string|max:100',
            'has_multiple_locations' => 'required|boolean',
            'trip_instructions' => 'nullable|string',
            'status' => 'required|in:pending,assigned,in_transit,delivered,cancelled',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',

            // Pickup
            'pickup_location' => 'required|string|max:500',
            'pickup_lat' => 'nullable|numeric|between:-90,90',
            'pickup_lng' => 'nullable|numeric|between:-180,180',

            // Single drop
            'drop_location' => 'required_if:has_multiple_locations,0|nullable|string|max:500',
            'drop_lat' => 'nullable|numeric|between:-90,90',
            'drop_lng' => 'nullable|numeric|between:-180,180',

            // Multiple drop locations
            'locations' => 'required_if:has_multiple_locations,1|nullable|array',
            'locations.*.id' => 'nullable|integer',
            'locations.*.address' => 'required_with:locations|string|max:500',
            'locations.*.lat' => 'nullable|numeric|between:-90,90',
            'locations.*.lng' => 'nullable|numeric|between:-180,180',
            'locations.*.location_name' => 'nullable|string|max:255',
            'locations.*.contact_name' => 'nullable|string|max:255',
            'locations.*.contact_phone' => 'nullable|string|max:50',
            'locations.*.notes' => 'nullable|string|max:1000',
        ]);

        $driver = Driver::find($validated['driver_id']);
        $oldDriverId = $trip->driver_id;
        $oldShipmentId = $trip->shipment_id;
        $oldStatus = $trip->status;

        // Vehicle availability check
        $vehicleInUse = Trip::whereIn('status', ['assigned', 'in_transit'])
            ->where('vehicle_id', $validated['vehicle_id'])
            ->where('id', '!=', $trip->id)
            ->first();

        if ($vehicleInUse) {
            return back()
                ->withErrors(['vehicle_id' => __('This vehicle is already in use on trip: ' . $vehicleInUse->trip_number)])
                ->withInput();
        }

        // Own vehicle driver check
        if ($driver->driver_type === 'own_vehicle' && $driver->own_vehicle_id) {
            if ($driver->own_vehicle_id != $validated['vehicle_id']) {
                return back()
                    ->withErrors(['driver_id' => __('This driver can only be assigned to their own vehicle.')])
                    ->withInput();
            }
        }

        // Vehicle ownership check
        $vehicleOwner = Driver::where('driver_type', 'own_vehicle')
            ->where('own_vehicle_id', $validated['vehicle_id'])
            ->first();

        if ($vehicleOwner && $vehicleOwner->id != $validated['driver_id']) {
            return back()->withErrors(['vehicle_id' => __('This vehicle is owned by another driver.')])
                ->withInput();
        }

        // Driver availability
        if ($oldDriverId != $validated['driver_id']) {
            if (!$driver->isAvailable()) {
                return back()->withErrors(['driver_id' => __('This driver is not available.')])
                    ->withInput();
            }
        }

        DB::beginTransaction();
        try {
            // Update trip
            $trip->update([
                'vehicle_id' => $validated['vehicle_id'],
                'driver_id' => $validated['driver_id'],
                'shipment_id' => $validated['shipment_id'] ?? null,
                'shipment_reference' => $validated['shipment_reference'] ?? null,
                'has_multiple_locations' => $validated['has_multiple_locations'],
                'pickup_location' => $validated['pickup_location'],
                'pickup_lat' => $validated['pickup_lat'] ?? null,
                'pickup_lng' => $validated['pickup_lng'] ?? null,
                'drop_location' => !$validated['has_multiple_locations'] ? ($validated['drop_location'] ?? null) : null,
                'drop_lat' => !$validated['has_multiple_locations'] ? ($validated['drop_lat'] ?? null) : null,
                'drop_lng' => !$validated['has_multiple_locations'] ? ($validated['drop_lng'] ?? null) : null,
                'trip_instructions' => $validated['trip_instructions'],
                'status' => $validated['status'],
                'start_date' => $validated['start_date'],
                'end_date' => $validated['end_date'],
            ]);

            // Handle multiple locations
            if ($validated['has_multiple_locations']) {
                // Get existing location IDs
                $existingIds = $trip->locations()->pluck('id')->toArray();
                $updatedIds = [];

                if (!empty($validated['locations'])) {
                    $sequence = 1;
                    foreach ($validated['locations'] as $locationData) {
                        if (empty($locationData['address'])) {
                            continue;
                        }

                        if (!empty($locationData['id']) && in_array($locationData['id'], $existingIds)) {
                            // Update existing location
                            TripLocation::where('id', $locationData['id'])
                                ->where('trip_id', $trip->id)
                                ->update([
                                    'sequence' => $sequence,
                                    'location_name' => $locationData['location_name'] ?? null,
                                    'address' => $locationData['address'],
                                    'lat' => $locationData['lat'] ?? null,
                                    'lng' => $locationData['lng'] ?? null,
                                    'contact_name' => $locationData['contact_name'] ?? null,
                                    'contact_phone' => $locationData['contact_phone'] ?? null,
                                    'notes' => $locationData['notes'] ?? null,
                                ]);
                            $updatedIds[] = $locationData['id'];
                        } else {
                            // Create new location
                            $newLocation = TripLocation::create([
                                'trip_id' => $trip->id,
                                'sequence' => $sequence,
                                'location_name' => $locationData['location_name'] ?? null,
                                'address' => $locationData['address'],
                                'lat' => $locationData['lat'] ?? null,
                                'lng' => $locationData['lng'] ?? null,
                                'contact_name' => $locationData['contact_name'] ?? null,
                                'contact_phone' => $locationData['contact_phone'] ?? null,
                                'notes' => $locationData['notes'] ?? null,
                                'status' => 'pending',
                            ]);
                            $updatedIds[] = $newLocation->id;
                        }
                        $sequence++;
                    }
                }

                // Delete removed locations
                $toDelete = array_diff($existingIds, $updatedIds);
                if (!empty($toDelete)) {
                    TripLocation::whereIn('id', $toDelete)->where('trip_id', $trip->id)->delete();
                }
            } else {
                // If switching to single location, delete all trip_locations
                $trip->locations()->delete();
            }

            // Handle shipment status
            if ($oldShipmentId && $oldShipmentId != $validated['shipment_id']) {
                $oldShipment = Shipment::find($oldShipmentId);
                if ($oldShipment) {
                    $oldShipment->update([
                        'vehicle_id' => null,
                        'driver_id' => null,
                        'status' => 'pending',
                    ]);
                }
            }

            if ($validated['shipment_id']) {
                $shipment = Shipment::find($validated['shipment_id']);
                if ($shipment) {
                    $shipmentData = [
                        'vehicle_id' => $validated['vehicle_id'],
                        'driver_id' => $validated['driver_id'],
                    ];

                    switch ($validated['status']) {
                        case 'assigned':
                        case 'in_transit':
                            $shipmentData['status'] = 'assigned';
                            break;
                        case 'delivered':
                            $shipmentData['status'] = 'delivered';
                            break;
                        case 'cancelled':
                        case 'pending':
                            $shipmentData['status'] = 'pending';
                            $shipmentData['vehicle_id'] = null;
                            $shipmentData['driver_id'] = null;
                            break;
                    }

                    $shipment->update($shipmentData);
                }
            }

            // Handle driver status
            if ($oldDriverId != $validated['driver_id']) {
                if ($oldDriverId) {
                    $oldDriver = Driver::find($oldDriverId);
                    if ($oldDriver) {
                        $oldDriver->update(['status' => 'active']);
                    }
                }

                if (in_array($validated['status'], ['assigned', 'in_transit'])) {
                    $driver->update(['status' => 'on_trip']);
                } else {
                    $driver->update(['status' => 'active']);
                }
            } elseif ($oldStatus != $validated['status']) {
                if (in_array($validated['status'], ['assigned', 'in_transit'])) {
                    $driver->update(['status' => 'on_trip']);
                } elseif (in_array($validated['status'], ['delivered', 'cancelled'])) {
                    $driver->update(['status' => 'active']);
                }
            }

            DB::commit();

            return redirect()->route('admin.trips.index')
                ->with('success', __('updated_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Trip update failed: ' . $e->getMessage());
            return back()->with('error', __('error') . ': ' . $e->getMessage())
                ->withInput();
        }
    }

    public function destroy(Trip $trip)
    {
        if (!in_array($trip->status, ['pending', 'cancelled'])) {
            return back()->with('error', __('Cannot delete trip that is in progress or delivered.'));
        }

        DB::beginTransaction();
        try {
            // Reset shipment
            if ($trip->shipment_id) {
                $shipment = Shipment::find($trip->shipment_id);
                if ($shipment) {
                    $shipment->update([
                        'vehicle_id' => null,
                        'driver_id' => null,
                        'status' => 'pending',
                    ]);
                }
            }

            // Free driver
            if ($trip->driver_id) {
                $driver = Driver::find($trip->driver_id);
                if ($driver && $driver->status === 'on_trip') {
                    $driver->update(['status' => 'active']);
                }
            }

            // Delete files
            foreach ($trip->files as $file) {
                Storage::delete($file->file_path);
            }

            // Delete locations (cascaded by foreign key, but explicit for clarity)
            $trip->locations()->delete();

            $trip->delete();

            DB::commit();

            return redirect()->route('admin.trips.index')
                ->with('success', __('deleted_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', __('error') . ': ' . $e->getMessage());
        }
    }

    /**
     * Update location sequence (for drag-drop reordering)
     */
    public function updateLocationSequence(Request $request, Trip $trip)
    {
        $validated = $request->validate([
            'locations' => 'required|array',
            'locations.*.id' => 'required|integer|exists:trip_locations,id',
            'locations.*.sequence' => 'required|integer|min:1',
        ]);

        DB::beginTransaction();
        try {
            foreach ($validated['locations'] as $locationData) {
                TripLocation::where('id', $locationData['id'])
                    ->where('trip_id', $trip->id)
                    ->update(['sequence' => $locationData['sequence']]);
            }

            DB::commit();

            return response()->json(['success' => true, 'message' => __('Sequence updated')]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function uploadFile(Request $request, Trip $trip)
    {
        $request->validate([
            'file' => 'required|file|max:10240',
        ]);

        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $fileName = time() . '_' . $file->getClientOriginalName();
            $filePath = $file->storeAs('trip-files', $fileName, 'public');

            TripFile::create([
                'trip_id' => $trip->id,
                'file_name' => $file->getClientOriginalName(),
                'file_path' => $filePath,
                'file_type' => $file->getClientMimeType(),
                'file_size' => $file->getSize(),
                'uploaded_by' => Auth::id(),
            ]);

            return back()->with('success', __('File uploaded successfully.'));
        }

        return back()->with('error', __('No file uploaded.'));
    }
}
