<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Trip;
use App\Models\Vehicle;
use App\Models\Driver;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TripReportController extends Controller
{
    /**
     * Display trip report with filters
     */
    public function index(Request $request)
    {
        $query = Trip::with(['vehicle', 'driver']);

        // Apply filters
        if ($request->filled('date_from')) {
            $query->whereDate('start_date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('start_date', '<=', $request->date_to);
        }
        if ($request->filled('vehicle_id')) {
            $query->where('vehicle_id', $request->vehicle_id);
        }
        if ($request->filled('driver_id')) {
            $query->where('driver_id', $request->driver_id);
        }
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Get trips
        $trips = $query->orderBy('start_date', 'desc')->paginate(20)->withQueryString();

        // Calculate summary statistics
        $summaryQuery = Trip::query();
        if ($request->filled('date_from')) {
            $summaryQuery->whereDate('start_date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $summaryQuery->whereDate('start_date', '<=', $request->date_to);
        }
        if ($request->filled('vehicle_id')) {
            $summaryQuery->where('vehicle_id', $request->vehicle_id);
        }
        if ($request->filled('driver_id')) {
            $summaryQuery->where('driver_id', $request->driver_id);
        }
        if ($request->filled('status')) {
            $summaryQuery->where('status', $request->status);
        }

        $totalTrips = $summaryQuery->count();
        $summary = [
            'total_trips' => $totalTrips,
            'completed' => (clone $summaryQuery)->where('status', 'completed')->count(),
            'in_transit' => (clone $summaryQuery)->where('status', 'in_transit')->count(),
            'pending' => (clone $summaryQuery)->where('status', 'pending')->count(),
            'cancelled' => (clone $summaryQuery)->where('status', 'cancelled')->count(),
        ];

        // Get data for filters
        $vehicles = Vehicle::where('status', 'active')->orderBy('vehicle_number')->get();
        $drivers = Driver::where('status', 'active')->orderBy('name')->get();

        // By status summary
        $byStatus = Trip::query()
            ->select('status', DB::raw('COUNT(*) as count'))
            ->when($request->filled('date_from'), fn($q) => $q->whereDate('start_date', '>=', $request->date_from))
            ->when($request->filled('date_to'), fn($q) => $q->whereDate('start_date', '<=', $request->date_to))
            ->when($request->filled('vehicle_id'), fn($q) => $q->where('vehicle_id', $request->vehicle_id))
            ->when($request->filled('driver_id'), fn($q) => $q->where('driver_id', $request->driver_id))
            ->groupBy('status')
            ->get();

        // By vehicle summary
        $byVehicle = Trip::query()
            ->select('vehicle_id', DB::raw('COUNT(*) as total_trips'), DB::raw('SUM(CASE WHEN status = "completed" THEN 1 ELSE 0 END) as completed'))
            ->when($request->filled('date_from'), fn($q) => $q->whereDate('start_date', '>=', $request->date_from))
            ->when($request->filled('date_to'), fn($q) => $q->whereDate('start_date', '<=', $request->date_to))
            ->when($request->filled('vehicle_id'), fn($q) => $q->where('vehicle_id', $request->vehicle_id))
            ->when($request->filled('driver_id'), fn($q) => $q->where('driver_id', $request->driver_id))
            ->when($request->filled('status'), fn($q) => $q->where('status', $request->status))
            ->groupBy('vehicle_id')
            ->with('vehicle')
            ->get();

        // By driver summary
        $byDriver = Trip::query()
            ->select('driver_id', DB::raw('COUNT(*) as total_trips'), DB::raw('SUM(CASE WHEN status = "completed" THEN 1 ELSE 0 END) as completed'))
            ->when($request->filled('date_from'), fn($q) => $q->whereDate('start_date', '>=', $request->date_from))
            ->when($request->filled('date_to'), fn($q) => $q->whereDate('start_date', '<=', $request->date_to))
            ->when($request->filled('vehicle_id'), fn($q) => $q->where('vehicle_id', $request->vehicle_id))
            ->when($request->filled('driver_id'), fn($q) => $q->where('driver_id', $request->driver_id))
            ->when($request->filled('status'), fn($q) => $q->where('status', $request->status))
            ->groupBy('driver_id')
            ->with('driver')
            ->get();

        return view('admin.reports.trip.index', compact(
            'trips',
            'summary',
            'vehicles',
            'drivers',
            'byStatus',
            'byVehicle',
            'byDriver'
        ));
    }

    /**
     * Export trip report
     */
    public function export(Request $request)
    {
        $format = $request->get('format', 'csv');

        $query = Trip::with(['vehicle', 'driver']);

        // Apply filters
        if ($request->filled('date_from')) {
            $query->whereDate('start_date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('start_date', '<=', $request->date_to);
        }
        if ($request->filled('vehicle_id')) {
            $query->where('vehicle_id', $request->vehicle_id);
        }
        if ($request->filled('driver_id')) {
            $query->where('driver_id', $request->driver_id);
        }
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $trips = $query->orderBy('start_date', 'desc')->get();

        // Calculate summary
        $summary = [
            'total_trips' => $trips->count(),
            'completed' => $trips->where('status', 'completed')->count(),
            'in_transit' => $trips->where('status', 'in_transit')->count(),
            'pending' => $trips->where('status', 'pending')->count(),
            'cancelled' => $trips->where('status', 'cancelled')->count(),
        ];

        $filename = 'trip_report_' . date('Y-m-d_His');

        switch ($format) {
            case 'pdf':
                return $this->exportPdf($trips, $summary, $filename, $request);
            case 'html':
                return $this->exportHtml($trips, $summary, $filename, $request);
            case 'excel':
            case 'csv':
            default:
                return $this->exportCsv($trips, $summary, $filename);
        }
    }

    /**
     * Export as CSV
     */
    private function exportCsv($trips, $summary, $filename)
    {
        $output = fopen('php://temp', 'r+');

        // BOM for Excel
        fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

        // Header
        fputcsv($output, ['TRIP SUMMARY REPORT']);
        fputcsv($output, ['Generated: ' . now()->format('Y-m-d H:i:s')]);
        fputcsv($output, []);

        // Summary
        fputcsv($output, ['=== SUMMARY ===']);
        fputcsv($output, ['Total Trips', $summary['total_trips']]);
        fputcsv($output, ['Completed', $summary['completed']]);
        fputcsv($output, ['In Transit', $summary['in_transit']]);
        fputcsv($output, ['Pending', $summary['pending']]);
        fputcsv($output, ['Cancelled', $summary['cancelled']]);
        fputcsv($output, []);

        // Data
        fputcsv($output, ['=== TRIP DETAILS ===']);
        fputcsv($output, ['Trip Number', 'Scheduled Date', 'Vehicle', 'Driver', 'Origin', 'Destination', 'Status', 'Start Time', 'End Time']);

        foreach ($trips as $trip) {
            fputcsv($output, [
                $trip->trip_number,
                $trip->start_date,
                $trip->vehicle->vehicle_number ?? '-',
                $trip->driver->name ?? '-',
                $trip->origin ?? '-',
                $trip->destination ?? '-',
                ucfirst($trip->status),
                $trip->actual_start_time ?? '-',
                $trip->actual_end_time ?? '-',
            ]);
        }

        rewind($output);
        $content = stream_get_contents($output);
        fclose($output);

        return response($content)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '.csv"');
    }

    /**
     * Export as HTML
     */
    private function exportHtml($trips, $summary, $filename, $request)
    {
        $html = $this->generateHtml($trips, $summary, $request);

        return response($html)
            ->header('Content-Type', 'text/html')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '.html"');
    }

    /**
     * Export as PDF
     */
    private function exportPdf($trips, $summary, $filename, $request)
    {
        $html = $this->generateHtml($trips, $summary, $request);

        if (class_exists(\Barryvdh\DomPDF\Facade\Pdf::class)) {
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadHTML($html);
            $pdf->setPaper('A4', 'landscape');
            return $pdf->download($filename . '.pdf');
        } elseif (class_exists(\Dompdf\Dompdf::class)) {
            $dompdf = new \Dompdf\Dompdf();
            $dompdf->loadHtml($html);
            $dompdf->setPaper('A4', 'landscape');
            $dompdf->render();

            return response($dompdf->output())
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="' . $filename . '.pdf"');
        }

        return response($html)
            ->header('Content-Type', 'text/html')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '.html"');
    }

    /**
     * Generate HTML content for export
     */
    private function generateHtml($trips, $summary, $request)
    {
        $dateRange = '';
        if ($request->filled('date_from') || $request->filled('date_to')) {
            $dateRange = ($request->date_from ?? 'All') . ' to ' . ($request->date_to ?? 'All');
        } else {
            $dateRange = 'All Time';
        }

        $completionRate = $summary['total_trips'] > 0
            ? round(($summary['completed'] / $summary['total_trips']) * 100, 1)
            : 0;

        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Trip Summary Report</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 11px; color: #333; padding: 20px; }
        .header { text-align: center; margin-bottom: 20px; border-bottom: 2px solid #3b82f6; padding-bottom: 15px; }
        .header h1 { color: #3b82f6; font-size: 22px; margin-bottom: 5px; }
        .header .subtitle { color: #6b7280; font-size: 12px; }
        .meta { margin-bottom: 20px; background: #eff6ff; padding: 10px; border-radius: 5px; }
        .summary { margin-bottom: 20px; background: #f3f4f6; padding: 15px; border-radius: 5px; }
        .summary-grid { display: table; width: 100%; }
        .summary-item { display: table-cell; text-align: center; padding: 10px; background: white; border-radius: 5px; }
        .summary-value { font-size: 18px; font-weight: bold; color: #3b82f6; }
        .summary-label { font-size: 9px; color: #6b7280; text-transform: uppercase; }
        .status-completed { color: #16a34a; }
        .status-in_transit { color: #2563eb; }
        .status-pending { color: #ca8a04; }
        .status-cancelled { color: #dc2626; }
        table { width: 100%; border-collapse: collapse; margin-top: 15px; }
        th { background: #3b82f6; color: white; padding: 8px; text-align: left; font-size: 10px; }
        td { padding: 6px 8px; border-bottom: 1px solid #e5e7eb; font-size: 10px; }
        tr:nth-child(even) { background: #eff6ff; }
        .footer { margin-top: 20px; text-align: center; color: #6b7280; font-size: 9px; border-top: 1px solid #e5e7eb; padding-top: 10px; }
        .badge { padding: 2px 6px; border-radius: 3px; font-size: 9px; font-weight: bold; }
        .badge-completed { background: #dcfce7; color: #16a34a; }
        .badge-in_transit { background: #dbeafe; color: #2563eb; }
        .badge-pending { background: #fef9c3; color: #ca8a04; }
        .badge-cancelled { background: #fee2e2; color: #dc2626; }
    </style>
</head>
<body>
    <div class="header">
        <h1>Trip Summary Report</h1>
        <div class="subtitle">LogiFleet Fleet Management System</div>
    </div>
    
    <div class="meta">
        <table style="width: 100%;">
            <tr>
                <td style="text-align: center; padding: 5px;">
                    <div style="font-size: 9px; color: #6b7280;">Date Range</div>
                    <div style="font-weight: bold; color: #1e40af;">' . $dateRange . '</div>
                </td>
                <td style="text-align: center; padding: 5px;">
                    <div style="font-size: 9px; color: #6b7280;">Generated</div>
                    <div style="font-weight: bold; color: #1e40af;">' . now()->format('M d, Y H:i') . '</div>
                </td>
                <td style="text-align: center; padding: 5px;">
                    <div style="font-size: 9px; color: #6b7280;">Completion Rate</div>
                    <div style="font-weight: bold; color: #16a34a;">' . $completionRate . '%</div>
                </td>
            </tr>
        </table>
    </div>

    <div class="summary">
        <table style="width: 100%;">
            <tr>
                <td style="text-align: center; padding: 10px; background: white; border-radius: 5px;">
                    <div class="summary-value">' . $summary['total_trips'] . '</div>
                    <div class="summary-label">Total Trips</div>
                </td>
                <td style="text-align: center; padding: 10px; background: white; border-radius: 5px;">
                    <div class="summary-value status-completed">' . $summary['completed'] . '</div>
                    <div class="summary-label">Completed</div>
                </td>
                <td style="text-align: center; padding: 10px; background: white; border-radius: 5px;">
                    <div class="summary-value status-in_transit">' . $summary['in_transit'] . '</div>
                    <div class="summary-label">In Transit</div>
                </td>
                <td style="text-align: center; padding: 10px; background: white; border-radius: 5px;">
                    <div class="summary-value status-pending">' . $summary['pending'] . '</div>
                    <div class="summary-label">Pending</div>
                </td>
                <td style="text-align: center; padding: 10px; background: white; border-radius: 5px;">
                    <div class="summary-value status-cancelled">' . $summary['cancelled'] . '</div>
                    <div class="summary-label">Cancelled</div>
                </td>
            </tr>
        </table>
    </div>

    <h3 style="margin-bottom: 10px; color: #374151;">Trip Details</h3>
    <table>
        <thead>
            <tr>
                <th>Trip Number</th>
                <th>Scheduled Date</th>
                <th>Vehicle</th>
                <th>Driver</th>
                <th>Origin</th>
                <th>Destination</th>
                <th>Status</th>
                <th>Start Time</th>
                <th>End Time</th>
            </tr>
        </thead>
        <tbody>';

        foreach ($trips as $trip) {
            $statusClass = 'badge-' . $trip->status;
            $html .= '<tr>
                <td>' . htmlspecialchars($trip->trip_number) . '</td>
                <td>' . $trip->start_date . '</td>
                <td>' . htmlspecialchars($trip->vehicle->vehicle_number ?? '-') . '</td>
                <td>' . htmlspecialchars($trip->driver->name ?? '-') . '</td>
                <td>' . htmlspecialchars($trip->origin ?? '-') . '</td>
                <td>' . htmlspecialchars($trip->destination ?? '-') . '</td>
                <td><span class="badge ' . $statusClass . '">' . ucfirst(str_replace('_', ' ', $trip->status)) . '</span></td>
                <td>' . ($trip->actual_start_time ?? '-') . '</td>
                <td>' . ($trip->actual_end_time ?? '-') . '</td>
            </tr>';
        }

        $html .= '</tbody>
    </table>
    
    <div class="footer">
        <p>Generated by LogiFleet Fleet Management System on ' . now()->format('F d, Y \a\t h:i A') . '</p>
    </div>
</body>
</html>';

        return $html;
    }
}
