<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Driver;
use App\Models\Trip;
use App\Models\Delivery;
use App\Models\Vehicle;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class DriverApiController extends Controller
{
    /**
     * Driver Login
     * POST /api/driver/login
     */
    public function login(Request $request)
    {
        $validated = $request->validate([
            'mobile' => 'required|string',
            'password' => 'required|string',
        ]);

        $driver = Driver::where('mobile', $validated['mobile'])->first();

        if (!$driver || !Hash::check($validated['password'], $driver->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        if ($driver->status === 'inactive') {
            return response()->json([
                'success' => false,
                'message' => 'Your account is inactive. Please contact administrator.'
            ], 403);
        }

        // Create Sanctum token
        $token = $driver->createToken('driver-app')->plainTextToken;

        // Load relationships
        $driver->load('ownVehicle.vehicleType');

        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'data' => [
                'driver' => [
                    'id' => $driver->id,
                    'name' => $driver->name,
                    'mobile' => $driver->mobile,
                    'license_number' => $driver->license_number,
                    'driver_type' => $driver->driver_type,
                    'status' => $driver->status,
                    'address' => $driver->address,
                    'emergency_contact' => $driver->emergency_contact,
                    'own_vehicle' => $driver->ownVehicle ? [
                        'id' => $driver->ownVehicle->id,
                        'vehicle_number' => $driver->ownVehicle->vehicle_number,
                        'vehicle_type' => $driver->ownVehicle->vehicleType->name ?? null,
                    ] : null,
                ],
                'token' => $token
            ]
        ], 200);
    }

    /**
     * Driver Logout
     * POST /api/driver/logout
     */
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Logged out successfully'
        ], 200);
    }

    /**
     * Get Driver Profile
     * GET /api/driver/profile
     */
    public function profile(Request $request)
    {
        $driver = $request->user();
        $driver->load('ownVehicle.vehicleType');

        return response()->json([
            'success' => true,
            'data' => [
                'id' => $driver->id,
                'name' => $driver->name,
                'mobile' => $driver->mobile,
                'license_number' => $driver->license_number,
                'driver_type' => $driver->driver_type,
                'status' => $driver->status,
                'address' => $driver->address,
                'emergency_contact' => $driver->emergency_contact,
                'own_vehicle' => $driver->ownVehicle ? [
                    'id' => $driver->ownVehicle->id,
                    'vehicle_number' => $driver->ownVehicle->vehicle_number,
                    'vehicle_type' => $driver->ownVehicle->vehicleType->name ?? null,
                    'status' => $driver->ownVehicle->status,
                ] : null,
            ]
        ], 200);
    }

    /**
     * Get Dashboard Statistics
     * GET /api/driver/dashboard
     */
    public function dashboard(Request $request)
    {
        $driver = $request->user();

        $stats = [
            'total_trips' => Trip::where('driver_id', $driver->id)->count(),
            'pending_trips' => Trip::where('driver_id', $driver->id)
                ->whereIn('status', ['pending', 'assigned'])
                ->count(),
            'in_transit_trips' => Trip::where('driver_id', $driver->id)
                ->where('status', 'in_transit')
                ->count(),
            'completed_trips' => Trip::where('driver_id', $driver->id)
                ->where('status', 'delivered')
                ->count(),
            'cancelled_trips' => Trip::where('driver_id', $driver->id)
                ->where('status', 'cancelled')
                ->count(),
        ];

        // Get recent trips
        $recentTrips = Trip::where('driver_id', $driver->id)
            ->with(['vehicle.vehicleType', 'shipment', 'deliveries'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get()
            ->map(function ($trip) {
                return $this->formatTripData($trip);
            });

        return response()->json([
            'success' => true,
            'data' => [
                'statistics' => $stats,
                'recent_trips' => $recentTrips,
                'driver_status' => $driver->status,
            ]
        ], 200);
    }

    /**
     * Get All Trips for Driver
     * GET /api/driver/trips
     */
    public function trips(Request $request)
    {
        $driver = $request->user();

        $query = Trip::where('driver_id', $driver->id)
            ->with(['vehicle.vehicleType', 'shipment', 'deliveries']);

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Sort by latest
        $trips = $query->orderBy('created_at', 'desc')->get();

        $formattedTrips = $trips->map(function ($trip) {
            return $this->formatTripData($trip);
        });

        return response()->json([
            'success' => true,
            'data' => $formattedTrips
        ], 200);
    }

    /**
     * Get Single Trip Details
     * GET /api/driver/trips/{id}
     */
    public function tripDetails(Request $request, $id)
    {
        $driver = $request->user();

        $trip = Trip::where('id', $id)
            ->where('driver_id', $driver->id)
            ->with(['vehicle.vehicleType', 'shipment', 'deliveries', 'cost', 'files'])
            ->first();

        if (!$trip) {
            return response()->json([
                'success' => false,
                'message' => 'Trip not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $this->formatTripData($trip, true)
        ], 200);
    }

    /**
     * Update Trip Status
     * POST /api/driver/trips/{id}/update-status
     */
    public function updateTripStatus(Request $request, $id)
    {
        $driver = $request->user();

        $validated = $request->validate([
            'status' => 'required|in:in_transit,delivered,cancelled',
            'remarks' => 'nullable|string|max:1000',
        ]);

        $trip = Trip::where('id', $id)
            ->where('driver_id', $driver->id)
            ->first();

        if (!$trip) {
            return response()->json([
                'success' => false,
                'message' => 'Trip not found'
            ], 404);
        }

        // Validate status transitions
        if ($trip->status === 'delivered' || $trip->status === 'cancelled') {
            return response()->json([
                'success' => false,
                'message' => 'Cannot update a completed or cancelled trip'
            ], 400);
        }

        DB::beginTransaction();
        try {
            $oldStatus = $trip->status;
            $trip->status = $validated['status'];

            // Set dates based on status
            if ($validated['status'] === 'in_transit' && !$trip->start_date) {
                $trip->start_date = now();
            }

            if ($validated['status'] === 'delivered') {
                $trip->end_date = now();

                // Update driver status back to active
                $driver->status = 'active';
                $driver->save();

                // Update vehicle status back to available
                if ($trip->vehicle) {
                    $trip->vehicle->status = 'available';
                    $trip->vehicle->save();
                }

                // Update shipment status if linked
                if ($trip->shipment) {
                    $trip->shipment->status = 'delivered';
                    $trip->shipment->save();
                }
            }

            if ($validated['status'] === 'cancelled') {
                // Update driver status back to active
                $driver->status = 'active';
                $driver->save();

                // Update vehicle status back to available
                if ($trip->vehicle) {
                    $trip->vehicle->status = 'available';
                    $trip->vehicle->save();
                }

                // Update shipment status if linked
                if ($trip->shipment) {
                    $trip->shipment->status = 'cancelled';
                    $trip->shipment->save();
                }
            }

            $trip->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Trip status updated successfully',
                'data' => $this->formatTripData($trip->fresh())
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to update trip status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Create/Update Delivery for Trip
     * POST /api/driver/trips/{id}/delivery
     */
    public function createDelivery(Request $request, $id)
    {
        $driver = $request->user();

        $validated = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_phone' => 'nullable|string|max:20',
            'customer_email' => 'nullable|email|max:255',
            'delivery_address' => 'required|string',
            'delivery_remarks' => 'nullable|string',
            'delivery_status' => 'required|in:pending,partial,completed,failed',
            'pod_photo' => 'nullable|image|mimes:jpeg,jpg,png|max:5120', // 5MB max
            'signature_photo' => 'nullable|image|mimes:jpeg,jpg,png|max:2048', // 2MB max
        ]);

        $trip = Trip::where('id', $id)
            ->where('driver_id', $driver->id)
            ->first();

        if (!$trip) {
            return response()->json([
                'success' => false,
                'message' => 'Trip not found'
            ], 404);
        }

        DB::beginTransaction();
        try {
            // Handle POD photo upload
            $podPath = null;
            if ($request->hasFile('pod_photo')) {
                $podPath = $request->file('pod_photo')->store('deliveries/pod', 'public');
            }

            // Handle signature photo upload
            $signaturePath = null;
            if ($request->hasFile('signature_photo')) {
                $signaturePath = $request->file('signature_photo')->store('deliveries/signatures', 'public');
            }

            // Get admin user ID for created_by
            $adminUser = DB::table('users')->where('email', 'admin@logiflow.com')->first();
            $createdById = $adminUser ? $adminUser->id : 1;

            $delivery = Delivery::create([
                'trip_id' => $trip->id,
                'customer_name' => $validated['customer_name'],
                'customer_phone' => $validated['customer_phone'] ?? null,
                'customer_email' => $validated['customer_email'] ?? null,
                'delivery_address' => $validated['delivery_address'],
                'delivered_at' => now(),
                'signature_path' => $signaturePath,
                'pod_file_path' => $podPath,
                'delivery_remarks' => $validated['delivery_remarks'] ?? null,
                'delivery_status' => $validated['delivery_status'],
                'created_by' => $createdById,
            ]);

            // If delivery is completed, update trip status
            if ($validated['delivery_status'] === 'completed') {
                $trip->status = 'delivered';
                $trip->end_date = now();
                $trip->save();

                // Update driver and vehicle status
                $driver->status = 'active';
                $driver->save();


                // Update shipment status if linked
                if ($trip->shipment) {
                    $trip->shipment->status = 'delivered';
                    $trip->shipment->save();
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Delivery recorded successfully',
                'data' => [
                    'delivery_id' => $delivery->id,
                    'trip_status' => $trip->status,
                ]
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to create delivery: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get Deliveries for a Trip
     * GET /api/driver/trips/{id}/deliveries
     */
    public function tripDeliveries(Request $request, $id)
    {
        $driver = $request->user();

        $trip = Trip::where('id', $id)
            ->where('driver_id', $driver->id)
            ->first();

        if (!$trip) {
            return response()->json([
                'success' => false,
                'message' => 'Trip not found'
            ], 404);
        }

        $deliveries = Delivery::where('trip_id', $trip->id)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($delivery) {
                return [
                    'id' => $delivery->id,
                    'customer_name' => $delivery->customer_name,
                    'customer_phone' => $delivery->customer_phone,
                    'customer_email' => $delivery->customer_email,
                    'delivery_address' => $delivery->delivery_address,
                    'delivered_at' => $delivery->delivered_at,
                    'signature_path' => $delivery->signature_path ? url('storage/' . $delivery->signature_path) : null,
                    'pod_file_path' => $delivery->pod_file_path ? url('storage/' . $delivery->pod_file_path) : null,
                    'delivery_remarks' => $delivery->delivery_remarks,
                    'delivery_status' => $delivery->delivery_status,
                    'created_at' => $delivery->created_at->format('Y-m-d H:i:s'),
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $deliveries
        ], 200);
    }

    /**
     * Helper: Format Trip Data
     */
    private function formatTripData($trip, $detailed = false)
    {
        $data = [
            'id' => $trip->id,
            'trip_number' => $trip->trip_number,
            'status' => $trip->status,
            'pickup_location' => $trip->pickup_location,
            'drop_location' => $trip->drop_location,
            'has_multiple_locations' => (bool) $trip->has_multiple_locations,
            'trip_instructions' => $trip->trip_instructions,
            'start_date' => $trip->start_date,
            'end_date' => $trip->end_date,
            'created_at' => $trip->created_at->format('Y-m-d H:i:s'),
            'vehicle' => $trip->vehicle ? [
                'id' => $trip->vehicle->id,
                'vehicle_number' => $trip->vehicle->vehicle_number,
                'vehicle_type' => $trip->vehicle->vehicleType->name ?? null,
                'status' => $trip->vehicle->status,
            ] : null,
            'shipment' => $trip->shipment ? [
                'id' => $trip->shipment->id,
                'shipment_number' => $trip->shipment->shipment_number,
                'status' => $trip->shipment->status,
                'priority' => $trip->shipment->priority,
                'cargo_weight' => $trip->shipment->cargo_weight,
            ] : null,
            'deliveries_count' => $trip->deliveries ? $trip->deliveries->count() : 0,
        ];

        if ($detailed) {
            $data['deliveries'] = $trip->deliveries ? $trip->deliveries->map(function ($delivery) {
                return [
                    'id' => $delivery->id,
                    'customer_name' => $delivery->customer_name,
                    'customer_phone' => $delivery->customer_phone,
                    'delivery_address' => $delivery->delivery_address,
                    'delivered_at' => $delivery->delivered_at,
                    'delivery_status' => $delivery->delivery_status,
                    'signature_path' => $delivery->signature_path ? url('storage/' . $delivery->signature_path) : null,
                    'pod_file_path' => $delivery->pod_file_path ? url('storage/' . $delivery->pod_file_path) : null,
                ];
            }) : [];

            $data['trip_costs'] = $trip->cost ? [[
                'id' => $trip->cost->id,
                'cost_type' => $trip->cost->cost_type,
                'amount' => $trip->cost->amount,
                'remarks' => $trip->cost->remarks,
            ]] : [];

            $data['trip_files'] = $trip->files ? $trip->files->map(function ($file) {
                return [
                    'id' => $file->id,
                    'file_name' => $file->file_name,
                    'file_path' => $file->file_path ? url('storage/' . $file->file_path) : null,
                    'file_type' => $file->file_type,
                    'file_size' => $file->file_size,
                ];
            }) : [];
        }

        return $data;
    }
}
