<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Trip;
use App\Models\TripLocation;
use App\Models\Delivery;

class RouteOptimizerController extends Controller
{
    /**
     * Display the route optimizer form
     */
    public function index()
    {
        // Get active trips with locations
        $trips = Trip::with(['vehicle', 'driver', 'locations', 'deliveries'])
            ->whereIn('status', ['assigned', 'in_transit'])
            ->where(function ($query) {
                // Trips with multiple locations in trip_locations table
                $query->whereHas('locations')
                    // OR trips with both pickup and drop coordinates
                    ->orWhere(function ($q) {
                        $q->whereNotNull('pickup_lat')
                            ->whereNotNull('pickup_lng')
                            ->where(function ($inner) {
                                // Has drop coordinates
                                $inner->whereNotNull('drop_lat')
                                    ->whereNotNull('drop_lng');
                            });
                    });
            })
            ->orderBy('created_at', 'desc')
            ->get();

        return view('route-optimizer.index', compact('trips'));
    }

    /**
     * Optimize route for a specific trip
     */
    public function optimize(Request $request)
    {
        $validated = $request->validate([
            'trip_id' => 'required|exists:trips,id',
            'optimization_priority' => 'required|in:distance,time,fuel',
            'avoid_highways' => 'boolean',
            'avoid_tolls' => 'boolean',
        ]);

        $trip = Trip::with(['locations', 'vehicle', 'driver'])->findOrFail($validated['trip_id']);

        // Prepare locations with coordinates
        $locations = $this->prepareLocations($trip);

        // Check if we have enough locations with coordinates
        $locationsWithCoords = array_filter($locations, fn($loc) => $loc['latitude'] != 0 && $loc['longitude'] != 0);

        if (count($locationsWithCoords) < 2) {
            return back()->with('error', __('At least 2 locations with coordinates are required for route optimization.'));
        }

        // Call OpenAI for route optimization
        $optimizedRoute = $this->callOpenAI($locations, $validated);

        if (!$optimizedRoute['success']) {
            return back()->with('error', $optimizedRoute['message']);
        }

        return view('route-optimizer.result', [
            'trip' => $trip,
            'optimizedRoute' => $optimizedRoute['data'],
            'originalLocations' => $locations,
            'settings' => $validated,
        ]);
    }

    /**
     * API endpoint for Flutter app
     */
    public function apiOptimize(Request $request)
    {
        $validated = $request->validate([
            'trip_id' => 'required|exists:trips,id',
            'optimization_priority' => 'required|in:distance,time,fuel',
            'avoid_highways' => 'boolean',
            'avoid_tolls' => 'boolean',
            'current_latitude' => 'nullable|numeric',
            'current_longitude' => 'nullable|numeric',
        ]);

        $trip = Trip::with(['locations', 'vehicle', 'driver'])->findOrFail($validated['trip_id']);

        // Prepare locations with coordinates
        $locations = $this->prepareLocations(
            $trip,
            $validated['current_latitude'] ?? null,
            $validated['current_longitude'] ?? null
        );

        $locationsWithCoords = array_filter($locations, fn($loc) => $loc['latitude'] != 0 && $loc['longitude'] != 0);

        if (count($locationsWithCoords) < 2) {
            return response()->json([
                'success' => false,
                'message' => __('At least 2 locations with coordinates are required.'),
            ], 400);
        }

        // Call OpenAI for route optimization
        $optimizedRoute = $this->callOpenAI($locations, $validated);

        if (!$optimizedRoute['success']) {
            return response()->json([
                'success' => false,
                'message' => $optimizedRoute['message'],
            ], 500);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'trip_id' => $trip->id,
                'optimized_route' => $optimizedRoute['data'],
                'original_locations' => $locations,
                'settings' => $validated,
            ],
        ]);
    }

    /**
     * Prepare locations array from trip data with coordinates
     */
    private function prepareLocations(Trip $trip, ?float $currentLat = null, ?float $currentLng = null): array
    {
        $locations = [];

        // 1. Add PICKUP location as starting point
        if (!empty($trip->pickup_location)) {
            $locations[] = [
                'id' => 'pickup',
                'name' => __('Pickup Location'),
                'address' => $trip->pickup_location,
                'latitude' => $currentLat ?? (float) ($trip->pickup_lat ?? 0),
                'longitude' => $currentLng ?? (float) ($trip->pickup_lng ?? 0),
                'type' => 'origin',
                'sequence' => 0,
            ];
        }

        // 2. If trip has multiple locations - use trip_locations table
        if ($trip->has_multiple_locations && $trip->locations->isNotEmpty()) {
            foreach ($trip->locations as $location) {
                $locations[] = [
                    'id' => 'location_' . $location->id,
                    'name' => $location->location_name ?? __('Drop') . ' #' . $location->sequence,
                    'address' => $location->address,
                    'latitude' => (float) ($location->lat ?? 0),
                    'longitude' => (float) ($location->lng ?? 0),
                    'type' => 'drop',
                    'sequence' => $location->sequence,
                    'contact_name' => $location->contact_name,
                    'contact_phone' => $location->contact_phone,
                    'notes' => $location->notes,
                    'status' => $location->status ?? 'pending',
                ];
            }
        }
        // 3. Single drop location
        elseif (!empty($trip->drop_location)) {
            $locations[] = [
                'id' => 'drop',
                'name' => __('Drop Location'),
                'address' => $trip->drop_location,
                'latitude' => (float) ($trip->drop_lat ?? 0),
                'longitude' => (float) ($trip->drop_lng ?? 0),
                'type' => 'destination',
                'sequence' => 1,
            ];
        }

        // 4. Also include deliveries if present (legacy support)
        if ($trip->deliveries && $trip->deliveries->isNotEmpty()) {
            foreach ($trip->deliveries as $delivery) {
                // Check if not already added via trip_locations
                $existingIds = array_column($locations, 'id');
                $deliveryId = 'delivery_' . $delivery->id;

                if (!in_array($deliveryId, $existingIds)) {
                    $locations[] = [
                        'id' => $deliveryId,
                        'name' => $delivery->customer_name ?? __('Delivery') . ' #' . $delivery->id,
                        'address' => $delivery->delivery_address,
                        'latitude' => 0, // Deliveries don't have coords yet
                        'longitude' => 0,
                        'type' => 'delivery',
                        'contact_name' => $delivery->customer_name,
                        'contact_phone' => $delivery->customer_phone,
                        'status' => $delivery->delivery_status ?? 'pending',
                    ];
                }
            }
        }

        return $locations;
    }

    /**
     * Call OpenAI API for route optimization
     */
    private function callOpenAI(array $locations, array $settings): array
    {
        $apiKey = config('services.openai.api_key');

        if (empty($apiKey)) {
            return [
                'success' => false,
                'message' => __('OpenAI API key not configured. Please add OPENAI_API_KEY to your .env file.'),
            ];
        }

        $prompt = $this->buildOptimizationPrompt($locations, $settings);

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(60)->post('https://api.openai.com/v1/chat/completions', [
                'model' => config('services.openai.model', 'gpt-4o'),
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a logistics route optimization expert. Analyze delivery locations with their GPS coordinates and provide the most efficient route order. Use the provided latitude/longitude coordinates to calculate accurate distances. Always respond with valid JSON only, no markdown or extra text.',
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt,
                    ],
                ],
                'temperature' => 0.3,
                'max_tokens' => 2000,
            ]);

            if (!$response->successful()) {
                Log::error('OpenAI API Error', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);
                return [
                    'success' => false,
                    'message' => __('Failed to get optimization from AI service.'),
                ];
            }

            $result = $response->json();
            $content = $result['choices'][0]['message']['content'] ?? '';

            $optimizedData = $this->parseAIResponse($content, $locations);

            return [
                'success' => true,
                'data' => $optimizedData,
            ];
        } catch (\Exception $e) {
            Log::error('Route Optimization Error', [
                'message' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => __('An error occurred during route optimization: ') . $e->getMessage(),
            ];
        }
    }

    /**
     * Build the optimization prompt for OpenAI with coordinates
     */
    private function buildOptimizationPrompt(array $locations, array $settings): string
    {
        $locationsJson = json_encode($locations, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        $priority = $settings['optimization_priority'];
        $avoidHighways = !empty($settings['avoid_highways']);
        $avoidTolls = !empty($settings['avoid_tolls']);

        $constraints = [];
        if ($avoidHighways) $constraints[] = 'avoid highways';
        if ($avoidTolls) $constraints[] = 'avoid toll roads';
        $constraintsText = !empty($constraints) ? implode(', ', $constraints) : 'none';

        return <<<PROMPT
Optimize the following delivery route for a logistics fleet.

LOCATIONS TO VISIT (with GPS coordinates):
{$locationsJson}

OPTIMIZATION PRIORITY: {$priority}
- distance: Minimize total travel distance using GPS coordinates
- time: Minimize total travel time considering typical traffic
- fuel: Minimize fuel consumption (consider distance and traffic patterns)

CONSTRAINTS: {$constraintsText}

INSTRUCTIONS:
1. The route MUST start from the location with type "origin" (pickup point)
2. Use the latitude/longitude coordinates to calculate actual road distances
3. Visit all "drop" and "delivery" locations in the optimal order
4. End at the last delivery point (no return to origin required)
5. Consider traffic patterns and road conditions based on location
6. Locations with coordinates (0, 0) should be placed based on address only

IMPORTANT: Use the GPS coordinates provided to calculate accurate distances and durations.

Respond with ONLY a valid JSON object (no markdown, no code blocks) in this exact format:
{
    "optimized_order": [
        {
            "sequence": 1,
            "location_id": "pickup",
            "location_name": "Pickup Location",
            "address": "address here",
            "latitude": 13.0827,
            "longitude": 80.2707,
            "estimated_arrival": "09:00",
            "distance_from_previous_km": 0,
            "duration_from_previous_min": 0,
            "instructions": "Start your delivery route from here"
        }
    ],
    "total_distance_km": 0,
    "total_duration_min": 0,
    "estimated_fuel_liters": 0,
    "optimization_notes": "Brief explanation of why this route order is optimal",
    "route_polyline": "encoded polyline string if available",
    "savings": {
        "distance_saved_percent": 0,
        "time_saved_percent": 0
    }
}
PROMPT;
    }

    /**
     * Parse and validate AI response
     */
    private function parseAIResponse(string $content, array $originalLocations): array
    {
        // Clean up the response
        $content = preg_replace('/^```json\s*/', '', $content);
        $content = preg_replace('/\s*```$/', '', $content);
        $content = trim($content);

        $data = json_decode($content, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            Log::warning('Failed to parse AI response', ['content' => $content]);
            return $this->createFallbackRoute($originalLocations);
        }

        if (!isset($data['optimized_order']) || !is_array($data['optimized_order'])) {
            return $this->createFallbackRoute($originalLocations);
        }

        return $data;
    }

    /**
     * Create a fallback route with original order
     */
    private function createFallbackRoute(array $locations): array
    {
        $optimizedOrder = [];
        $sequence = 1;

        foreach ($locations as $location) {
            $optimizedOrder[] = [
                'sequence' => $sequence++,
                'location_id' => $location['id'],
                'location_name' => $location['name'],
                'address' => $location['address'],
                'latitude' => $location['latitude'] ?? 0,
                'longitude' => $location['longitude'] ?? 0,
                'estimated_arrival' => '--:--',
                'distance_from_previous_km' => 0,
                'duration_from_previous_min' => 0,
                'instructions' => $location['type'] === 'origin'
                    ? __('Start your delivery route')
                    : __('Proceed to this location'),
            ];
        }

        return [
            'optimized_order' => $optimizedOrder,
            'total_distance_km' => 0,
            'total_duration_min' => 0,
            'estimated_fuel_liters' => 0,
            'optimization_notes' => __('Using original order (AI optimization unavailable)'),
            'savings' => [
                'distance_saved_percent' => 0,
                'time_saved_percent' => 0,
            ],
        ];
    }

    /**
     * Apply optimized route to trip - update sequences
     */
    public function applyRoute(Request $request)
    {
        $validated = $request->validate([
            'trip_id' => 'required|exists:trips,id',
            'optimized_order' => 'required|array',
            'optimized_order.*.location_id' => 'required',
            'optimized_order.*.sequence' => 'required|integer',
        ]);

        $trip = Trip::findOrFail($validated['trip_id']);

        // Update location sequences based on optimized order
        foreach ($validated['optimized_order'] as $item) {
            // Handle trip_locations
            if (str_starts_with($item['location_id'], 'location_')) {
                $locationId = str_replace('location_', '', $item['location_id']);
                if (is_numeric($locationId)) {
                    TripLocation::where('id', $locationId)
                        ->where('trip_id', $trip->id)
                        ->update(['sequence' => $item['sequence']]);
                }
            }
            // Handle legacy deliveries
            elseif (str_starts_with($item['location_id'], 'delivery_')) {
                $deliveryId = str_replace('delivery_', '', $item['location_id']);
                if (is_numeric($deliveryId)) {
                    Delivery::where('id', $deliveryId)
                        ->where('trip_id', $trip->id)
                        ->update(['sequence' => $item['sequence']]);
                }
            }
        }

        return redirect()->route('admin.trips.show', $trip->id)
            ->with('success', __('Optimized route applied successfully.'));
    }

    /**
     * API endpoint to apply optimized route
     */
    public function apiApplyRoute(Request $request)
    {
        $validated = $request->validate([
            'trip_id' => 'required|exists:trips,id',
            'optimized_order' => 'required|array',
            'optimized_order.*.location_id' => 'required',
            'optimized_order.*.sequence' => 'required|integer',
        ]);

        $trip = Trip::findOrFail($validated['trip_id']);

        foreach ($validated['optimized_order'] as $item) {
            if (str_starts_with($item['location_id'], 'location_')) {
                $locationId = str_replace('location_', '', $item['location_id']);
                if (is_numeric($locationId)) {
                    TripLocation::where('id', $locationId)
                        ->where('trip_id', $trip->id)
                        ->update(['sequence' => $item['sequence']]);
                }
            } elseif (str_starts_with($item['location_id'], 'delivery_')) {
                $deliveryId = str_replace('delivery_', '', $item['location_id']);
                if (is_numeric($deliveryId)) {
                    Delivery::where('id', $deliveryId)
                        ->where('trip_id', $trip->id)
                        ->update(['sequence' => $item['sequence']]);
                }
            }
        }

        return response()->json([
            'success' => true,
            'message' => __('Optimized route applied successfully.'),
        ]);
    }

    /**
     * Get route preview with map data
     */
    public function getRoutePreview(Request $request)
    {
        $tripId = $request->input('trip_id');
        $trip = Trip::with(['locations'])->findOrFail($tripId);

        $coordinates = $trip->getAllCoordinatesForRoute();

        return response()->json([
            'success' => true,
            'coordinates' => $coordinates,
            'pickup' => [
                'lat' => $trip->pickup_lat,
                'lng' => $trip->pickup_lng,
                'address' => $trip->pickup_location,
            ],
            'drops' => $trip->has_multiple_locations
                ? $trip->locations->map(fn($l) => [
                    'id' => $l->id,
                    'lat' => $l->lat,
                    'lng' => $l->lng,
                    'address' => $l->address,
                    'name' => $l->location_name,
                    'sequence' => $l->sequence,
                ])
                : [[
                    'lat' => $trip->drop_lat,
                    'lng' => $trip->drop_lng,
                    'address' => $trip->drop_location,
                ]],
        ]);
    }
}
