<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Kreait\Firebase\Factory;
use App\Models\Driver;

class TrackingController extends Controller
{
    private $database;

    public function __construct()
    {
        $factory = (new Factory)
            ->withServiceAccount(config('firebase.credentials'))
            ->withDatabaseUri(config('firebase.database_url'));

        $this->database = $factory->createDatabase();
    }

    /**
     * Display live tracking map
     */
    public function index()
    {
        // Fixed: Use 'ownVehicle' instead of 'vehicle'
        $drivers = Driver::with(['ownVehicle'])->get();
        return view('admin.tracking.index', compact('drivers'));
    }

    /**
     * Get all driver locations from Firebase
     */
    public function getDriverLocations()
    {
        try {
            $locations = $this->database
                ->getReference('driver_locations')
                ->getValue();

            if (empty($locations)) {
                return response()->json([]);
            }

            // Format data for frontend
            $formattedLocations = [];
            foreach ($locations as $key => $location) {
                if (isset($location['is_online']) && $location['is_online']) {
                    $formattedLocations[] = [
                        'driver_id' => $location['driver_id'],
                        'driver_name' => $location['driver_name'],
                        'vehicle_id' => $location['vehicle_id'] ?? null,
                        'vehicle_number' => $location['vehicle_number'] ?? 'N/A',
                        'trip_id' => $location['trip_id'] ?? null,
                        'latitude' => $location['latitude'],
                        'longitude' => $location['longitude'],
                        'accuracy' => $location['accuracy'],
                        'speed' => $location['speed'] ?? 0,
                        'heading' => $location['heading'] ?? 0,
                        'status' => $location['status'],
                        'battery_level' => $location['battery_level'] ?? 100,
                        'last_updated' => $location['last_updated'],
                    ];
                }
            }

            return response()->json($formattedLocations);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get single driver location
     */
    public function getDriverLocation($driverId)
    {
        try {
            $location = $this->database
                ->getReference('driver_locations/driver_' . $driverId)
                ->getValue();

            if (!$location) {
                return response()->json(['error' => 'Driver not found or offline'], 404);
            }

            return response()->json($location);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

}
