<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class FuelLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'fuel_number',
        'vehicle_id',
        'driver_id',
        'fuel_type',
        'quantity',
        'price_per_unit',
        'total_cost',
        'odometer_reading',
        'previous_odometer',
        'distance_traveled',
        'fuel_efficiency',
        'fill_date',
        'fill_time',
        'is_full_tank',
        'fuel_station',
        'station_location',
        'receipt_number',
        'payment_method',
        'trip_id',
        'notes',
        'created_by',
    ];

    protected $casts = [
        'quantity' => 'decimal:2',
        'price_per_unit' => 'decimal:2',
        'total_cost' => 'decimal:2',
        'odometer_reading' => 'decimal:2',
        'previous_odometer' => 'decimal:2',
        'distance_traveled' => 'decimal:2',
        'fuel_efficiency' => 'decimal:2',
        'fill_date' => 'date',
        'fill_time' => 'datetime',
        'is_full_tank' => 'boolean',
    ];

    // ===== Relationships =====

    public function vehicle()
    {
        return $this->belongsTo(Vehicle::class);
    }

    public function driver()
    {
        return $this->belongsTo(Driver::class);
    }

    public function trip()
    {
        return $this->belongsTo(Trip::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // ===== Scopes =====

    public function scopeByVehicle($query, $vehicleId)
    {
        return $query->where('vehicle_id', $vehicleId);
    }

    public function scopeByDriver($query, $driverId)
    {
        return $query->where('driver_id', $driverId);
    }

    public function scopeByFuelType($query, $fuelType)
    {
        return $query->where('fuel_type', $fuelType);
    }

    public function scopeThisMonth($query)
    {
        return $query->whereMonth('fill_date', now()->month)
            ->whereYear('fill_date', now()->year);
    }

    public function scopeThisYear($query)
    {
        return $query->whereYear('fill_date', now()->year);
    }

    public function scopeDateRange($query, $from, $to)
    {
        return $query->whereBetween('fill_date', [$from, $to]);
    }

    // ===== Accessors =====

    /**
     * Get formatted fuel efficiency
     */
    public function getFormattedEfficiencyAttribute()
    {
        if ($this->fuel_efficiency) {
            return number_format($this->fuel_efficiency, 2) . ' km/L';
        }
        return '-';
    }

    /**
     * Get fuel type display name
     */
    public function getFuelTypeDisplayAttribute()
    {
        return match ($this->fuel_type) {
            'diesel' => __('Diesel'),
            'petrol' => __('Petrol'),
            'cng' => __('CNG'),
            'electric' => __('Electric'),
            default => ucfirst($this->fuel_type)
        };
    }

    /**
     * Get payment method display name
     */
    public function getPaymentMethodDisplayAttribute()
    {
        return match ($this->payment_method) {
            'cash' => __('Cash'),
            'card' => __('Card'),
            'company_account' => __('Company Account'),
            'fuel_card' => __('Fuel Card'),
            default => ucfirst($this->payment_method)
        };
    }

    // ===== Helper Methods =====

    /**
     * Generate unique fuel number
     */
    public static function generateFuelNumber()
    {
        $year = date('Y');
        $lastLog = self::whereYear('created_at', $year)
            ->orderBy('id', 'desc')
            ->first();

        $nextNumber = 1;
        if ($lastLog) {
            preg_match('/FUEL-\d{4}-(\d+)/', $lastLog->fuel_number, $matches);
            $nextNumber = isset($matches[1]) ? intval($matches[1]) + 1 : 1;
        }

        return sprintf('FUEL-%s-%04d', $year, $nextNumber);
    }

    /**
     * Calculate fuel efficiency based on previous fill
     */
    public function calculateEfficiency()
    {
        if (!$this->is_full_tank || !$this->odometer_reading || !$this->quantity) {
            return null;
        }

        // Find previous full tank fill for this vehicle
        $previousFill = self::where('vehicle_id', $this->vehicle_id)
            ->where('id', '<', $this->id)
            ->where('is_full_tank', true)
            ->whereNotNull('odometer_reading')
            ->orderBy('fill_date', 'desc')
            ->orderBy('id', 'desc')
            ->first();

        if ($previousFill && $previousFill->odometer_reading) {
            $distance = $this->odometer_reading - $previousFill->odometer_reading;
            if ($distance > 0 && $this->quantity > 0) {
                return round($distance / $this->quantity, 2);
            }
        }

        return null;
    }

    /**
     * Get previous odometer reading for this vehicle
     */
    public static function getPreviousOdometer($vehicleId, $excludeId = null)
    {
        $query = self::where('vehicle_id', $vehicleId)
            ->whereNotNull('odometer_reading')
            ->orderBy('fill_date', 'desc')
            ->orderBy('id', 'desc');

        if ($excludeId) {
            $query->where('id', '!=', $excludeId);
        }

        return $query->value('odometer_reading');
    }

    /**
     * Get total fuel cost for a vehicle
     */
    public static function getTotalCostByVehicle($vehicleId)
    {
        return self::where('vehicle_id', $vehicleId)->sum('total_cost');
    }

    /**
     * Get total fuel quantity for a vehicle
     */
    public static function getTotalQuantityByVehicle($vehicleId)
    {
        return self::where('vehicle_id', $vehicleId)->sum('quantity');
    }

    /**
     * Get average efficiency for a vehicle
     */
    public static function getAverageEfficiencyByVehicle($vehicleId)
    {
        return self::where('vehicle_id', $vehicleId)
            ->whereNotNull('fuel_efficiency')
            ->where('fuel_efficiency', '>', 0)
            ->avg('fuel_efficiency');
    }

    /**
     * Get fuel type badge class
     */
    public function getFuelTypeBadgeClass()
    {
        return match ($this->fuel_type) {
            'diesel' => 'warning',
            'petrol' => 'info',
            'cng' => 'success',
            'electric' => 'primary',
            default => 'secondary'
        };
    }

    /**
     * Get payment method badge class
     */
    public function getPaymentBadgeClass()
    {
        return match ($this->payment_method) {
            'cash' => 'success',
            'card' => 'info',
            'company_account' => 'primary',
            'fuel_card' => 'warning',
            default => 'secondary'
        };
    }
}
