<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Report extends Model
{
    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'report_number',
        'title',
        'report_type',
        'description',
        'parameters',
        'date_from',
        'date_to',
        'vehicle_id',
        'driver_id',
        'vehicle_type_id',
        'format',
        'file_path',
        'file_size',
        'status',
        'generated_at',
        'error_message',
        'is_scheduled',
        'schedule_frequency',
        'next_run_at',
        'last_run_at',
        'summary_data',
        'created_by',
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'parameters' => 'array',
        'summary_data' => 'array',
        'date_from' => 'date',
        'date_to' => 'date',
        'generated_at' => 'datetime',
        'next_run_at' => 'datetime',
        'last_run_at' => 'datetime',
        'is_scheduled' => 'boolean',
    ];

    /**
     * Report type options
     */
    public const TYPES = [
        'fuel_consumption' => 'Fuel Consumption',
        'trip_summary' => 'Trip Summary',
        'vehicle_utilization' => 'Vehicle Utilization',
        'driver_performance' => 'Driver Performance',
        'cost_analysis' => 'Cost Analysis',
        'delivery_performance' => 'Delivery Performance',
        'custom' => 'Custom Report',
    ];

    /**
     * Format options
     */
    public const FORMATS = [
        'pdf' => 'PDF Document',
        'excel' => 'Excel Spreadsheet',
        'csv' => 'CSV File',
        'html' => 'HTML Report',
    ];

    /**
     * Status options
     */
    public const STATUSES = [
        'draft' => 'Draft',
        'generating' => 'Generating',
        'completed' => 'Completed',
        'failed' => 'Failed',
        'scheduled' => 'Scheduled',
    ];

    /**
     * Schedule frequency options
     */
    public const FREQUENCIES = [
        'daily' => 'Daily',
        'weekly' => 'Weekly',
        'monthly' => 'Monthly',
        'quarterly' => 'Quarterly',
    ];

    /**
     * Get the user who created the report.
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the vehicle for this report.
     */
    public function vehicle()
    {
        return $this->belongsTo(Vehicle::class);
    }

    /**
     * Get the driver for this report.
     */
    public function driver()
    {
        return $this->belongsTo(Driver::class);
    }

    /**
     * Get the vehicle type for this report.
     */
    public function vehicleType()
    {
        return $this->belongsTo(VehicleType::class);
    }

    /**
     * Scope for filtering by type
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('report_type', $type);
    }

    /**
     * Scope for filtering by status
     */
    public function scopeOfStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope for completed reports
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope for scheduled reports
     */
    public function scopeScheduled($query)
    {
        return $query->where('is_scheduled', true);
    }

    /**
     * Check if report is downloadable
     */
    public function isDownloadable(): bool
    {
        return $this->status === 'completed' && !empty($this->file_path);
    }

    /**
     * Get formatted file size
     */
    public function getFormattedFileSizeAttribute(): string
    {
        if (!$this->file_size) {
            return '-';
        }

        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = $this->file_size;
        $i = 0;

        while ($bytes >= 1024 && $i < count($units) - 1) {
            $bytes /= 1024;
            $i++;
        }

        return round($bytes, 2) . ' ' . $units[$i];
    }

    /**
     * Get the date range display
     */
    public function getDateRangeAttribute(): string
    {
        if ($this->date_from && $this->date_to) {
            return $this->date_from->format('M d, Y') . ' - ' . $this->date_to->format('M d, Y');
        }

        if ($this->date_from) {
            return 'From ' . $this->date_from->format('M d, Y');
        }

        if ($this->date_to) {
            return 'Until ' . $this->date_to->format('M d, Y');
        }

        return 'All Time';
    }

    /**
     * Generate next report number
     */
    public static function generateReportNumber(): string
    {
        $prefix = 'RPT';
        $year = date('Y');
        $month = date('m');

        $lastReport = self::where('report_number', 'like', "{$prefix}-{$year}{$month}-%")
            ->orderBy('report_number', 'desc')
            ->first();

        if ($lastReport) {
            $lastNumber = (int) substr($lastReport->report_number, -4);
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }

        return sprintf('%s-%s%s-%04d', $prefix, $year, $month, $newNumber);
    }
}
