<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class ReportTemplate extends Model
{
    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'name',
        'report_type',
        'description',
        'default_parameters',
        'is_system',
        'is_active',
        'created_by',
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'default_parameters' => 'array',
        'is_system' => 'boolean',
        'is_active' => 'boolean',
    ];

    /**
     * Get the user who created the template.
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Scope for active templates
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for system templates
     */
    public function scopeSystem($query)
    {
        return $query->where('is_system', true);
    }

    /**
     * Get template icon based on report type
     */
    public function getIconAttribute(): string
    {
        $icons = [
            'fuel_consumption' => 'fuel-pump',
            'trip_summary' => 'geo-alt',
            'vehicle_utilization' => 'truck',
            'driver_performance' => 'person-badge',
            'cost_analysis' => 'currency-dollar',
            'delivery_performance' => 'box-seam',
            'custom' => 'file-earmark',
        ];

        return $icons[$this->report_type] ?? 'file-earmark';
    }

    /**
     * Get template color based on report type
     */
    public function getColorAttribute(): string
    {
        $colors = [
            'fuel_consumption' => 'warning',
            'trip_summary' => 'primary',
            'vehicle_utilization' => 'info',
            'driver_performance' => 'success',
            'cost_analysis' => 'danger',
            'delivery_performance' => 'secondary',
            'custom' => 'dark',
        ];

        return $colors[$this->report_type] ?? 'secondary';
    }
}
