<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Trip extends Model
{
    use HasFactory;

    protected $fillable = [
        'trip_number',
        'vehicle_id',
        'driver_id',
        'shipment_id',
        'shipment_reference',
        'has_multiple_locations',
        'pickup_location',
        'pickup_lat',
        'pickup_lng',
        'drop_location',
        'drop_lat',
        'drop_lng',
        'trip_instructions',
        'status',
        'start_date',
        'end_date',
        'created_by',
    ];

    protected $casts = [
        'has_multiple_locations' => 'boolean',
        'pickup_lat' => 'decimal:8',
        'pickup_lng' => 'decimal:8',
        'drop_lat' => 'decimal:8',
        'drop_lng' => 'decimal:8',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the vehicle assigned to this trip
     */
    public function vehicle()
    {
        return $this->belongsTo(Vehicle::class);
    }

    /**
     * Get the driver assigned to this trip
     */
    public function driver()
    {
        return $this->belongsTo(Driver::class);
    }

    /**
     * Get the shipment associated with this trip
     */
    public function shipment()
    {
        return $this->belongsTo(Shipment::class);
    }

    /**
     * Get the user who created this trip
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the files associated with this trip
     */
    public function files()
    {
        return $this->hasMany(TripFile::class);
    }

    /**
     * Get the cost details for this trip (single cost)
     */
    public function cost()
    {
        return $this->hasOne(TripCost::class);
    }

    /**
     * Get all costs for this trip (multiple costs)
     */
    public function tripCosts()
    {
        return $this->hasMany(TripCost::class);
    }

    /**
     * Get the delivery details for this trip (single delivery)
     */
    public function delivery()
    {
        return $this->hasOne(Delivery::class);
    }

    /**
     * Get all deliveries for this trip (multiple deliveries)
     */
    public function deliveries()
    {
        return $this->hasMany(Delivery::class);
    }

    /**
     * Get all drop locations for this trip (multiple locations)
     * Ordered by sequence
     */
    public function locations()
    {
        return $this->hasMany(TripLocation::class)->orderBy('sequence');
    }

    /**
     * Get pending locations
     */
    public function pendingLocations()
    {
        return $this->hasMany(TripLocation::class)
            ->where('status', 'pending')
            ->orderBy('sequence');
    }

    /**
     * Get completed locations
     */
    public function completedLocations()
    {
        return $this->hasMany(TripLocation::class)
            ->where('status', 'completed')
            ->orderBy('sequence');
    }

    /**
     * Check if pickup has coordinates
     */
    public function hasPickupCoordinates()
    {
        return !is_null($this->pickup_lat) && !is_null($this->pickup_lng);
    }

    /**
     * Check if drop has coordinates (single location)
     */
    public function hasDropCoordinates()
    {
        return !is_null($this->drop_lat) && !is_null($this->drop_lng);
    }

    /**
     * Get pickup coordinates as array
     */
    public function getPickupCoordinatesAttribute()
    {
        if ($this->hasPickupCoordinates()) {
            return [
                'lat' => (float) $this->pickup_lat,
                'lng' => (float) $this->pickup_lng,
            ];
        }
        return null;
    }

    /**
     * Get drop coordinates as array (single location)
     */
    public function getDropCoordinatesAttribute()
    {
        if ($this->hasDropCoordinates()) {
            return [
                'lat' => (float) $this->drop_lat,
                'lng' => (float) $this->drop_lng,
            ];
        }
        return null;
    }

    /**
     * Get all location coordinates for route optimization
     * Returns pickup + all drop locations in sequence order
     */
    public function getAllCoordinatesForRoute()
    {
        $coordinates = [];

        // Add pickup location
        if ($this->hasPickupCoordinates()) {
            $coordinates[] = [
                'id' => 'pickup',
                'type' => 'pickup',
                'name' => __('Pickup Location'),
                'address' => $this->pickup_location,
                'lat' => (float) $this->pickup_lat,
                'lng' => (float) $this->pickup_lng,
                'sequence' => 0,
            ];
        }

        // For multiple locations - use trip_locations table
        if ($this->has_multiple_locations) {
            foreach ($this->locations as $location) {
                if ($location->hasCoordinates()) {
                    $coordinates[] = [
                        'id' => 'location_' . $location->id,
                        'type' => 'drop',
                        'name' => $location->location_name ?? __('Drop') . ' #' . $location->sequence,
                        'address' => $location->address,
                        'lat' => (float) $location->lat,
                        'lng' => (float) $location->lng,
                        'sequence' => $location->sequence,
                        'contact_name' => $location->contact_name,
                        'contact_phone' => $location->contact_phone,
                        'notes' => $location->notes,
                        'status' => $location->status,
                    ];
                }
            }
        } else {
            // Single drop location
            if ($this->hasDropCoordinates()) {
                $coordinates[] = [
                    'id' => 'drop',
                    'type' => 'drop',
                    'name' => __('Drop Location'),
                    'address' => $this->drop_location,
                    'lat' => (float) $this->drop_lat,
                    'lng' => (float) $this->drop_lng,
                    'sequence' => 1,
                ];
            }
        }

        return $coordinates;
    }

    /**
     * Get total locations count
     */
    public function getTotalLocationsCountAttribute()
    {
        if ($this->has_multiple_locations) {
            return $this->locations()->count();
        }
        return $this->drop_location ? 1 : 0;
    }

    /**
     * Scope to filter trips by status
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope to get active trips (assigned or in_transit)
     */
    public function scopeActive($query)
    {
        return $query->whereIn('status', ['assigned', 'in_transit']);
    }

    /**
     * Scope to get completed trips (delivered)
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'delivered');
    }

    /**
     * Check if trip is active
     */
    public function isActive()
    {
        return in_array($this->status, ['assigned', 'in_transit']);
    }

    /**
     * Check if trip is completed
     */
    public function isCompleted()
    {
        return $this->status === 'delivered';
    }

    /**
     * Check if trip is cancelled
     */
    public function isCancelled()
    {
        return $this->status === 'cancelled';
    }

    /**
     * Check if trip has drop location
     */
    public function hasDropLocation()
    {
        return $this->has_multiple_locations && !empty($this->drop_location);
    }

    /**
     * Get status badge color
     */
    public function getStatusColorAttribute()
    {
        return match ($this->status) {
            'pending' => 'secondary',
            'assigned' => 'primary',
            'in_transit' => 'info',
            'delivered' => 'success',
            'cancelled' => 'danger',
            default => 'secondary'
        };
    }

    /**
     * Get formatted trip duration
     */
    public function getDurationAttribute()
    {
        if (!$this->start_date || !$this->end_date) {
            return null;
        }

        $diff = $this->start_date->diff($this->end_date);

        if ($diff->days > 0) {
            return $diff->days . ' day(s)';
        }

        return $diff->format('%H hours %I minutes');
    }

    /**
     * Generate unique trip number
     */
    public static function generateTripNumber()
    {
        $lastTrip = self::latest('id')->first();
        return 'TRIP-' . str_pad(($lastTrip ? $lastTrip->id + 1 : 1), 6, '0', STR_PAD_LEFT);
    }
}
