<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class TripLocation extends Model
{
    use HasFactory;

    protected $fillable = [
        'trip_id',
        'sequence',
        'location_name',
        'address',
        'lat',
        'lng',
        'contact_name',
        'contact_phone',
        'notes',
        'status',
        'completed_at',
    ];

    protected $casts = [
        'lat' => 'decimal:8',
        'lng' => 'decimal:8',
        'sequence' => 'integer',
        'completed_at' => 'datetime',
    ];

    /**
     * Get the trip that owns this location
     */
    public function trip()
    {
        return $this->belongsTo(Trip::class);
    }

    /**
     * Check if location has coordinates
     */
    public function hasCoordinates()
    {
        return !is_null($this->lat) && !is_null($this->lng);
    }

    /**
     * Get coordinates as array
     */
    public function getCoordinatesAttribute()
    {
        if ($this->hasCoordinates()) {
            return [
                'lat' => (float) $this->lat,
                'lng' => (float) $this->lng,
            ];
        }
        return null;
    }

    /**
     * Get status badge color
     */
    public function getStatusColorAttribute()
    {
        return match ($this->status) {
            'pending' => 'secondary',
            'in_progress' => 'info',
            'completed' => 'success',
            'skipped' => 'warning',
            default => 'secondary'
        };
    }

    /**
     * Scope for pending locations
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope for completed locations
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope ordered by sequence
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sequence', 'asc');
    }

    /**
     * Mark as completed
     */
    public function markCompleted()
    {
        $this->update([
            'status' => 'completed',
            'completed_at' => now(),
        ]);
    }

    /**
     * Mark as skipped
     */
    public function markSkipped()
    {
        $this->update([
            'status' => 'skipped',
        ]);
    }
}
