<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class VehicleMaintenance extends Model
{
    use HasFactory;

    protected $fillable = [
        'vehicle_id',
        'maintenance_type_id',
        'maintenance_number',
        'status',
        'priority',
        'scheduled_date',
        'completed_date',
        'odometer_reading',
        'next_due_km',
        'next_due_date',
        'cost',
        'labor_cost',
        'parts_cost',
        'vendor_name',
        'vendor_contact',
        'invoice_number',
        'description',
        'notes',
        'performed_by',
        'created_by',
    ];

    protected $casts = [
        'scheduled_date' => 'date',
        'completed_date' => 'date',
        'next_due_date' => 'date',
        'cost' => 'decimal:2',
        'labor_cost' => 'decimal:2',
        'parts_cost' => 'decimal:2',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($maintenance) {
            if (empty($maintenance->maintenance_number)) {
                $maintenance->maintenance_number = self::generateMaintenanceNumber();
            }
        });
    }

    public static function generateMaintenanceNumber()
    {
        $date = now()->format('Ymd');
        $lastMaintenance = self::whereDate('created_at', today())
            ->orderBy('id', 'desc')
            ->first();

        $newNumber = $lastMaintenance
            ? (int) substr($lastMaintenance->maintenance_number, -4) + 1
            : 1;

        return 'MNT-' . $date . '-' . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
    }

    public function vehicle()
    {
        return $this->belongsTo(Vehicle::class);
    }

    public function maintenanceType()
    {
        return $this->belongsTo(MaintenanceType::class);
    }

    public function parts()
    {
        return $this->hasMany(MaintenancePart::class);
    }

    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function getTotalCostAttribute()
    {
        return $this->labor_cost + $this->parts_cost;
    }

    public function getIsOverdueAttribute()
    {
        return $this->status === 'scheduled' && $this->scheduled_date < today();
    }

    public function scopeScheduled($query)
    {
        return $query->where('status', 'scheduled');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeOverdue($query)
    {
        return $query->where('status', 'scheduled')
            ->where('scheduled_date', '<', today());
    }

    public function scopeDueSoon($query, $days = 7)
    {
        return $query->where('status', 'scheduled')
            ->whereBetween('scheduled_date', [today(), today()->addDays($days)]);
    }
}
