@extends('layouts.app')

@section('title', __('Fuel Logs') . ' - ' . __('app_name'))

@push('styles')
<link rel="stylesheet" href="{{ asset('css/adminlistpage.css') }}">
@endpush

@section('content')
<div class="container-fluid">
    {{-- Page Header --}}
    <div class="list-page-header">
        <div class="list-page-header-left">
            <h1 class="list-page-title">
                <i class="bi bi-fuel-pump"></i>
                {{ __('Fuel Logs') }}
            </h1>
            <p class="list-page-subtitle">{{ __('Track fuel consumption and costs for your fleet') }}</p>
        </div>
        <div class="list-page-header-actions">
            @if(auth()->user()->hasPermission('fuel.create'))
            <a href="{{ route('admin.fuel-logs.create') }}" class="btn-create">
                <i class="bi bi-plus-lg"></i>
                {{ __('Add Fuel Log') }}
            </a>
            @endif
        </div>
    </div>

    {{-- Flash Messages --}}
    @if(session('success'))
        <div class="alert alert-success alert-dismissible fade show mb-3" role="alert">
            <i class="bi bi-check-circle me-2"></i>{{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    @if(session('error'))
        <div class="alert alert-danger alert-dismissible fade show mb-3" role="alert">
            <i class="bi bi-exclamation-triangle me-2"></i>{{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    {{-- Stats Bar --}}
    <div class="list-stats-bar">
        <div class="list-stat-card gradient-violet">
            <div class="list-stat-icon">
                <i class="bi bi-fuel-pump"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $fuelLogs->total() }}</div>
                <div class="list-stat-label">{{ __('Total Records') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-graph-up-arrow"></i>
                {{ __('All time') }}
            </div>
        </div>
        <div class="list-stat-card gradient-rose">
            <div class="list-stat-icon">
                <i class="bi bi-droplet"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ number_format($stats['total_quantity'] ?? 0, 0) }}</div>
                <div class="list-stat-label">{{ __('Total Liters') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-water"></i>
                {{ __('Fuel consumed') }}
            </div>
        </div>
        <div class="list-stat-card gradient-cyan">
            <div class="list-stat-icon">
                <i class="bi bi-currency-dollar"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">${{ number_format($stats['total_cost'] ?? 0, 0) }}</div>
                <div class="list-stat-label">{{ __('Total Cost') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-cash-stack"></i>
                {{ __('All expenses') }}
            </div>
        </div>
        <div class="list-stat-card gradient-amber">
            <div class="list-stat-icon">
                <i class="bi bi-speedometer2"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $stats['avg_efficiency'] ? number_format($stats['avg_efficiency'], 1) : '-' }}</div>
                <div class="list-stat-label">{{ __('Avg km/L') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-lightning"></i>
                {{ __('Efficiency') }}
            </div>
        </div>
    </div>

    {{-- Filter Bar --}}
    <div class="list-filter-bar">
        <div class="list-filter-header">
            <div class="list-filter-title">
                <i class="bi bi-funnel"></i>
                {{ __('Filters') }}
            </div>
            @if(request()->hasAny(['search', 'vehicle_id', 'driver_id', 'fuel_type', 'payment_method', 'date_from', 'date_to']))
                <a href="{{ route('admin.fuel-logs.index') }}" class="list-filter-clear">
                    <i class="bi bi-x-circle"></i>
                    {{ __('Clear Filters') }}
                </a>
            @endif
        </div>
        <form action="{{ route('admin.fuel-logs.index') }}" method="GET">
            <div class="list-filter-row">
                <div class="list-filter-group search-group">
                    <label>{{ __('Search') }}</label>
                    <div class="list-search-box">
                        <i class="bi bi-search"></i>
                        <input type="text" name="search" class="form-control" 
                               placeholder="{{ __('Search by fuel number...') }}"
                               value="{{ request('search') }}">
                    </div>
                </div>
                <div class="list-filter-group">
                    <label>{{ __('Vehicle') }}</label>
                    <select name="vehicle_id" class="form-select">
                        <option value="">{{ __('All Vehicles') }}</option>
                        @foreach($vehicles as $vehicle)
                            <option value="{{ $vehicle->id }}" {{ request('vehicle_id') == $vehicle->id ? 'selected' : '' }}>
                                {{ $vehicle->vehicle_number }}
                            </option>
                        @endforeach
                    </select>
                </div>
                <div class="list-filter-group">
                    <label>{{ __('Fuel Type') }}</label>
                    <select name="fuel_type" class="form-select">
                        <option value="">{{ __('All Types') }}</option>
                        <option value="diesel" {{ request('fuel_type') === 'diesel' ? 'selected' : '' }}>{{ __('Diesel') }}</option>
                        <option value="petrol" {{ request('fuel_type') === 'petrol' ? 'selected' : '' }}>{{ __('Petrol') }}</option>
                        <option value="cng" {{ request('fuel_type') === 'cng' ? 'selected' : '' }}>{{ __('CNG') }}</option>
                        <option value="electric" {{ request('fuel_type') === 'electric' ? 'selected' : '' }}>{{ __('Electric') }}</option>
                    </select>
                </div>
                <div class="list-filter-group">
                    <label>{{ __('Payment') }}</label>
                    <select name="payment_method" class="form-select">
                        <option value="">{{ __('All Methods') }}</option>
                        <option value="cash" {{ request('payment_method') === 'cash' ? 'selected' : '' }}>{{ __('Cash') }}</option>
                        <option value="card" {{ request('payment_method') === 'card' ? 'selected' : '' }}>{{ __('Card') }}</option>
                        <option value="company_account" {{ request('payment_method') === 'company_account' ? 'selected' : '' }}>{{ __('Company Account') }}</option>
                        <option value="fuel_card" {{ request('payment_method') === 'fuel_card' ? 'selected' : '' }}>{{ __('Fuel Card') }}</option>
                    </select>
                </div>
                <div class="list-filter-group">
                    <label>{{ __('Date From') }}</label>
                    <input type="date" name="date_from" class="form-control" value="{{ request('date_from') }}">
                </div>
                <div class="list-filter-group">
                    <label>{{ __('Date To') }}</label>
                    <input type="date" name="date_to" class="form-control" value="{{ request('date_to') }}">
                </div>
                <div class="list-filter-buttons">
                    <button type="submit" class="btn-filter btn-filter-primary">
                        <i class="bi bi-search"></i>
                        {{ __('Apply') }}
                    </button>
                </div>
            </div>
        </form>
    </div>

    {{-- Data Table --}}
    <div class="list-card">
        @if($fuelLogs->count() > 0)
            <div class="list-table-responsive">
                <table class="list-table">
                    <thead>
                        <tr>
                            <th>{{ __('Fuel #') }}</th>
                            <th>{{ __('Vehicle') }}</th>
                            <th>{{ __('Fill Date') }}</th>
                            <th>{{ __('Fuel Type') }}</th>
                            <th>{{ __('Quantity') }}</th>
                            <th>{{ __('Total Cost') }}</th>
                            <th>{{ __('Efficiency') }}</th>
                            <th>{{ __('Payment') }}</th>
                            <th class="text-center">{{ __('Actions') }}</th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach($fuelLogs as $fuelLog)
                            @php
                                $fuelTypeClasses = [
                                    'diesel' => 'warning',
                                    'petrol' => 'info',
                                    'cng' => 'success',
                                    'electric' => 'primary'
                                ];
                                $paymentClasses = [
                                    'cash' => 'success',
                                    'card' => 'info',
                                    'company_account' => 'primary',
                                    'fuel_card' => 'warning'
                                ];
                            @endphp
                            <tr>
                                <td>
                                    <div class="table-primary-text">
                                        <a href="{{ route('admin.fuel-logs.show', $fuelLog) }}">
                                            {{ $fuelLog->fuel_number }}
                                        </a>
                                    </div>
                                    @if($fuelLog->is_full_tank)
                                        <span class="badge bg-success badge-sm">{{ __('Full Tank') }}</span>
                                    @endif
                                </td>
                                <td>
                                    <div class="table-primary-text">{{ $fuelLog->vehicle->vehicle_number }}</div>
                                    <div class="table-secondary-text">{{ $fuelLog->vehicle->vehicleType->name ?? '-' }}</div>
                                </td>
                                <td>
                                    <div class="table-primary-text">{{ $fuelLog->fill_date->format('M d, Y') }}</div>
                                    @if($fuelLog->fill_time)
                                        <div class="table-secondary-text">{{ \Carbon\Carbon::parse($fuelLog->fill_time)->format('h:i A') }}</div>
                                    @endif
                                </td>
                                <td>
                                    <span class="badge bg-{{ $fuelTypeClasses[$fuelLog->fuel_type] ?? 'secondary' }}">
                                        {{ __(ucfirst($fuelLog->fuel_type)) }}
                                    </span>
                                </td>
                                <td>
                                    <div class="table-primary-text">{{ number_format($fuelLog->quantity, 2) }} L</div>
                                    <div class="table-secondary-text">${{ number_format($fuelLog->price_per_unit, 2) }}/L</div>
                                </td>
                                <td>
                                    <div class="table-primary-text text-success fw-semibold">${{ number_format($fuelLog->total_cost, 2) }}</div>
                                </td>
                                <td>
                                    @if($fuelLog->fuel_efficiency)
                                        <span class="badge bg-success">{{ number_format($fuelLog->fuel_efficiency, 2) }} km/L</span>
                                    @else
                                        <span class="text-muted">-</span>
                                    @endif
                                </td>
                                <td>
                                    <span class="badge bg-{{ $paymentClasses[$fuelLog->payment_method] ?? 'secondary' }}">
                                        {{ __(ucfirst(str_replace('_', ' ', $fuelLog->payment_method))) }}
                                    </span>
                                </td>
                                <td>
                                    <div class="table-actions">
                                        @if(auth()->user()->hasPermission('fuel.view'))
                                            <a href="{{ route('admin.fuel-logs.show', $fuelLog) }}" 
                                               class="btn-table-action btn-table-view" 
                                               title="{{ __('View') }}">
                                                <i class="bi bi-eye"></i>
                                            </a>
                                        @endif
                                        @if(auth()->user()->hasPermission('fuel.edit'))
                                            <a href="{{ route('admin.fuel-logs.edit', $fuelLog) }}" 
                                               class="btn-table-action btn-table-edit" 
                                               title="{{ __('Edit') }}">
                                                <i class="bi bi-pencil"></i>
                                            </a>
                                        @endif
                                        @if(auth()->user()->hasPermission('fuel.delete'))
                                            <form action="{{ route('admin.fuel-logs.destroy', $fuelLog) }}" method="POST" class="d-inline"
                                                  onsubmit="return confirm('{{ __('Are you sure you want to delete this fuel log?') }}')">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" 
                                                        class="btn-table-action btn-table-delete" 
                                                        title="{{ __('Delete') }}">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </form>
                                        @endif
                                    </div>
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>

            {{-- Pagination --}}
            @if($fuelLogs->hasPages())
            <div class="list-pagination">
                <div class="list-pagination-info">
                    {{ __('Showing') }} {{ $fuelLogs->firstItem() }} {{ __('to') }} {{ $fuelLogs->lastItem() }} {{ __('of') }} {{ $fuelLogs->total() }} {{ __('results') }}
                </div>
                <nav class="list-pagination-nav">
                    {{ $fuelLogs->links() }}
                </nav>
            </div>
            @endif
        @else
            {{-- Empty State --}}
            <div class="list-empty-state">
                <div class="list-empty-state-icon">
                    <i class="bi bi-fuel-pump"></i>
                </div>
                <h4 class="list-empty-state-title">{{ __('No Fuel Logs Found') }}</h4>
                <p class="list-empty-state-text">
                    {{ __('There are no fuel logs matching your criteria. Add a new fuel log to get started.') }}
                </p>
                @if(auth()->user()->hasPermission('fuel.create'))
                <a href="{{ route('admin.fuel-logs.create') }}" class="btn-create">
                    <i class="bi bi-plus-lg"></i>
                    {{ __('Add First Fuel Log') }}
                </a>
                @endif
            </div>
        @endif
    </div>
</div>
@endsection