@extends('layouts.app')

@section('title', __('Shipments') . ' - ' . __('app_name'))

@push('styles')
<link rel="stylesheet" href="{{ asset('css/adminlistpage.css') }}">
@endpush

@section('content')
<div class="container-fluid">
    {{-- Page Header --}}
    <div class="list-page-header">
        <div class="list-page-header-left">
            <h1 class="list-page-title">
                <i class="bi bi-box-seam"></i>
                {{ __('Shipments') }}
            </h1>
            <p class="list-page-subtitle">{{ __('Manage and track all your shipments') }}</p>
        </div>
        <div class="list-page-header-actions">
            <a href="{{ route('admin.shipments.create') }}" class="btn-create">
                <i class="bi bi-plus-lg"></i>
                {{ __('New Shipment') }}
            </a>
        </div>
    </div>

    {{-- Flash Messages --}}
    @if(session('success'))
        <div class="alert alert-success alert-dismissible fade show mb-3" role="alert">
            <i class="bi bi-check-circle me-2"></i>{{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    @if(session('error'))
        <div class="alert alert-danger alert-dismissible fade show mb-3" role="alert">
            <i class="bi bi-exclamation-triangle me-2"></i>{{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    {{-- Stats Bar --}}
    <div class="list-stats-bar">
        <div class="list-stat-card gradient-violet">
            <div class="list-stat-icon">
                <i class="bi bi-box-seam"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $shipments->total() }}</div>
                <div class="list-stat-label">{{ __('Total Shipments') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-graph-up-arrow"></i>
                {{ __('All time') }}
            </div>
        </div>
        <div class="list-stat-card gradient-rose">
            <div class="list-stat-icon">
                <i class="bi bi-clock"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $shipments->where('status', 'pending')->count() }}</div>
                <div class="list-stat-label">{{ __('Pending') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-hourglass-split"></i>
                {{ __('Awaiting') }}
            </div>
        </div>
        <div class="list-stat-card gradient-cyan">
            <div class="list-stat-icon">
                <i class="bi bi-truck"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $shipments->where('status', 'in_transit')->count() }}</div>
                <div class="list-stat-label">{{ __('In Transit') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-geo-alt"></i>
                {{ __('On route') }}
            </div>
        </div>
        <div class="list-stat-card gradient-amber">
            <div class="list-stat-icon">
                <i class="bi bi-check-circle"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $shipments->where('status', 'delivered')->count() }}</div>
                <div class="list-stat-label">{{ __('Delivered') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-check2-all"></i>
                {{ __('Completed') }}
            </div>
        </div>
    </div>

    {{-- Filter Bar --}}
    <div class="list-filter-bar">
        <div class="list-filter-header">
            <div class="list-filter-title">
                <i class="bi bi-funnel"></i>
                {{ __('Filters') }}
            </div>
            @if(request()->hasAny(['search', 'status', 'priority', 'vehicle_type']))
                <a href="{{ route('admin.shipments.index') }}" class="list-filter-clear">
                    <i class="bi bi-x-circle"></i>
                    {{ __('Clear Filters') }}
                </a>
            @endif
        </div>
        <form action="{{ route('admin.shipments.index') }}" method="GET">
            <div class="list-filter-row">
                <div class="list-filter-group search-group">
                    <label>{{ __('Search') }}</label>
                    <div class="list-search-box">
                        <i class="bi bi-search"></i>
                        <input type="text" name="search" class="form-control" 
                               placeholder="{{ __('Search by shipment number...') }}"
                               value="{{ request('search') }}">
                    </div>
                </div>
                <div class="list-filter-group">
                    <label>{{ __('Status') }}</label>
                    <select name="status" class="form-select">
                        <option value="">{{ __('All Statuses') }}</option>
                        <option value="pending" {{ request('status') === 'pending' ? 'selected' : '' }}>{{ __('Pending') }}</option>
                        <option value="assigned" {{ request('status') === 'assigned' ? 'selected' : '' }}>{{ __('Assigned') }}</option>
                        <option value="in_transit" {{ request('status') === 'in_transit' ? 'selected' : '' }}>{{ __('In Transit') }}</option>
                        <option value="delivered" {{ request('status') === 'delivered' ? 'selected' : '' }}>{{ __('Delivered') }}</option>
                        <option value="cancelled" {{ request('status') === 'cancelled' ? 'selected' : '' }}>{{ __('Cancelled') }}</option>
                    </select>
                </div>
                <div class="list-filter-group">
                    <label>{{ __('Priority') }}</label>
                    <select name="priority" class="form-select">
                        <option value="">{{ __('All Priorities') }}</option>
                        <option value="normal" {{ request('priority') === 'normal' ? 'selected' : '' }}>{{ __('Normal') }}</option>
                        <option value="high" {{ request('priority') === 'high' ? 'selected' : '' }}>{{ __('High') }}</option>
                        <option value="urgent" {{ request('priority') === 'urgent' ? 'selected' : '' }}>{{ __('Urgent') }}</option>
                    </select>
                </div>
                <div class="list-filter-group">
                    <label>{{ __('Vehicle Type') }}</label>
                    <select name="vehicle_type" class="form-select">
                        <option value="">{{ __('All Types') }}</option>
                        @foreach($vehicleTypes ?? [] as $type)
                            <option value="{{ $type->id }}" {{ request('vehicle_type') == $type->id ? 'selected' : '' }}>
                                {{ $type->name }}
                            </option>
                        @endforeach
                    </select>
                </div>
                <div class="list-filter-buttons">
                    <button type="submit" class="btn-filter btn-filter-primary">
                        <i class="bi bi-search"></i>
                        {{ __('Apply') }}
                    </button>
                </div>
            </div>
        </form>
    </div>

    {{-- Data Table --}}
    <div class="list-table-container">
        <div class="list-table-header">
            <h5 class="list-table-title">
                {{ __('Shipments List') }}
                <span class="list-table-count">({{ $shipments->total() }} {{ __('records') }})</span>
            </h5>
        </div>

        @if($shipments->count() > 0)
            <div class="table-responsive">
                <table class="list-table">
                    <thead>
                        <tr>
                            <th>{{ __('Shipment #') }}</th>
                            <th>{{ __('Vehicle Type') }}</th>
                            <th>{{ __('Cargo Weight') }}</th>
                            <th>{{ __('Priority') }}</th>
                            <th>{{ __('Status') }}</th>
                            <th>{{ __('Created By') }}</th>
                            <th>{{ __('Created At') }}</th>
                            <th>{{ __('Actions') }}</th>
                        </tr>
                    </thead>
                    <tbody>
                        @php
                            $statusClasses = [
                                'pending' => 'pending',
                                'assigned' => 'assigned',
                                'in_transit' => 'in-transit',
                                'delivered' => 'delivered',
                                'cancelled' => 'cancelled'
                            ];
                            $statusIcons = [
                                'pending' => 'clock',
                                'assigned' => 'check-circle',
                                'in_transit' => 'truck',
                                'delivered' => 'check-all',
                                'cancelled' => 'x-circle'
                            ];
                            $priorityClasses = [
                                'normal' => 'normal',
                                'high' => 'high',
                                'urgent' => 'urgent'
                            ];
                        @endphp
                        @foreach($shipments as $shipment)
                            <tr>
                                <td class="table-cell-id">
                                    <a href="{{ route('admin.shipments.show', $shipment) }}">
                                        {{ $shipment->shipment_number }}
                                    </a>
                                </td>
                                <td>
                                    <span class="d-flex align-items-center gap-2">
                                        <i class="bi bi-truck text-muted"></i>
                                        {{ $shipment->vehicleType->name }}
                                    </span>
                                </td>
                                <td>{{ $shipment->cargo_weight ? number_format($shipment->cargo_weight, 2) . ' T' : '-' }}</td>
                                <td>
                                    <span class="table-badge table-badge-{{ $priorityClasses[$shipment->priority] ?? 'normal' }}">
                                        {{ __(ucfirst($shipment->priority)) }}
                                    </span>
                                </td>
                                <td>
                                    <span class="table-badge table-badge-{{ $statusClasses[$shipment->status] ?? 'pending' }}">
                                        <i class="bi bi-{{ $statusIcons[$shipment->status] ?? 'circle' }}"></i>
                                        {{ __(ucfirst(str_replace('_', ' ', $shipment->status))) }}
                                    </span>
                                </td>
                                <td>
                                    <div class="table-cell-user">
                                        <div class="table-cell-user-avatar">
                                            <i class="bi bi-person"></i>
                                        </div>
                                        <span>{{ $shipment->creator->name }}</span>
                                    </div>
                                </td>
                                <td class="table-cell-date">
                                    <i class="bi bi-calendar"></i>
                                    {{ $shipment->created_at->format('d M Y, h:i A') }}
                                </td>
                                <td>
                                    <div class="table-actions">
                                        <a href="{{ route('admin.shipments.show', $shipment) }}" 
                                           class="btn-table-action btn-table-view" 
                                           title="{{ __('View') }}">
                                            <i class="bi bi-eye"></i>
                                        </a>
                                        @if($shipment->status === 'pending')
                                            <a href="{{ route('admin.shipments.edit', $shipment) }}" 
                                               class="btn-table-action btn-table-edit" 
                                               title="{{ __('Edit') }}">
                                                <i class="bi bi-pencil"></i>
                                            </a>
                                            <form action="{{ route('admin.shipments.destroy', $shipment) }}" 
                                                  method="POST" 
                                                  class="d-inline"
                                                  onsubmit="return confirm('{{ __('Are you sure?') }}')">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" 
                                                        class="btn-table-action btn-table-delete" 
                                                        title="{{ __('Delete') }}">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </form>
                                        @endif
                                    </div>
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>

            {{-- Pagination --}}
            @if($shipments->hasPages())
            <div class="list-pagination">
                <div class="list-pagination-info">
                    {{ __('Showing') }} {{ $shipments->firstItem() }} {{ __('to') }} {{ $shipments->lastItem() }} {{ __('of') }} {{ $shipments->total() }} {{ __('results') }}
                </div>
                <nav class="list-pagination-nav">
                    {{ $shipments->links() }}
                </nav>
            </div>
            @endif
        @else
            {{-- Empty State --}}
            <div class="list-empty-state">
                <div class="list-empty-state-icon">
                    <i class="bi bi-box-seam"></i>
                </div>
                <h4 class="list-empty-state-title">{{ __('No Shipments Found') }}</h4>
                <p class="list-empty-state-text">
                    {{ __('There are no shipments matching your criteria. Create a new shipment to get started.') }}
                </p>
                <a href="{{ route('admin.shipments.create') }}" class="btn-create">
                    <i class="bi bi-plus-lg"></i>
                    {{ __('Create First Shipment') }}
                </a>
            </div>
        @endif
    </div>
</div>
@endsection