@extends('layouts.app')

@section('title', __('edit_trip') . ' - ' . __('app_name'))

@push('styles')
<link rel="stylesheet" href="{{ asset('css/admincreatepage.css') }}">
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
<style>
.map-container { position: relative; border-radius: 12px; overflow: hidden; border: 1px solid #e2e8f0; margin-top: 10px; }
.map-box { height: 250px; width: 100%; z-index: 1; }
.map-search-wrapper { position: relative; margin-bottom: 10px; }
.map-search-input { width: 100%; padding: 12px 45px 12px 16px; border: 1px solid #e2e8f0; border-radius: 8px; font-size: 14px; }
.map-search-input:focus { outline: none; border-color: #667eea; box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1); }
.search-btn { position: absolute; right: 8px; top: 50%; transform: translateY(-50%); background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 6px; padding: 8px 12px; cursor: pointer; }
.coordinates-display { background: #f8fafc; border-radius: 8px; padding: 10px 14px; margin-top: 10px; display: flex; gap: 15px; font-size: 12px; }
.coord-item { display: flex; align-items: center; gap: 6px; }
.coord-label { color: #64748b; font-weight: 500; }
.coord-value { font-family: monospace; color: #334155; background: white; padding: 2px 6px; border-radius: 4px; border: 1px solid #e2e8f0; }
.search-suggestions { position: absolute; top: 100%; left: 0; right: 0; background: white; border: 1px solid #e2e8f0; border-top: none; border-radius: 0 0 8px 8px; max-height: 200px; overflow-y: auto; z-index: 1000; box-shadow: 0 4px 6px rgba(0,0,0,0.1); }
.suggestion-item { padding: 10px 16px; cursor: pointer; border-bottom: 1px solid #f1f5f9; font-size: 13px; }
.suggestion-item:hover { background: #f8fafc; }
.location-badge { display: inline-flex; align-items: center; gap: 6px; padding: 4px 10px; border-radius: 20px; font-size: 12px; font-weight: 500; }
.location-badge.pickup { background: #dcfce7; color: #166534; }
.location-badge.drop { background: #fee2e2; color: #dc2626; }
.location-type-card { border: 2px solid #e2e8f0; border-radius: 12px; padding: 20px; cursor: pointer; transition: all 0.2s ease; background: white; }
.location-type-card:hover { border-color: #667eea; }
.location-type-card.selected { border-color: #667eea; background: linear-gradient(135deg, rgba(102,126,234,0.05) 0%, rgba(118,75,162,0.05) 100%); }
.location-type-card input[type="radio"] { display: none; }
.location-type-icon { width: 48px; height: 48px; border-radius: 12px; display: flex; align-items: center; justify-content: center; font-size: 24px; margin-bottom: 12px; }
.location-type-icon.single { background: linear-gradient(135deg, #10b981 0%, #059669 100%); color: white; }
.location-type-icon.multiple { background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: white; }
.location-type-title { font-weight: 600; color: #1e293b; margin-bottom: 4px; }
.location-type-desc { font-size: 13px; color: #64748b; }
.drop-location-item { background: #f8fafc; border: 1px solid #e2e8f0; border-radius: 12px; padding: 20px; margin-bottom: 16px; position: relative; }
.drop-location-item .location-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; padding-bottom: 10px; border-bottom: 1px solid #e2e8f0; }
.drop-location-item .location-number { display: flex; align-items: center; gap: 10px; }
.drop-location-item .location-number span.seq-num { width: 28px; height: 28px; background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); color: white; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-weight: 600; font-size: 14px; }
.drop-location-item .location-title { font-weight: 600; color: #1e293b; }
.remove-location-btn { background: #fee2e2; color: #dc2626; border: none; border-radius: 8px; padding: 8px 12px; cursor: pointer; font-size: 13px; display: flex; align-items: center; gap: 6px; }
.remove-location-btn:hover { background: #fecaca; }
.add-location-btn { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 10px; padding: 14px 24px; cursor: pointer; font-size: 14px; font-weight: 600; display: flex; align-items: center; gap: 8px; margin-top: 16px; width: 100%; justify-content: center; }
.add-location-btn:hover { transform: translateY(-2px); box-shadow: 0 4px 12px rgba(102,126,234,0.4); }
.location-row { display: grid; grid-template-columns: 1fr 1fr; gap: 12px; margin-bottom: 12px; }
.location-row.full { grid-template-columns: 1fr; }
.trip-header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 12px; margin-bottom: 20px; }
.trip-number { font-size: 24px; font-weight: 700; }
.trip-meta { display: flex; gap: 20px; font-size: 14px; opacity: 0.9; margin-top: 8px; }
</style>
@endpush

@section('content')
<div class="create-page-container">
    <div class="create-page-header">
        <div class="create-page-header-content">
            <div class="create-page-breadcrumb">
                <a href="{{ route('admin.dashboard') }}">{{ __('Dashboard') }}</a>
                <i class="bi bi-chevron-right"></i>
                <a href="{{ route('admin.trips.index') }}">{{ __('trips') }}</a>
                <i class="bi bi-chevron-right"></i>
                <span>{{ __('Edit') }}</span>
            </div>
            <h1 class="create-page-title">
                <div class="create-page-title-icon"><i class="bi bi-pencil"></i></div>
                {{ __('edit_trip') }}
            </h1>
        </div>
        <div class="create-page-actions">
            <a href="{{ route('admin.trips.show', $trip) }}" class="create-btn create-btn-secondary">
                <i class="bi bi-eye"></i> {{ __('View') }}
            </a>
        </div>
    </div>

    <div class="trip-header">
        <div class="trip-number">{{ $trip->trip_number }}</div>
        <div class="trip-meta">
            <span><i class="bi bi-calendar3 me-1"></i> {{ $trip->created_at->format('M d, Y') }}</span>
            <span><i class="bi bi-circle-fill me-1" style="font-size:8px;"></i> {{ __(ucfirst($trip->status)) }}</span>
        </div>
    </div>

    @if($errors->any())
    <div class="create-alert danger">
        <div class="create-alert-icon"><i class="bi bi-exclamation-triangle"></i></div>
        <div class="create-alert-content">
            <div class="create-alert-title">{{ __('Validation Error') }}</div>
        </div>
    </div>
    @endif

    <form action="{{ route('admin.trips.update', $trip) }}" method="POST" id="tripForm">
        @csrf
        @method('PUT')
        <div class="create-form-layout">
            <div class="create-form-main">
                
                {{-- Vehicle & Driver --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-icon primary"><i class="bi bi-truck"></i></div>
                        <div>
                            <h5 class="create-card-title">{{ __('Vehicle & Driver') }}</h5>
                        </div>
                    </div>
                    <div class="create-card-body">
                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label for="vehicle_id" class="create-label required">{{ __('vehicle') }}</label>
                                <select class="create-select" id="vehicle_id" name="vehicle_id" required>
                                    @foreach($allVehicles as $vehicle)
                                        @php $inUse = in_array($vehicle->id, $vehiclesInUse); @endphp
                                        <option value="{{ $vehicle->id }}" {{ $inUse ? 'disabled' : '' }} {{ old('vehicle_id', $trip->vehicle_id) == $vehicle->id ? 'selected' : '' }}>
                                            {{ $vehicle->vehicle_number }} @if($inUse)({{ __('In use') }})@endif
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                            <div class="create-form-group">
                                <label for="driver_id" class="create-label required">{{ __('driver') }}</label>
                                <select class="create-select" id="driver_id" name="driver_id" required>
                                    @foreach($allDrivers as $driver)
                                        @php $onTrip = in_array($driver->id, $driversOnTrip); @endphp
                                        <option value="{{ $driver->id }}" {{ $onTrip ? 'disabled' : '' }} {{ old('driver_id', $trip->driver_id) == $driver->id ? 'selected' : '' }}>
                                            {{ $driver->name }} @if($onTrip)({{ __('On Trip') }})@endif
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                        </div>
                    </div>
                </div>

                {{-- Shipment & Status --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-icon info"><i class="bi bi-box-seam"></i></div>
                        <div><h5 class="create-card-title">{{ __('Shipment & Status') }}</h5></div>
                    </div>
                    <div class="create-card-body">
                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-label">{{ __('shipment') }}</label>
                                <select class="create-select" name="shipment_id">
                                    <option value="">{{ __('No Shipment') }}</option>
                                    @foreach($shipments as $shipment)
                                        <option value="{{ $shipment->id }}" {{ old('shipment_id', $trip->shipment_id) == $shipment->id ? 'selected' : '' }}>
                                            {{ $shipment->shipment_number }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                            <div class="create-form-group">
                                <label class="create-label">{{ __('Reference') }}</label>
                                <input type="text" class="create-input" name="shipment_reference" value="{{ old('shipment_reference', $trip->shipment_reference) }}">
                            </div>
                        </div>
                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-label required">{{ __('status') }}</label>
                                <select class="create-select" name="status" required>
                                    @foreach(['pending', 'assigned', 'in_transit', 'delivered', 'cancelled'] as $status)
                                        <option value="{{ $status }}" {{ old('status', $trip->status) == $status ? 'selected' : '' }}>
                                            {{ __($status) }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                            <div class="create-form-group">
                                <label class="create-label">{{ __('Start Date') }}</label>
                                <input type="datetime-local" class="create-input" name="start_date" value="{{ old('start_date', $trip->start_date?->format('Y-m-d\TH:i')) }}">
                            </div>
                        </div>
                    </div>
                </div>

                {{-- Location Type --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-icon warning"><i class="bi bi-signpost-split"></i></div>
                        <div><h5 class="create-card-title">{{ __('Location Type') }}</h5></div>
                    </div>
                    <div class="create-card-body">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="location-type-card {{ old('has_multiple_locations', $trip->has_multiple_locations) == '0' ? 'selected' : '' }}" id="singleCard">
                                    <input type="radio" name="has_multiple_locations" value="0" {{ old('has_multiple_locations', $trip->has_multiple_locations) == '0' ? 'checked' : '' }}>
                                    <div class="location-type-icon single"><i class="bi bi-geo-alt"></i></div>
                                    <div class="location-type-title">{{ __('Single Drop') }}</div>
                                    <div class="location-type-desc">{{ __('One pickup, one drop') }}</div>
                                </label>
                            </div>
                            <div class="col-md-6">
                                <label class="location-type-card {{ old('has_multiple_locations', $trip->has_multiple_locations) == '1' ? 'selected' : '' }}" id="multipleCard">
                                    <input type="radio" name="has_multiple_locations" value="1" {{ old('has_multiple_locations', $trip->has_multiple_locations) == '1' ? 'checked' : '' }}>
                                    <div class="location-type-icon multiple"><i class="bi bi-geo"></i></div>
                                    <div class="location-type-title">{{ __('Multiple Drops') }}</div>
                                    <div class="location-type-desc">{{ __('One pickup, multiple drops') }}</div>
                                </label>
                            </div>
                        </div>
                    </div>
                </div>

                {{-- Pickup Location --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-icon success"><i class="bi bi-geo-alt-fill"></i></div>
                        <div><h5 class="create-card-title">{{ __('Pickup Location') }}</h5></div>
                        <span class="location-badge pickup"><i class="bi bi-arrow-up-circle"></i> {{ __('Pickup') }}</span>
                    </div>
                    <div class="create-card-body">
                        <div class="map-search-wrapper">
                            <input type="text" class="map-search-input" id="pickup_location" name="pickup_location" value="{{ old('pickup_location', $trip->pickup_location) }}" required>
                            <button type="button" class="search-btn" onclick="geocodeAddress('pickup')"><i class="bi bi-search"></i></button>
                            <div class="search-suggestions" id="pickup_suggestions"></div>
                        </div>
                        <div class="map-container"><div id="pickup_map" class="map-box"></div></div>
                        <div class="coordinates-display">
                            <div class="coord-item"><span class="coord-label">Lat:</span><span class="coord-value" id="pickup_lat_display">{{ $trip->pickup_lat ? number_format($trip->pickup_lat, 6) : '--' }}</span></div>
                            <div class="coord-item"><span class="coord-label">Lng:</span><span class="coord-value" id="pickup_lng_display">{{ $trip->pickup_lng ? number_format($trip->pickup_lng, 6) : '--' }}</span></div>
                        </div>
                        <input type="hidden" name="pickup_lat" id="pickup_lat" value="{{ old('pickup_lat', $trip->pickup_lat) }}">
                        <input type="hidden" name="pickup_lng" id="pickup_lng" value="{{ old('pickup_lng', $trip->pickup_lng) }}">
                    </div>
                </div>

                {{-- Single Drop --}}
                <div class="create-card" id="singleDropCard">
                    <div class="create-card-header">
                        <div class="create-card-icon danger"><i class="bi bi-geo-fill"></i></div>
                        <div><h5 class="create-card-title">{{ __('Drop Location') }}</h5></div>
                        <span class="location-badge drop"><i class="bi bi-arrow-down-circle"></i> {{ __('Drop') }}</span>
                    </div>
                    <div class="create-card-body">
                        <div class="map-search-wrapper">
                            <input type="text" class="map-search-input" id="drop_location" name="drop_location" value="{{ old('drop_location', $trip->drop_location) }}">
                            <button type="button" class="search-btn" onclick="geocodeAddress('drop')"><i class="bi bi-search"></i></button>
                            <div class="search-suggestions" id="drop_suggestions"></div>
                        </div>
                        <div class="map-container"><div id="drop_map" class="map-box"></div></div>
                        <div class="coordinates-display">
                            <div class="coord-item"><span class="coord-label">Lat:</span><span class="coord-value" id="drop_lat_display">{{ $trip->drop_lat ? number_format($trip->drop_lat, 6) : '--' }}</span></div>
                            <div class="coord-item"><span class="coord-label">Lng:</span><span class="coord-value" id="drop_lng_display">{{ $trip->drop_lng ? number_format($trip->drop_lng, 6) : '--' }}</span></div>
                        </div>
                        <input type="hidden" name="drop_lat" id="drop_lat" value="{{ old('drop_lat', $trip->drop_lat) }}">
                        <input type="hidden" name="drop_lng" id="drop_lng" value="{{ old('drop_lng', $trip->drop_lng) }}">
                    </div>
                </div>

                {{-- Multiple Drops --}}
                <div class="create-card" id="multipleDropCard" style="display: none;">
                    <div class="create-card-header">
                        <div class="create-card-icon danger"><i class="bi bi-geo-fill"></i></div>
                        <div><h5 class="create-card-title">{{ __('Drop Locations') }}</h5></div>
                        <span class="location-badge drop"><i class="bi bi-list-ol"></i> <span id="locationCount">0</span> {{ __('Drops') }}</span>
                    </div>
                    <div class="create-card-body">
                        <div id="dropLocationsContainer"></div>
                        <button type="button" class="add-location-btn" onclick="addDropLocation()">
                            <i class="bi bi-plus-circle"></i> {{ __('Add Drop Location') }}
                        </button>
                    </div>
                </div>

                {{-- Instructions --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-icon secondary"><i class="bi bi-card-text"></i></div>
                        <div><h5 class="create-card-title">{{ __('Instructions') }}</h5></div>
                    </div>
                    <div class="create-card-body">
                        <textarea class="create-textarea" name="trip_instructions" rows="3">{{ old('trip_instructions', $trip->trip_instructions) }}</textarea>
                    </div>
                </div>

                {{-- Submit --}}
                <div class="create-card">
                    <div class="create-card-body">
                        <div class="create-form-actions">
                            <a href="{{ route('admin.trips.index') }}" class="create-btn create-btn-secondary">
                                <i class="bi bi-x-lg"></i> {{ __('cancel') }}
                            </a>
                            <button type="submit" class="create-btn create-btn-primary">
                                <i class="bi bi-check-circle"></i> {{ __('Update Trip') }}
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <div class="create-form-sidebar">
                <div class="create-sidebar-card">
                    <div class="create-sidebar-header">
                        <div class="create-sidebar-icon success"><i class="bi bi-lightning"></i></div>
                        <h5 class="create-sidebar-title">{{ __('Actions') }}</h5>
                    </div>
                    <div class="create-sidebar-body">
                        <div class="create-quick-actions">
                            <a href="{{ route('admin.trips.show', $trip) }}" class="create-quick-action-btn">
                                <i class="bi bi-eye"></i> {{ __('View Trip') }}
                            </a>
                            <a href="{{ route('admin.trips.index') }}" class="create-quick-action-btn">
                                <i class="bi bi-list-ul"></i> {{ __('All Trips') }}
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </form>
</div>
@endsection

@push('scripts')
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
<script>
const defaultLat = 13.0827, defaultLng = 80.2707;
let maps = {}, markers = {};
let dropLocationIndex = 100; // Start high to avoid conflicts with existing IDs

// Existing locations from database
const existingLocations = @json($trip->locations);
const existingPickup = { lat: {{ $trip->pickup_lat ?? 'null' }}, lng: {{ $trip->pickup_lng ?? 'null' }} };
const existingDrop = { lat: {{ $trip->drop_lat ?? 'null' }}, lng: {{ $trip->drop_lng ?? 'null' }} };

document.addEventListener('DOMContentLoaded', function() {
    initMap('pickup');
    initMap('drop');
    
    // Set existing markers
    if (existingPickup.lat && existingPickup.lng) {
        setMarker('pickup', existingPickup.lat, existingPickup.lng);
    }
    if (existingDrop.lat && existingDrop.lng) {
        setMarker('drop', existingDrop.lat, existingDrop.lng);
    }
    
    updateLocationTypeUI();
    
    document.querySelectorAll('input[name="has_multiple_locations"]').forEach(radio => {
        radio.addEventListener('change', updateLocationTypeUI);
    });
    
    document.addEventListener('click', e => {
        if (!e.target.closest('.map-search-wrapper')) {
            document.querySelectorAll('.search-suggestions').forEach(el => el.style.display = 'none');
        }
    });
    
    document.querySelectorAll('.map-search-input').forEach(input => {
        input.addEventListener('keypress', e => {
            if (e.key === 'Enter') {
                e.preventDefault();
                const type = input.id.replace('_location', '').replace('locations_', '').replace('_address', '');
                geocodeAddress(type);
            }
        });
    });
});

function initMap(type, index = null) {
    const mapId = index !== null ? `locations_${index}_map` : `${type}_map`;
    const mapEl = document.getElementById(mapId);
    if (!mapEl || maps[mapId]) return;
    
    const map = L.map(mapId).setView([defaultLat, defaultLng], 10);
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', { attribution: '© OpenStreetMap' }).addTo(map);
    map.on('click', e => {
        setMarker(type, e.latlng.lat, e.latlng.lng, index);
        reverseGeocode(type, e.latlng.lat, e.latlng.lng, index);
    });
    maps[mapId] = map;
    setTimeout(() => map.invalidateSize(), 100);
}

function setMarker(type, lat, lng, index = null) {
    const mapId = index !== null ? `locations_${index}_map` : `${type}_map`;
    const map = maps[mapId];
    if (!map) return;
    
    const color = type === 'pickup' ? '#10b981' : '#ef4444';
    const icon = L.divIcon({
        className: 'custom-marker',
        html: `<div style="background:${color};width:24px;height:24px;border-radius:50%;border:3px solid white;box-shadow:0 2px 6px rgba(0,0,0,0.3);"></div>`,
        iconSize: [24, 24], iconAnchor: [12, 12]
    });
    
    if (markers[mapId]) {
        markers[mapId].setLatLng([lat, lng]);
    } else {
        markers[mapId] = L.marker([lat, lng], {icon, draggable: true}).addTo(map);
        markers[mapId].on('dragend', e => {
            const pos = e.target.getLatLng();
            updateCoords(type, pos.lat, pos.lng, index);
            reverseGeocode(type, pos.lat, pos.lng, index);
        });
    }
    map.setView([lat, lng], 15);
    updateCoords(type, lat, lng, index);
}

function updateCoords(type, lat, lng, index = null) {
    if (index !== null) {
        document.getElementById(`locations_${index}_lat`).value = lat.toFixed(8);
        document.getElementById(`locations_${index}_lng`).value = lng.toFixed(8);
        document.getElementById(`locations_${index}_lat_display`).textContent = lat.toFixed(6);
        document.getElementById(`locations_${index}_lng_display`).textContent = lng.toFixed(6);
    } else {
        document.getElementById(`${type}_lat`).value = lat.toFixed(8);
        document.getElementById(`${type}_lng`).value = lng.toFixed(8);
        document.getElementById(`${type}_lat_display`).textContent = lat.toFixed(6);
        document.getElementById(`${type}_lng_display`).textContent = lng.toFixed(6);
    }
}

async function geocodeAddress(type, index = null) {
    const inputId = index !== null ? `locations_${index}_address` : `${type}_location`;
    const address = document.getElementById(inputId)?.value;
    if (!address?.trim()) return;
    
    try {
        const response = await fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(address)}&limit=5`);
        const data = await response.json();
        if (data.length > 0) showSuggestions(type, data, index);
    } catch (error) { console.error('Geocode error:', error); }
}

function showSuggestions(type, results, index = null) {
    const containerId = index !== null ? `locations_${index}_suggestions` : `${type}_suggestions`;
    const container = document.getElementById(containerId);
    if (!container) return;
    
    container.innerHTML = '';
    results.forEach(result => {
        const div = document.createElement('div');
        div.className = 'suggestion-item';
        div.textContent = result.display_name;
        div.onclick = () => {
            const inputId = index !== null ? `locations_${index}_address` : `${type}_location`;
            document.getElementById(inputId).value = result.display_name;
            setMarker(type, parseFloat(result.lat), parseFloat(result.lon), index);
            container.style.display = 'none';
        };
        container.appendChild(div);
    });
    container.style.display = 'block';
}

async function reverseGeocode(type, lat, lng, index = null) {
    try {
        const response = await fetch(`https://nominatim.openstreetmap.org/reverse?format=json&lat=${lat}&lon=${lng}`);
        const data = await response.json();
        if (data.display_name) {
            const inputId = index !== null ? `locations_${index}_address` : `${type}_location`;
            document.getElementById(inputId).value = data.display_name;
        }
    } catch (error) { console.error('Reverse geocode error:', error); }
}

function updateLocationTypeUI() {
    const isMultiple = document.querySelector('input[name="has_multiple_locations"]:checked')?.value === '1';
    
    document.getElementById('singleCard').classList.toggle('selected', !isMultiple);
    document.getElementById('multipleCard').classList.toggle('selected', isMultiple);
    document.getElementById('singleDropCard').style.display = isMultiple ? 'none' : 'block';
    document.getElementById('multipleDropCard').style.display = isMultiple ? 'block' : 'none';
    
    if (isMultiple) {
        setTimeout(() => Object.values(maps).forEach(map => map.invalidateSize()), 100);
        
        // Load existing locations or add one
        if (document.getElementById('dropLocationsContainer').children.length === 0) {
            if (existingLocations.length > 0) {
                existingLocations.forEach(loc => addDropLocation(loc));
            } else {
                addDropLocation();
            }
        }
    }
}

function addDropLocation(existingData = null) {
    const container = document.getElementById('dropLocationsContainer');
    const index = existingData?.id || dropLocationIndex++;
    const count = container.children.length + 1;
    
    const html = `
        <div class="drop-location-item" id="location_${index}" data-index="${index}">
            <div class="location-header">
                <div class="location-number">
                    <span class="seq-num">${count}</span>
                    <span class="location-title">{{ __('Drop Location') }} #${count}</span>
                </div>
                <button type="button" class="remove-location-btn" onclick="removeDropLocation(${index})">
                    <i class="bi bi-trash"></i> {{ __('Remove') }}
                </button>
            </div>
            ${existingData ? `<input type="hidden" name="locations[${index}][id]" value="${existingData.id}">` : ''}
            <div class="location-row">
                <div class="create-form-group">
                    <label class="create-label required">{{ __('Address') }}</label>
                    <div class="map-search-wrapper">
                        <input type="text" class="map-search-input" id="locations_${index}_address" name="locations[${index}][address]" value="${existingData?.address || ''}" required>
                        <button type="button" class="search-btn" onclick="geocodeAddress('drop', ${index})"><i class="bi bi-search"></i></button>
                        <div class="search-suggestions" id="locations_${index}_suggestions"></div>
                    </div>
                </div>
                <div class="create-form-group">
                    <label class="create-label">{{ __('Location Name') }}</label>
                    <input type="text" class="create-input" name="locations[${index}][location_name]" value="${existingData?.location_name || ''}">
                </div>
            </div>
            <div class="location-row">
                <div class="create-form-group">
                    <label class="create-label">{{ __('Contact Name') }}</label>
                    <input type="text" class="create-input" name="locations[${index}][contact_name]" value="${existingData?.contact_name || ''}">
                </div>
                <div class="create-form-group">
                    <label class="create-label">{{ __('Contact Phone') }}</label>
                    <input type="text" class="create-input" name="locations[${index}][contact_phone]" value="${existingData?.contact_phone || ''}">
                </div>
            </div>
            <div class="location-row full">
                <div class="create-form-group">
                    <label class="create-label">{{ __('Notes') }}</label>
                    <input type="text" class="create-input" name="locations[${index}][notes]" value="${existingData?.notes || ''}">
                </div>
            </div>
            <div class="map-container"><div id="locations_${index}_map" class="map-box"></div></div>
            <div class="coordinates-display">
                <div class="coord-item"><span class="coord-label">Lat:</span><span class="coord-value" id="locations_${index}_lat_display">${existingData?.lat ? parseFloat(existingData.lat).toFixed(6) : '--'}</span></div>
                <div class="coord-item"><span class="coord-label">Lng:</span><span class="coord-value" id="locations_${index}_lng_display">${existingData?.lng ? parseFloat(existingData.lng).toFixed(6) : '--'}</span></div>
            </div>
            <input type="hidden" name="locations[${index}][lat]" id="locations_${index}_lat" value="${existingData?.lat || ''}">
            <input type="hidden" name="locations[${index}][lng]" id="locations_${index}_lng" value="${existingData?.lng || ''}">
        </div>
    `;
    
    container.insertAdjacentHTML('beforeend', html);
    updateLocationCount();
    
    setTimeout(() => {
        initMap('drop', index);
        if (existingData?.lat && existingData?.lng) {
            setMarker('drop', parseFloat(existingData.lat), parseFloat(existingData.lng), index);
        }
    }, 50);
    
    document.getElementById(`locations_${index}_address`).addEventListener('keypress', e => {
        if (e.key === 'Enter') { e.preventDefault(); geocodeAddress('drop', index); }
    });
}

function removeDropLocation(index) {
    const el = document.getElementById(`location_${index}`);
    if (el) {
        const mapId = `locations_${index}_map`;
        if (maps[mapId]) { maps[mapId].remove(); delete maps[mapId]; delete markers[mapId]; }
        el.remove();
        updateLocationCount();
        renumberLocations();
    }
}

function updateLocationCount() {
    document.getElementById('locationCount').textContent = document.getElementById('dropLocationsContainer').children.length;
}

function renumberLocations() {
    document.querySelectorAll('#dropLocationsContainer .drop-location-item').forEach((item, i) => {
        const num = i + 1;
        item.querySelector('.seq-num').textContent = num;
        item.querySelector('.location-title').textContent = `{{ __('Drop Location') }} #${num}`;
    });
}

const vehicleOwners = @json($vehicleOwners);
const driverOwnVehicles = @json($driverOwnVehicles);
document.getElementById('vehicle_id').addEventListener('change', function() { if (vehicleOwners[this.value]) document.getElementById('driver_id').value = vehicleOwners[this.value]; });
document.getElementById('driver_id').addEventListener('change', function() { if (driverOwnVehicles[this.value]) document.getElementById('vehicle_id').value = driverOwnVehicles[this.value]; });
</script>
@endpush