@extends('layouts.app')

@section('title', __('Route Optimizer') . ' - ' . __('app_name'))

@push('styles')
<link rel="stylesheet" href="{{ asset('css/adminlistpage.css') }}">
<style>
/* ===== Route Optimizer Styles (Using Theme Variables) ===== */

/* Hero Section */
.optimizer-hero {
    background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
    border-radius: 20px;
    padding: 2rem 2.5rem;
    margin-bottom: 1.5rem;
    position: relative;
    overflow: hidden;
    box-shadow: 0 10px 40px rgba(249, 115, 22, 0.25);
}
.optimizer-hero::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -20%;
    width: 400px;
    height: 400px;
    background: radial-gradient(circle, rgba(255,255,255,0.15) 0%, transparent 70%);
}
.optimizer-hero-content { position: relative; z-index: 2; display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 1rem; }
.optimizer-hero-title { font-size: 1.75rem; font-weight: 700; color: white; margin: 0 0 0.5rem; display: flex; align-items: center; gap: 0.75rem; }
.optimizer-hero-title i { width: 50px; height: 50px; background: rgba(255,255,255,0.2); border-radius: 14px; display: flex; align-items: center; justify-content: center; font-size: 1.5rem; }
.optimizer-hero-subtitle { color: rgba(255,255,255,0.9); font-size: 0.95rem; margin: 0; }
.ai-badge { display: inline-flex; align-items: center; gap: 0.5rem; background: rgba(255,255,255,0.2); backdrop-filter: blur(10px); padding: 0.625rem 1.25rem; border-radius: 50px; color: white; font-weight: 600; font-size: 0.85rem; border: 1px solid rgba(255,255,255,0.3); }

/* Layout */
.optimizer-layout { display: grid; grid-template-columns: 1fr 360px; gap: 1.5rem; }
@media (max-width: 1200px) { .optimizer-layout { grid-template-columns: 1fr; } }

/* Cards */
.optimizer-card { background: var(--card-bg); border: 1px solid var(--border-color); border-radius: 16px; box-shadow: 0 2px 8px rgba(0,0,0,0.04); }
[data-theme="dark"] .optimizer-card { box-shadow: 0 2px 8px rgba(0,0,0,0.2); }
.optimizer-card-header { padding: 1.25rem 1.5rem; border-bottom: 1px solid var(--border-color); display: flex; align-items: center; gap: 1rem; }
.optimizer-card-icon { width: 46px; height: 46px; border-radius: 12px; display: flex; align-items: center; justify-content: center; font-size: 1.25rem; }
.optimizer-card-icon.primary { background: var(--primary-light); color: var(--primary-color); }
.optimizer-card-icon.success { background: var(--success-light); color: var(--success-color); }
.optimizer-card-icon.warning { background: var(--warning-light); color: var(--warning-color); }
.optimizer-card-icon.info { background: var(--info-light); color: var(--info-color); }
.optimizer-card-title { font-size: 1.1rem; font-weight: 600; color: var(--text-primary); margin: 0; }
.optimizer-card-subtitle { font-size: 0.8rem; color: var(--text-muted); margin: 0.25rem 0 0; }
.optimizer-card-body { padding: 1.5rem; }

/* Empty State */
.optimizer-empty { text-align: center; padding: 3rem 1.5rem; }
.optimizer-empty-icon { width: 90px; height: 90px; background: var(--body-bg); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 1.25rem; font-size: 2.5rem; color: var(--text-muted); }
.optimizer-empty-title { font-size: 1.15rem; font-weight: 600; color: var(--text-primary); margin-bottom: 0.5rem; }
.optimizer-empty-text { color: var(--text-secondary); margin-bottom: 1.5rem; font-size: 0.9rem; }

/* Trip Selection */
.trip-list { display: flex; flex-direction: column; gap: 1rem; }
.trip-item { border: 2px solid var(--border-color); border-radius: 14px; padding: 1.25rem; cursor: pointer; transition: all 0.2s ease; background: var(--card-bg); }
.trip-item:hover { border-color: var(--primary-color); }
.trip-item.selected { border-color: var(--primary-color); background: var(--primary-light); }
[data-theme="dark"] .trip-item.selected { background: rgba(249, 115, 22, 0.15); }
.trip-item input[type="radio"] { display: none; }
.trip-item-header { display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 1rem; }
.trip-number { font-size: 1rem; font-weight: 600; color: var(--primary-color); }
.trip-meta { display: flex; gap: 1rem; flex-wrap: wrap; margin-top: 0.5rem; }
.trip-meta-item { display: flex; align-items: center; gap: 0.4rem; font-size: 0.8rem; color: var(--text-secondary); }
.trip-meta-item i { color: var(--text-muted); }
.trip-status { padding: 0.35rem 0.75rem; border-radius: 20px; font-size: 0.75rem; font-weight: 600; text-transform: uppercase; }
.trip-status.assigned { background: var(--info-light); color: var(--info-color); }
.trip-status.in_transit { background: var(--success-light); color: var(--success-color); }
.stops-count { background: var(--warning-light); color: #92400e; padding: 0.35rem 0.75rem; border-radius: 20px; font-size: 0.75rem; font-weight: 600; display: flex; align-items: center; gap: 0.35rem; }
[data-theme="dark"] .stops-count { color: var(--warning-color); }

/* Route Preview */
.route-preview { background: var(--body-bg); border-radius: 10px; padding: 1rem; margin-top: 1rem; }
.route-stop { display: flex; align-items: flex-start; gap: 0.75rem; margin-bottom: 0.75rem; position: relative; }
.route-stop:last-child { margin-bottom: 0; }
.route-stop::before { content: ''; position: absolute; left: 11px; top: 26px; bottom: -12px; width: 2px; background: var(--border-color); }
.route-stop:last-child::before { display: none; }
.route-marker { width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 10px; font-weight: 700; color: white; flex-shrink: 0; position: relative; z-index: 1; }
.route-marker.pickup { background: var(--success-color); }
.route-marker.drop { background: var(--danger-color); }
.route-marker.delivery { background: var(--info-color); }
.route-stop-content { flex: 1; min-width: 0; }
.route-stop-label { font-size: 0.7rem; font-weight: 600; text-transform: uppercase; }
.route-stop-label.pickup { color: var(--success-color); }
.route-stop-label.drop { color: var(--danger-color); }
.route-stop-label.delivery { color: var(--info-color); }
.route-stop-address { font-size: 0.8rem; color: var(--text-secondary); line-height: 1.4; }
.route-more { background: var(--card-bg); border-radius: 6px; padding: 0.4rem 0.75rem; margin-left: 32px; font-size: 0.8rem; color: var(--primary-color); font-weight: 600; display: inline-flex; align-items: center; gap: 0.4rem; }

/* Priority Selection */
.priority-grid { display: grid; grid-template-columns: repeat(3, 1fr); gap: 0.75rem; margin-bottom: 1.5rem; }
@media (max-width: 576px) { .priority-grid { grid-template-columns: 1fr; } }
.priority-card { border: 2px solid var(--border-color); border-radius: 12px; padding: 1.25rem 1rem; cursor: pointer; transition: all 0.2s ease; text-align: center; background: var(--card-bg); }
.priority-card:hover { border-color: var(--primary-color); }
.priority-card.selected { border-color: var(--primary-color); background: var(--primary-light); }
[data-theme="dark"] .priority-card.selected { background: rgba(249, 115, 22, 0.15); }
.priority-card input[type="radio"] { display: none; }
.priority-icon { width: 44px; height: 44px; border-radius: 12px; display: flex; align-items: center; justify-content: center; font-size: 1.25rem; margin: 0 auto 0.75rem; }
.priority-card.distance .priority-icon { background: var(--info-light); color: var(--info-color); }
.priority-card.time .priority-icon { background: var(--success-light); color: var(--success-color); }
.priority-card.fuel .priority-icon { background: var(--warning-light); color: #92400e; }
[data-theme="dark"] .priority-card.fuel .priority-icon { color: var(--warning-color); }
.priority-card.selected .priority-icon { background: var(--primary-color); color: white; }
.priority-title { font-size: 0.9rem; font-weight: 600; color: var(--text-primary); margin-bottom: 0.25rem; }
.priority-desc { font-size: 0.75rem; color: var(--text-muted); }

/* Constraints */
.constraint-list { display: flex; flex-direction: column; gap: 0.75rem; margin-bottom: 1.5rem; }
.constraint-item { display: flex; align-items: center; justify-content: space-between; padding: 1rem 1.25rem; background: var(--body-bg); border-radius: 10px; transition: all 0.2s ease; }
.constraint-item:hover { background: var(--border-color); }
[data-theme="dark"] .constraint-item:hover { background: var(--sidebar-hover-bg); }
.constraint-info { display: flex; align-items: center; gap: 1rem; }
.constraint-icon { width: 38px; height: 38px; background: var(--card-bg); border-radius: 10px; display: flex; align-items: center; justify-content: center; font-size: 1rem; color: var(--text-secondary); box-shadow: 0 2px 6px rgba(0,0,0,0.04); }
[data-theme="dark"] .constraint-icon { box-shadow: 0 2px 6px rgba(0,0,0,0.15); }
.constraint-label { font-size: 0.9rem; font-weight: 600; color: var(--text-primary); }
.constraint-desc { font-size: 0.75rem; color: var(--text-muted); }

/* Toggle Switch */
.toggle-switch { position: relative; width: 48px; height: 26px; }
.toggle-switch input { opacity: 0; width: 0; height: 0; }
.toggle-slider { position: absolute; cursor: pointer; top: 0; left: 0; right: 0; bottom: 0; background-color: var(--border-color); transition: .3s; border-radius: 26px; }
.toggle-slider:before { position: absolute; content: ""; height: 20px; width: 20px; left: 3px; bottom: 3px; background-color: white; transition: .3s; border-radius: 50%; box-shadow: 0 2px 4px rgba(0,0,0,0.2); }
.toggle-switch input:checked + .toggle-slider { background: var(--primary-color); }
.toggle-switch input:checked + .toggle-slider:before { transform: translateX(22px); }

/* Buttons */
.optimizer-btn { padding: 0.875rem 1.5rem; border-radius: 10px; font-size: 0.9rem; font-weight: 600; border: none; cursor: pointer; display: inline-flex; align-items: center; gap: 0.5rem; transition: all 0.2s ease; text-decoration: none; }
.optimizer-btn-primary { background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%); color: white; box-shadow: 0 4px 12px rgba(249, 115, 22, 0.25); }
.optimizer-btn-primary:hover { transform: translateY(-2px); box-shadow: 0 6px 20px rgba(249, 115, 22, 0.35); color: white; }
.optimizer-btn-primary:disabled { opacity: 0.6; cursor: not-allowed; transform: none; }
.optimizer-btn-secondary { background: var(--card-bg); color: var(--text-secondary); border: 1px solid var(--border-color); }
.optimizer-btn-secondary:hover { border-color: var(--text-muted); background: var(--body-bg); color: var(--text-primary); }

/* Sidebar Cards */
.sidebar-card { background: var(--card-bg); border: 1px solid var(--border-color); border-radius: 16px; box-shadow: 0 2px 8px rgba(0,0,0,0.04); overflow: hidden; margin-bottom: 1.25rem; }
[data-theme="dark"] .sidebar-card { box-shadow: 0 2px 8px rgba(0,0,0,0.2); }
.sidebar-header { padding: 1rem 1.25rem; border-bottom: 1px solid var(--border-color); display: flex; align-items: center; gap: 0.75rem; }
.sidebar-icon { width: 36px; height: 36px; border-radius: 10px; display: flex; align-items: center; justify-content: center; font-size: 1rem; }
.sidebar-icon.info { background: var(--info-light); color: var(--info-color); }
.sidebar-icon.success { background: var(--success-light); color: var(--success-color); }
.sidebar-icon.warning { background: var(--warning-light); color: #92400e; }
[data-theme="dark"] .sidebar-icon.warning { color: var(--warning-color); }
.sidebar-title { font-size: 0.95rem; font-weight: 600; color: var(--text-primary); }
.sidebar-body { padding: 1.25rem; }

/* Steps List */
.steps-list { display: flex; flex-direction: column; gap: 1.25rem; }
.step-item { display: flex; align-items: flex-start; gap: 1rem; }
.step-number { width: 32px; height: 32px; background: var(--primary-color); color: white; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 0.8rem; font-weight: 700; flex-shrink: 0; }
.step-content h6 { font-size: 0.9rem; font-weight: 600; color: var(--text-primary); margin: 0 0 0.25rem; }
.step-content p { font-size: 0.8rem; color: var(--text-muted); margin: 0; line-height: 1.4; }

/* Benefits List */
.benefits-list { display: flex; flex-direction: column; gap: 0.75rem; }
.benefit-item { display: flex; align-items: center; gap: 0.75rem; padding: 0.75rem 1rem; background: var(--success-light); border-radius: 10px; }
.benefit-icon { width: 28px; height: 28px; background: var(--success-color); color: white; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-size: 0.8rem; flex-shrink: 0; }
.benefit-text { font-size: 0.8rem; font-weight: 500; color: var(--success-color); }
[data-theme="dark"] .benefit-text { color: #34d399; }

/* Requirements List */
.requirements-list { display: flex; flex-direction: column; gap: 0.6rem; }
.requirement-item { display: flex; align-items: center; gap: 0.6rem; font-size: 0.8rem; color: #92400e; }
[data-theme="dark"] .requirement-item { color: var(--warning-color); }
.requirement-item i { font-size: 6px; color: var(--warning-color); }

/* Section Title */
.section-title { font-weight: 600; color: var(--text-primary); margin: 1.5rem 0 1rem; display: flex; align-items: center; gap: 0.5rem; font-size: 0.95rem; }
.section-title i { color: var(--primary-color); }
</style>
@endpush

@section('content')
<div class="container-fluid">
    {{-- Hero Header --}}
    <div class="optimizer-hero">
        <div class="optimizer-hero-content">
            <div>
                <h1 class="optimizer-hero-title">
                    <i class="bi bi-magic"></i>
                    {{ __('AI Route Optimizer') }}
                </h1>
                <p class="optimizer-hero-subtitle">{{ __('Optimize delivery routes using artificial intelligence for maximum efficiency') }}</p>
            </div>
            <div class="ai-badge">
                <i class="bi bi-robot"></i>
                {{ __('Powered by OpenAI') }}
            </div>
        </div>
    </div>

    {{-- Alerts --}}
    @if(session('error'))
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="bi bi-exclamation-triangle me-2"></i>{{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif
    @if(session('success'))
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="bi bi-check-circle me-2"></i>{{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    <div class="optimizer-layout">
        {{-- Main Content --}}
        <div class="optimizer-main">
            <div class="optimizer-card">
                <div class="optimizer-card-header">
                    <div class="optimizer-card-icon primary"><i class="bi bi-geo-alt"></i></div>
                    <div>
                        <h5 class="optimizer-card-title">{{ __('Select Trip to Optimize') }}</h5>
                        <p class="optimizer-card-subtitle">{{ __('Choose an active trip with multiple stops') }}</p>
                    </div>
                </div>
                <div class="optimizer-card-body">
                    @if($trips->isEmpty())
                        <div class="optimizer-empty">
                            <div class="optimizer-empty-icon"><i class="bi bi-inbox"></i></div>
                            <h4 class="optimizer-empty-title">{{ __('No Trips Available') }}</h4>
                            <p class="optimizer-empty-text">{{ __('Create a trip with pickup and drop locations to start optimizing routes.') }}</p>
                            <a href="{{ route('admin.trips.create') }}" class="optimizer-btn optimizer-btn-primary">
                                <i class="bi bi-plus-lg"></i> {{ __('Create Trip') }}
                            </a>
                        </div>
                    @else
                        <form action="{{ route('route-optimizer.optimize') }}" method="POST" id="optimizeForm">
                            @csrf
                            
                            {{-- Trip Selection --}}
                            <div class="trip-list">
                                @foreach($trips as $trip)
                                    @php
                                        $stopCount = ($trip->pickup_location ? 1 : 0) + ($trip->drop_location ? 1 : 0) + ($trip->locations->count() ?? 0) + ($trip->deliveries->count() ?? 0);
                                    @endphp
                                    <label class="trip-item {{ old('trip_id') == $trip->id ? 'selected' : '' }}" data-trip="{{ $trip->id }}">
                                        <input type="radio" name="trip_id" value="{{ $trip->id }}" required {{ old('trip_id') == $trip->id ? 'checked' : '' }}>
                                        
                                        <div class="trip-item-header">
                                            <div>
                                                <div class="trip-number">{{ $trip->trip_number }}</div>
                                                <div class="trip-meta">
                                                    <span class="trip-meta-item"><i class="bi bi-truck"></i> {{ $trip->vehicle->vehicle_number ?? 'N/A' }}</span>
                                                    <span class="trip-meta-item"><i class="bi bi-person"></i> {{ $trip->driver->name ?? 'N/A' }}</span>
                                                </div>
                                            </div>
                                            <div style="display: flex; gap: 0.5rem; align-items: center;">
                                                <span class="trip-status {{ $trip->status }}">{{ __(ucfirst(str_replace('_', ' ', $trip->status))) }}</span>
                                                <span class="stops-count"><i class="bi bi-pin-map"></i> {{ $stopCount }} {{ __('stops') }}</span>
                                            </div>
                                        </div>
                                        
                                        {{-- Route Preview --}}
                                        <div class="route-preview">
                                            @if($trip->pickup_location)
                                                <div class="route-stop">
                                                    <span class="route-marker pickup"><i class="bi bi-geo-alt-fill"></i></span>
                                                    <div class="route-stop-content">
                                                        <div class="route-stop-label pickup">{{ __('Pickup') }}</div>
                                                        <div class="route-stop-address">{{ Str::limit($trip->pickup_location, 55) }}</div>
                                                    </div>
                                                </div>
                                            @endif
                                            
                                            @if($trip->has_multiple_locations && $trip->locations->count() > 0)
                                                @foreach($trip->locations->take(2) as $index => $location)
                                                    <div class="route-stop">
                                                        <span class="route-marker delivery">{{ $index + 1 }}</span>
                                                        <div class="route-stop-content">
                                                            <div class="route-stop-label delivery">{{ $location->location_name ?? __('Drop') . ' #' . ($index + 1) }}</div>
                                                            <div class="route-stop-address">{{ Str::limit($location->address, 45) }}</div>
                                                        </div>
                                                    </div>
                                                @endforeach
                                                @if($trip->locations->count() > 2)
                                                    <span class="route-more"><i class="bi bi-three-dots"></i> +{{ $trip->locations->count() - 2 }} {{ __('more') }}</span>
                                                @endif
                                            @elseif($trip->drop_location)
                                                <div class="route-stop">
                                                    <span class="route-marker drop"><i class="bi bi-flag-fill"></i></span>
                                                    <div class="route-stop-content">
                                                        <div class="route-stop-label drop">{{ __('Drop') }}</div>
                                                        <div class="route-stop-address">{{ Str::limit($trip->drop_location, 55) }}</div>
                                                    </div>
                                                </div>
                                            @endif
                                        </div>
                                    </label>
                                @endforeach
                            </div>

                            {{-- Optimization Priority --}}
                            <h6 class="section-title"><i class="bi bi-sliders"></i> {{ __('Optimization Priority') }}</h6>
                            <div class="priority-grid">
                                <label class="priority-card distance {{ old('optimization_priority', 'distance') == 'distance' ? 'selected' : '' }}">
                                    <input type="radio" name="optimization_priority" value="distance" {{ old('optimization_priority', 'distance') == 'distance' ? 'checked' : '' }}>
                                    <div class="priority-icon"><i class="bi bi-signpost-2"></i></div>
                                    <div class="priority-title">{{ __('Distance') }}</div>
                                    <div class="priority-desc">{{ __('Shortest route') }}</div>
                                </label>
                                <label class="priority-card time {{ old('optimization_priority') == 'time' ? 'selected' : '' }}">
                                    <input type="radio" name="optimization_priority" value="time" {{ old('optimization_priority') == 'time' ? 'checked' : '' }}>
                                    <div class="priority-icon"><i class="bi bi-clock"></i></div>
                                    <div class="priority-title">{{ __('Time') }}</div>
                                    <div class="priority-desc">{{ __('Fastest route') }}</div>
                                </label>
                                <label class="priority-card fuel {{ old('optimization_priority') == 'fuel' ? 'selected' : '' }}">
                                    <input type="radio" name="optimization_priority" value="fuel" {{ old('optimization_priority') == 'fuel' ? 'checked' : '' }}>
                                    <div class="priority-icon"><i class="bi bi-fuel-pump"></i></div>
                                    <div class="priority-title">{{ __('Fuel') }}</div>
                                    <div class="priority-desc">{{ __('Most economical') }}</div>
                                </label>
                            </div>

                            {{-- Constraints --}}
                            <h6 class="section-title"><i class="bi bi-shield-check"></i> {{ __('Route Constraints') }}</h6>
                            <div class="constraint-list">
                                <div class="constraint-item">
                                    <div class="constraint-info">
                                        <div class="constraint-icon"><i class="bi bi-sign-no-parking"></i></div>
                                        <div>
                                            <div class="constraint-label">{{ __('Avoid Highways') }}</div>
                                            <div class="constraint-desc">{{ __('Prefer local roads') }}</div>
                                        </div>
                                    </div>
                                    <label class="toggle-switch">
                                        <input type="checkbox" name="avoid_highways" value="1" {{ old('avoid_highways') ? 'checked' : '' }}>
                                        <span class="toggle-slider"></span>
                                    </label>
                                </div>
                                <div class="constraint-item">
                                    <div class="constraint-info">
                                        <div class="constraint-icon"><i class="bi bi-cash-stack"></i></div>
                                        <div>
                                            <div class="constraint-label">{{ __('Avoid Toll Roads') }}</div>
                                            <div class="constraint-desc">{{ __('Skip toll charges') }}</div>
                                        </div>
                                    </div>
                                    <label class="toggle-switch">
                                        <input type="checkbox" name="avoid_tolls" value="1" {{ old('avoid_tolls') ? 'checked' : '' }}>
                                        <span class="toggle-slider"></span>
                                    </label>
                                </div>
                            </div>

                            {{-- Actions --}}
                            <div style="display: flex; gap: 1rem; margin-top: 1.5rem;">
                                <button type="submit" class="optimizer-btn optimizer-btn-primary" id="optimizeBtn">
                                    <i class="bi bi-magic"></i> {{ __('Optimize Route') }}
                                </button>
                                <a href="{{ route('admin.trips.index') }}" class="optimizer-btn optimizer-btn-secondary">
                                    {{ __('Cancel') }}
                                </a>
                            </div>
                        </form>
                    @endif
                </div>
            </div>
        </div>

        {{-- Sidebar --}}
        <div class="optimizer-sidebar">
            {{-- How It Works --}}
            <div class="sidebar-card">
                <div class="sidebar-header">
                    <div class="sidebar-icon info"><i class="bi bi-lightbulb"></i></div>
                    <h5 class="sidebar-title">{{ __('How It Works') }}</h5>
                </div>
                <div class="sidebar-body">
                    <div class="steps-list">
                        <div class="step-item">
                            <span class="step-number">1</span>
                            <div class="step-content">
                                <h6>{{ __('Select Trip') }}</h6>
                                <p>{{ __('Choose an active trip with locations') }}</p>
                            </div>
                        </div>
                        <div class="step-item">
                            <span class="step-number">2</span>
                            <div class="step-content">
                                <h6>{{ __('AI Analysis') }}</h6>
                                <p>{{ __('AI analyzes using GPS coordinates') }}</p>
                            </div>
                        </div>
                        <div class="step-item">
                            <span class="step-number">3</span>
                            <div class="step-content">
                                <h6>{{ __('Get Results') }}</h6>
                                <p>{{ __('Review and apply optimized route') }}</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            {{-- Benefits --}}
            <div class="sidebar-card">
                <div class="sidebar-header">
                    <div class="sidebar-icon success"><i class="bi bi-graph-up-arrow"></i></div>
                    <h5 class="sidebar-title">{{ __('Benefits') }}</h5>
                </div>
                <div class="sidebar-body">
                    <div class="benefits-list">
                        <div class="benefit-item">
                            <div class="benefit-icon"><i class="bi bi-fuel-pump"></i></div>
                            <span class="benefit-text">{{ __('Reduce fuel costs up to 20%') }}</span>
                        </div>
                        <div class="benefit-item">
                            <div class="benefit-icon"><i class="bi bi-clock-history"></i></div>
                            <span class="benefit-text">{{ __('Save delivery time') }}</span>
                        </div>
                        <div class="benefit-item">
                            <div class="benefit-icon"><i class="bi bi-gear"></i></div>
                            <span class="benefit-text">{{ __('Lower vehicle wear') }}</span>
                        </div>
                    </div>
                </div>
            </div>

            {{-- Requirements --}}
            <div class="sidebar-card">
                <div class="sidebar-header">
                    <div class="sidebar-icon warning"><i class="bi bi-exclamation-triangle"></i></div>
                    <h5 class="sidebar-title">{{ __('Requirements') }}</h5>
                </div>
                <div class="sidebar-body">
                    <div class="requirements-list">
                        <div class="requirement-item"><i class="bi bi-circle-fill"></i> {{ __('Trip: Assigned or In Transit') }}</div>
                        <div class="requirement-item"><i class="bi bi-circle-fill"></i> {{ __('Has pickup and drop locations') }}</div>
                        <div class="requirement-item"><i class="bi bi-circle-fill"></i> {{ __('GPS coordinates recommended') }}</div>
                        <div class="requirement-item"><i class="bi bi-circle-fill"></i> {{ __('OpenAI API configured') }}</div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    document.querySelectorAll('.trip-item').forEach(item => {
        item.addEventListener('click', function() {
            document.querySelectorAll('.trip-item').forEach(i => i.classList.remove('selected'));
            this.classList.add('selected');
        });
    });
    
    document.querySelectorAll('.priority-card').forEach(card => {
        card.addEventListener('click', function() {
            document.querySelectorAll('.priority-card').forEach(c => c.classList.remove('selected'));
            this.classList.add('selected');
        });
    });
    
    document.getElementById('optimizeForm')?.addEventListener('submit', function() {
        const btn = document.getElementById('optimizeBtn');
        btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>{{ __("Optimizing...") }}';
        btn.disabled = true;
    });
});
</script>
@endpush