@extends('layouts.app')

@section('title', __('Optimized Route') . ' - ' . __('app_name'))

@push('styles')
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
<link rel="stylesheet" href="https://unpkg.com/leaflet-routing-machine@3.2.12/dist/leaflet-routing-machine.css" />
<link rel="stylesheet" href="{{ asset('css/adminlistpage.css') }}">
<style>
/* ===== ROUTE RESULT - THEME COMPATIBLE ===== */
.result-container { padding: 0 0 40px; }

/* Back Navigation */
.result-nav {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 24px;
}
.result-breadcrumb {
    display: flex;
    align-items: center;
    gap: 10px;
    font-size: 14px;
}
.result-breadcrumb a { 
    color: var(--primary-color); 
    text-decoration: none; 
    font-weight: 500; 
}
.result-breadcrumb a:hover { text-decoration: underline; }
.result-breadcrumb i { color: var(--text-muted); font-size: 10px; }
.result-breadcrumb span { color: var(--text-secondary); }
.back-btn {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 10px 20px;
    background: var(--card-bg);
    border: 2px solid var(--border-color);
    border-radius: 10px;
    color: var(--text-secondary);
    font-weight: 600;
    font-size: 14px;
    text-decoration: none;
    transition: all 0.2s ease;
}
.back-btn:hover { 
    border-color: var(--primary-color); 
    color: var(--primary-color); 
    background: var(--body-bg); 
}

/* Hero Section */
.result-hero {
    background: linear-gradient(135deg, var(--success-color) 0%, #059669 50%, #047857 100%);
    border-radius: 20px;
    padding: 2rem 2.5rem;
    margin-bottom: 1.5rem;
    position: relative;
    overflow: hidden;
    box-shadow: 0 10px 40px rgba(16, 185, 129, 0.25);
}
.result-hero::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -20%;
    width: 400px;
    height: 400px;
    background: radial-gradient(circle, rgba(255,255,255,0.15) 0%, transparent 70%);
}
.result-hero-content { 
    position: relative; 
    z-index: 2; 
    display: flex; 
    justify-content: space-between; 
    align-items: center; 
    flex-wrap: wrap; 
    gap: 1rem; 
}
.result-hero-left h1 {
    font-size: 1.75rem;
    font-weight: 700;
    color: white;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}
.result-hero-left h1 i {
    width: 50px;
    height: 50px;
    background: rgba(255,255,255,0.2);
    border-radius: 14px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
}
.result-hero-left p { 
    color: rgba(255,255,255,0.9); 
    font-size: 0.95rem; 
    margin: 0; 
}
.result-hero-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    background: rgba(255,255,255,0.2);
    backdrop-filter: blur(10px);
    padding: 0.625rem 1.25rem;
    border-radius: 50px;
    color: white;
    font-weight: 600;
    font-size: 0.85rem;
    border: 1px solid rgba(255,255,255,0.3);
}

/* Stats Grid */
.stats-grid { 
    display: grid; 
    grid-template-columns: repeat(4, 1fr); 
    gap: 1.25rem; 
    margin-bottom: 1.5rem; 
}
@media (max-width: 1200px) { .stats-grid { grid-template-columns: repeat(2, 1fr); } }
@media (max-width: 576px) { .stats-grid { grid-template-columns: 1fr; } }

.stat-card {
    background: var(--card-bg);
    border-radius: 16px;
    padding: 1.5rem;
    box-shadow: 0 2px 8px rgba(0,0,0,0.04);
    border: 1px solid var(--border-color);
    position: relative;
    overflow: hidden;
}
[data-theme="dark"] .stat-card { box-shadow: 0 2px 8px rgba(0,0,0,0.2); }

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
}
.stat-card.distance::before { background: var(--info-color); }
.stat-card.duration::before { background: var(--success-color); }
.stat-card.fuel::before { background: var(--warning-color); }
.stat-card.stops::before { background: var(--primary-color); }

.stat-icon {
    width: 50px;
    height: 50px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.35rem;
    margin-bottom: 1rem;
}
.stat-card.distance .stat-icon { background: var(--info-light); color: var(--info-color); }
.stat-card.duration .stat-icon { background: var(--success-light); color: var(--success-color); }
.stat-card.fuel .stat-icon { background: var(--warning-light); color: #92400e; }
[data-theme="dark"] .stat-card.fuel .stat-icon { color: var(--warning-color); }
.stat-card.stops .stat-icon { background: var(--primary-light); color: var(--primary-color); }

.stat-label { 
    font-size: 0.75rem; 
    color: var(--text-muted); 
    font-weight: 500; 
    margin-bottom: 0.25rem; 
    text-transform: uppercase; 
    letter-spacing: 0.5px; 
}
.stat-value { 
    font-size: 1.75rem; 
    font-weight: 700; 
    color: var(--text-primary); 
    line-height: 1.2; 
}
.stat-value span { 
    font-size: 0.9rem; 
    font-weight: 600; 
    color: var(--text-muted); 
}

/* Savings Badge */
.savings-badge {
    position: absolute;
    top: 1rem;
    right: 1rem;
    background: var(--success-color);
    color: white;
    padding: 0.3rem 0.6rem;
    border-radius: 6px;
    font-size: 0.7rem;
    font-weight: 700;
}

/* Main Layout */
.result-layout { 
    display: grid; 
    grid-template-columns: 1fr 400px; 
    gap: 1.5rem; 
}
@media (max-width: 1200px) { .result-layout { grid-template-columns: 1fr; } }

/* Map Card */
.map-card {
    background: var(--card-bg);
    border-radius: 16px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.04);
    border: 1px solid var(--border-color);
    overflow: hidden;
}
[data-theme="dark"] .map-card { box-shadow: 0 2px 8px rgba(0,0,0,0.2); }

.map-header {
    padding: 1.25rem 1.5rem;
    border-bottom: 1px solid var(--border-color);
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1rem;
}
.map-header-left { display: flex; align-items: center; gap: 0.75rem; }
.map-icon {
    width: 44px;
    height: 44px;
    background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.25rem;
    color: white;
}
.map-title { 
    font-size: 1.1rem; 
    font-weight: 600; 
    color: var(--text-primary); 
}

/* Map Legend */
.map-legend {
    display: flex;
    gap: 1rem;
    background: var(--body-bg);
    padding: 0.625rem 1rem;
    border-radius: 10px;
    flex-wrap: wrap;
}
.legend-item { 
    display: flex; 
    align-items: center; 
    gap: 0.5rem; 
    font-size: 0.8rem; 
    color: var(--text-secondary); 
}
.legend-dot {
    width: 12px;
    height: 12px;
    border-radius: 50%;
    flex-shrink: 0;
}
.legend-dot.pickup { background: var(--success-color); }
.legend-dot.delivery { background: var(--info-color); }
.legend-dot.drop { background: var(--danger-color); }

.map-body { position: relative; }
#routeMap { height: 500px; width: 100%; }
.leaflet-routing-container { display: none; }

/* Loading Overlay */
.map-loading {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: var(--card-bg);
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    z-index: 1000;
}
.map-loading-spinner {
    width: 50px;
    height: 50px;
    border: 4px solid var(--border-color);
    border-top-color: var(--primary-color);
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin-bottom: 1rem;
}
@keyframes spin { to { transform: rotate(360deg); } }
.map-loading-text { 
    font-size: 0.95rem; 
    color: var(--text-secondary); 
    font-weight: 500; 
}
.map-loading-status { 
    font-size: 0.8rem; 
    color: var(--text-muted); 
    margin-top: 0.25rem; 
}

/* Route Timeline Card */
.timeline-card {
    background: var(--card-bg);
    border-radius: 16px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.04);
    border: 1px solid var(--border-color);
    overflow: hidden;
}
[data-theme="dark"] .timeline-card { box-shadow: 0 2px 8px rgba(0,0,0,0.2); }

.timeline-header {
    padding: 1.25rem 1.5rem;
    border-bottom: 1px solid var(--border-color);
    display: flex;
    align-items: center;
    gap: 0.75rem;
}
.timeline-icon {
    width: 44px;
    height: 44px;
    background: var(--warning-light);
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.25rem;
    color: #92400e;
}
[data-theme="dark"] .timeline-icon { color: var(--warning-color); }
.timeline-title { 
    font-size: 1.1rem; 
    font-weight: 600; 
    color: var(--text-primary); 
}
.timeline-body { 
    padding: 1.5rem; 
    max-height: 600px; 
    overflow-y: auto; 
}

/* Timeline Items */
.timeline-list { position: relative; }
.timeline-item {
    display: flex;
    gap: 1rem;
    margin-bottom: 1.5rem;
    position: relative;
}
.timeline-item:last-child { margin-bottom: 0; }
.timeline-item::before {
    content: '';
    position: absolute;
    left: 19px;
    top: 44px;
    bottom: -24px;
    width: 2px;
    background: var(--border-color);
}
.timeline-item:last-child::before { display: none; }

.timeline-marker {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.85rem;
    font-weight: 700;
    color: white;
    flex-shrink: 0;
    position: relative;
    z-index: 1;
    box-shadow: 0 2px 8px rgba(0,0,0,0.15);
}
.timeline-marker.pickup { background: var(--success-color); }
.timeline-marker.delivery { background: var(--info-color); }
.timeline-marker.drop { background: var(--danger-color); }

.timeline-content {
    flex: 1;
    background: var(--body-bg);
    border-radius: 12px;
    padding: 1rem 1.25rem;
    border: 1px solid var(--border-color);
}
.timeline-content-header { 
    display: flex; 
    justify-content: space-between; 
    align-items: flex-start; 
    margin-bottom: 0.5rem;
    gap: 0.5rem;
    flex-wrap: wrap;
}
.timeline-name { 
    font-size: 0.9rem; 
    font-weight: 600; 
    color: var(--text-primary); 
}
.timeline-time {
    font-size: 0.7rem;
    font-weight: 600;
    color: var(--primary-color);
    background: var(--primary-light);
    padding: 0.25rem 0.6rem;
    border-radius: 20px;
}
.timeline-address { 
    font-size: 0.8rem; 
    color: var(--text-secondary); 
    line-height: 1.5; 
    margin-bottom: 0.75rem; 
}

.timeline-stats { display: flex; gap: 1rem; flex-wrap: wrap; }
.timeline-stat {
    display: flex;
    align-items: center;
    gap: 0.35rem;
    font-size: 0.75rem;
    color: var(--text-muted);
}
.timeline-stat i { color: var(--text-muted); }
.timeline-stat strong { color: var(--text-primary); font-weight: 600; }

.timeline-instructions {
    margin-top: 0.75rem;
    padding-top: 0.75rem;
    border-top: 1px dashed var(--border-color);
    font-size: 0.8rem;
    color: var(--text-secondary);
    font-style: italic;
    display: flex;
    align-items: flex-start;
    gap: 0.5rem;
}
.timeline-instructions i { color: var(--warning-color); margin-top: 2px; }

/* Action Buttons */
.action-buttons {
    display: flex;
    gap: 1rem;
    padding: 1.25rem 1.5rem;
    border-top: 1px solid var(--border-color);
    background: var(--body-bg);
}
.action-btn {
    flex: 1;
    padding: 0.875rem 1.25rem;
    border-radius: 10px;
    font-size: 0.9rem;
    font-weight: 600;
    border: none;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    transition: all 0.2s ease;
    text-decoration: none;
}
.action-btn-primary {
    background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
    color: white;
    box-shadow: 0 4px 12px rgba(249, 115, 22, 0.25);
}
.action-btn-primary:hover { 
    transform: translateY(-2px); 
    box-shadow: 0 6px 20px rgba(249, 115, 22, 0.35); 
    color: white; 
}
.action-btn-secondary {
    background: var(--card-bg);
    color: var(--text-secondary);
    border: 1px solid var(--border-color);
}
.action-btn-secondary:hover { 
    border-color: var(--primary-color); 
    color: var(--primary-color); 
}

/* AI Notes Card */
.notes-card {
    background: var(--info-light);
    border-radius: 12px;
    padding: 1.25rem;
    margin-top: 1.5rem;
    border: 1px solid var(--info-color);
}
[data-theme="dark"] .notes-card {
    background: rgba(6, 182, 212, 0.1);
    border-color: rgba(6, 182, 212, 0.3);
}
.notes-header { 
    display: flex; 
    align-items: center; 
    gap: 0.75rem; 
    margin-bottom: 0.75rem; 
}
.notes-icon {
    width: 36px;
    height: 36px;
    background: var(--info-color);
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1rem;
    color: white;
}
.notes-title { 
    font-size: 0.95rem; 
    font-weight: 600; 
    color: var(--info-color); 
}
.notes-content { 
    font-size: 0.85rem; 
    color: var(--text-secondary); 
    line-height: 1.6; 
}

/* Scrollbar for timeline */
.timeline-body::-webkit-scrollbar { width: 6px; }
.timeline-body::-webkit-scrollbar-track { background: var(--body-bg); border-radius: 3px; }
.timeline-body::-webkit-scrollbar-thumb { background: var(--border-color); border-radius: 3px; }
.timeline-body::-webkit-scrollbar-thumb:hover { background: var(--text-muted); }

/* Responsive */
@media (max-width: 576px) {
    .result-hero { padding: 1.5rem; }
    .result-hero-left h1 { font-size: 1.35rem; }
    .result-hero-left h1 i { width: 40px; height: 40px; font-size: 1.25rem; }
    .map-header { flex-direction: column; align-items: flex-start; }
    .action-buttons { flex-direction: column; }
}
</style>
@endpush

@section('content')
<div class="result-container">
    {{-- Navigation --}}
    <div class="result-nav">
        <div class="result-breadcrumb">
            <a href="{{ route('route-optimizer.index') }}">{{ __('Route Optimizer') }}</a>
            <i class="bi bi-chevron-right"></i>
            <span>{{ __('Results') }}</span>
        </div>
        <a href="{{ route('route-optimizer.index') }}" class="back-btn">
            <i class="bi bi-arrow-left"></i> {{ __('Back') }}
        </a>
    </div>

    {{-- Hero Section --}}
    <div class="result-hero">
        <div class="result-hero-content">
            <div class="result-hero-left">
                <h1>
                    <i class="bi bi-check-circle"></i>
                    {{ __('Route Optimized!') }}
                </h1>
                <p>{{ $trip->trip_number }} - {{ $trip->vehicle->vehicle_number ?? 'N/A' }}</p>
            </div>
            <div class="result-hero-badge">
                <i class="bi bi-magic"></i>
                {{ __('AI Optimized') }}
            </div>
        </div>
    </div>

    {{-- Stats Grid --}}
    <div class="stats-grid">
        <div class="stat-card distance">
            @if(($optimizedRoute['savings']['distance_saved_percent'] ?? 0) > 0)
                <div class="savings-badge">-{{ number_format($optimizedRoute['savings']['distance_saved_percent'], 0) }}%</div>
            @endif
            <div class="stat-icon"><i class="bi bi-signpost-2"></i></div>
            <div class="stat-label">{{ __('Total Distance') }}</div>
            <div class="stat-value" id="totalDistance">{{ number_format($optimizedRoute['total_distance_km'] ?? 0, 1) }} <span>km</span></div>
        </div>
        <div class="stat-card duration">
            @if(($optimizedRoute['savings']['time_saved_percent'] ?? 0) > 0)
                <div class="savings-badge">-{{ number_format($optimizedRoute['savings']['time_saved_percent'], 0) }}%</div>
            @endif
            <div class="stat-icon"><i class="bi bi-clock"></i></div>
            <div class="stat-label">{{ __('Total Duration') }}</div>
            @php
                $totalMin = $optimizedRoute['total_duration_min'] ?? 0;
                $hours = floor($totalMin / 60);
                $mins = $totalMin % 60;
            @endphp
            <div class="stat-value" id="totalDuration">{{ $hours }}h {{ $mins }}<span>m</span></div>
        </div>
        <div class="stat-card fuel">
            <div class="stat-icon"><i class="bi bi-fuel-pump"></i></div>
            <div class="stat-label">{{ __('Est. Fuel') }}</div>
            <div class="stat-value">{{ number_format($optimizedRoute['estimated_fuel_liters'] ?? 0, 1) }} <span>L</span></div>
        </div>
        <div class="stat-card stops">
            <div class="stat-icon"><i class="bi bi-pin-map"></i></div>
            <div class="stat-label">{{ __('Total Stops') }}</div>
            <div class="stat-value">{{ count($optimizedRoute['optimized_order'] ?? []) }}</div>
        </div>
    </div>

    {{-- Main Layout --}}
    <div class="result-layout">
        {{-- Map Card --}}
        <div class="map-card">
            <div class="map-header">
                <div class="map-header-left">
                    <div class="map-icon"><i class="bi bi-map"></i></div>
                    <h5 class="map-title">{{ __('Route Map') }}</h5>
                </div>
                <div class="map-legend">
                    <div class="legend-item"><span class="legend-dot pickup"></span> {{ __('Pickup') }}</div>
                    <div class="legend-item"><span class="legend-dot delivery"></span> {{ __('Delivery') }}</div>
                    <div class="legend-item"><span class="legend-dot drop"></span> {{ __('Final Drop') }}</div>
                </div>
            </div>
            <div class="map-body">
                <div class="map-loading" id="mapLoading">
                    <div class="map-loading-spinner"></div>
                    <div class="map-loading-text">{{ __('Loading route...') }}</div>
                    <div class="map-loading-status" id="geocodeStatus">0 / {{ count($optimizedRoute['optimized_order'] ?? []) }}</div>
                </div>
                <div id="routeMap"></div>
            </div>
        </div>

        {{-- Timeline Card --}}
        <div class="timeline-card">
            <div class="timeline-header">
                <div class="timeline-icon"><i class="bi bi-list-ol"></i></div>
                <h5 class="timeline-title">{{ __('Optimized Sequence') }}</h5>
            </div>
            <div class="timeline-body">
                <div class="timeline-list">
                    @foreach($optimizedRoute['optimized_order'] ?? [] as $index => $stop)
                        @php
                            $isFirst = $index === 0;
                            $isLast = $index === count($optimizedRoute['optimized_order']) - 1;
                            $markerType = $isFirst ? 'pickup' : ($isLast ? 'drop' : 'delivery');
                        @endphp
                        <div class="timeline-item" id="stop-{{ $index }}">
                            <div class="timeline-marker {{ $markerType }}">
                                @if($isFirst)
                                    <i class="bi bi-geo-alt-fill"></i>
                                @elseif($isLast)
                                    <i class="bi bi-flag-fill"></i>
                                @else
                                    {{ $stop['sequence'] ?? $index + 1 }}
                                @endif
                            </div>
                            <div class="timeline-content">
                                <div class="timeline-content-header">
                                    <span class="timeline-name">{{ $stop['location_name'] ?? __('Location') . ' ' . ($index + 1) }}</span>
                                    <span class="timeline-time">{{ $stop['estimated_arrival'] ?? '--:--' }}</span>
                                </div>
                                <div class="timeline-address">{{ $stop['address'] ?? 'N/A' }}</div>
                                <div class="timeline-stats">
                                    <div class="timeline-stat">
                                        <i class="bi bi-signpost-2"></i>
                                        <strong class="stop-distance">{{ number_format($stop['distance_from_previous_km'] ?? 0, 1) }}</strong> km
                                    </div>
                                    <div class="timeline-stat">
                                        <i class="bi bi-clock"></i>
                                        <strong class="stop-duration">{{ $stop['duration_from_previous_min'] ?? 0 }}</strong> min
                                    </div>
                                </div>
                                @if(!empty($stop['instructions']))
                                    <div class="timeline-instructions">
                                        <i class="bi bi-info-circle"></i>
                                        <span>{{ $stop['instructions'] }}</span>
                                    </div>
                                @endif
                            </div>
                        </div>
                    @endforeach
                </div>

                {{-- AI Notes --}}
                @if(!empty($optimizedRoute['optimization_notes']))
                    <div class="notes-card">
                        <div class="notes-header">
                            <div class="notes-icon"><i class="bi bi-robot"></i></div>
                            <span class="notes-title">{{ __('AI Insights') }}</span>
                        </div>
                        <div class="notes-content">{{ $optimizedRoute['optimization_notes'] }}</div>
                    </div>
                @endif
            </div>
            
            {{-- Action Buttons --}}
            <div class="action-buttons">
                <form action="{{ route('route-optimizer.apply') }}" method="POST" style="flex: 1; display: flex;">
                    @csrf
                    <input type="hidden" name="trip_id" value="{{ $trip->id }}">
                    <input type="hidden" name="optimized_order" value="{{ json_encode($optimizedRoute['optimized_order'] ?? []) }}">
                    <button type="submit" class="action-btn action-btn-primary" style="flex: 1;">
                        <i class="bi bi-check-circle"></i> {{ __('Apply Route') }}
                    </button>
                </form>
                <a href="{{ route('admin.trips.show', $trip->id) }}" class="action-btn action-btn-secondary">
                    <i class="bi bi-eye"></i> {{ __('View Trip') }}
                </a>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
<script src="https://unpkg.com/leaflet-routing-machine@3.2.12/dist/leaflet-routing-machine.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const map = L.map('routeMap').setView([13.0827, 80.2707], 10);
    
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors'
    }).addTo(map);
    
    const stops = @json($optimizedRoute['optimized_order'] ?? []);
    
    function createMarkerIcon(color, content, isIcon = false) {
        return L.divIcon({
            className: 'custom-marker',
            html: `<div style="
                background: ${color};
                width: 32px;
                height: 32px;
                border-radius: 50%;
                border: 3px solid white;
                box-shadow: 0 4px 12px rgba(0,0,0,0.3);
                display: flex;
                align-items: center;
                justify-content: center;
                color: white;
                font-weight: bold;
                font-size: 12px;
            ">${isIcon ? '<i class="bi ' + content + '"></i>' : content}</div>`,
            iconSize: [32, 32],
            iconAnchor: [16, 16],
            popupAnchor: [0, -16]
        });
    }
    
    async function geocodeAddress(address) {
        try {
            const response = await fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(address)}&limit=1`, {
                headers: { 'User-Agent': 'LogiFlow Fleet Management' }
            });
            const data = await response.json();
            if (data && data.length > 0) {
                return { lat: parseFloat(data[0].lat), lng: parseFloat(data[0].lon) };
            }
        } catch (error) { console.error('Geocoding error:', error); }
        return null;
    }
    
    async function processRoute() {
        const waypoints = [];
        const markers = [];
        
        for (let i = 0; i < stops.length; i++) {
            const stop = stops[i];
            document.getElementById('geocodeStatus').textContent = `${i + 1} / ${stops.length}`;
            
            // Use provided coordinates if available
            let coords = null;
            if (stop.latitude && stop.longitude && stop.latitude !== 0 && stop.longitude !== 0) {
                coords = { lat: stop.latitude, lng: stop.longitude };
            } else {
                coords = await geocodeAddress(stop.address);
                await new Promise(resolve => setTimeout(resolve, 1100));
            }
            
            if (coords) {
                waypoints.push(L.latLng(coords.lat, coords.lng));
                
                let markerIcon;
                const isFirst = i === 0;
                const isLast = i === stops.length - 1;
                
                // Using theme colors
                if (isFirst) {
                    markerIcon = createMarkerIcon('#10b981', 'bi-geo-alt-fill', true); // success
                } else if (isLast) {
                    markerIcon = createMarkerIcon('#ef4444', 'bi-flag-fill', true); // danger
                } else {
                    markerIcon = createMarkerIcon('#06b6d4', i.toString()); // info
                }
                
                const marker = L.marker([coords.lat, coords.lng], { icon: markerIcon })
                    .addTo(map)
                    .bindPopup(`<strong>${stop.location_name}</strong><br><small>${stop.address}</small>`);
                markers.push(marker);
            }
        }
        
        document.getElementById('mapLoading').style.display = 'none';
        
        if (waypoints.length < 2) {
            if (waypoints.length === 1) map.setView(waypoints[0], 13);
            return;
        }
        
        const routingControl = L.Routing.control({
            waypoints: waypoints,
            routeWhileDragging: false,
            addWaypoints: false,
            draggableWaypoints: false,
            fitSelectedRoutes: true,
            showAlternatives: false,
            lineOptions: {
                styles: [
                    { color: '#f97316', opacity: 0.8, weight: 6 }, // primary-color
                    { color: '#ea580c', opacity: 1, weight: 2 }   // primary-dark
                ]
            },
            createMarker: function() { return null; },
            router: L.Routing.osrmv1({
                serviceUrl: 'https://router.project-osrm.org/route/v1',
                profile: 'driving'
            })
        }).addTo(map);
        
        routingControl.on('routesfound', function(e) {
            const route = e.routes[0];
            if (route) {
                const km = (route.summary.totalDistance / 1000).toFixed(1);
                const mins = Math.round(route.summary.totalTime / 60);
                const h = Math.floor(mins / 60);
                const m = mins % 60;
                
                document.getElementById('totalDistance').innerHTML = km + ' <span>km</span>';
                document.getElementById('totalDuration').innerHTML = h + 'h ' + m + '<span>m</span>';
            }
        });
        
        routingControl.on('routingerror', function() {
            if (waypoints.length >= 2) {
                L.polyline(waypoints, { color: '#f97316', weight: 4, opacity: 0.7, dashArray: '10, 10' }).addTo(map);
                map.fitBounds(L.latLngBounds(waypoints).pad(0.1));
            }
        });
    }
    
    processRoute();
});
</script>
@endpush