@extends('layouts.app')

@section('title', __('add') . ' ' . __('costs') . ' - ' . __('app_name'))

@push('styles')
<link rel="stylesheet" href="{{ asset('css/admincreatepage.css') }}">
@endpush

@section('content')
<div class="create-page-container">
    {{-- Page Header --}}
    <div class="create-page-header">
        <div class="create-page-header-content">
            <div class="create-page-breadcrumb">
                <a href="{{ route('admin.dashboard') }}">{{ __('dashboard') }}</a>
                <i class="bi bi-chevron-right"></i>
                <a href="{{ route('admin.costs.index') }}">{{ __('costs') }}</a>
                <i class="bi bi-chevron-right"></i>
                <span>{{ __('add') }}</span>
            </div>
            <h1 class="create-page-title">
                <div class="create-page-title-icon">
                    <i class="bi bi-plus-lg"></i>
                </div>
                {{ __('add') }} {{ __('costs') }}
            </h1>
            <p class="create-page-subtitle">{{ __('Add cost record for completed trip') }}</p>
        </div>
        <div class="create-page-actions">
            <a href="{{ route('admin.costs.index') }}" class="create-btn create-btn-secondary">
                <i class="bi bi-arrow-left"></i>
                {{ __('back') }}
            </a>
        </div>
    </div>

    {{-- No Trips Warning --}}
    @if($trips->isEmpty())
    <div class="create-alert danger">
        <div class="create-alert-icon">
            <i class="bi bi-exclamation-triangle"></i>
        </div>
        <div class="create-alert-content">
            <div class="create-alert-title">{{ __('no_trips_available') }}</div>
            <div class="create-alert-text">
                To add costs, you need completed trips (with "delivered" status) that don't have cost records yet. 
                Please ensure deliveries are completed before adding costs.
            </div>
        </div>
    </div>
    @endif

    {{-- Validation Errors --}}
    @if($errors->any())
        <div class="create-alert danger">
            <div class="create-alert-icon">
                <i class="bi bi-exclamation-triangle"></i>
            </div>
            <div class="create-alert-content">
                <div class="create-alert-title">{{ __('Please fix the following errors') }}</div>
                <div class="create-alert-text">
                    <ul>
                        @foreach($errors->all() as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                </div>
            </div>
        </div>
    @endif

    <form action="{{ route('admin.costs.store') }}" method="POST">
        @csrf

        <div class="create-form-layout">
            {{-- Main Form --}}
            <div class="create-form-main">
                {{-- Trip Selection Card --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-header-icon primary">
                            <i class="bi bi-geo-alt"></i>
                        </div>
                        <div class="create-card-header-content">
                            <h5 class="create-card-title">{{ __('trip') }} {{ __('selection') }}</h5>
                            <p class="create-card-subtitle">{{ __('Select a completed trip to add costs') }}</p>
                        </div>
                    </div>
                    <div class="create-card-body">
                        <div class="create-form-group">
                            <label class="create-form-label">
                                <i class="bi bi-truck"></i>
                                {{ __('select_trip') }}
                                <span class="create-form-required">*</span>
                            </label>
                            <select class="create-select @error('trip_id') is-invalid @enderror" 
                                    id="trip_id" 
                                    name="trip_id" 
                                    required
                                    {{ $trips->isEmpty() ? 'disabled' : '' }}
                                    onchange="updateTripInfo()">
                                <option value="">{{ __('select_option') }}</option>
                                @foreach($trips as $trip)
                                <option value="{{ $trip->id }}" 
                                        data-trip-number="{{ $trip->trip_number }}"
                                        data-vehicle="{{ $trip->vehicle->vehicle_number ?? 'N/A' }}"
                                        data-vehicle-type="{{ $trip->vehicle->vehicleType->name ?? 'N/A' }}"
                                        data-base-price="{{ $trip->vehicle->vehicleType->base_price ?? 0 }}"
                                        data-driver="{{ $trip->driver->name ?? 'N/A' }}"
                                        data-pickup="{{ $trip->pickup_location }}"
                                        data-drop="{{ $trip->drop_location ?? 'N/A' }}"
                                        data-end-date="{{ $trip->end_date ? \Carbon\Carbon::parse($trip->end_date)->format('M d, Y H:i') : 'N/A' }}"
                                        data-shipment="{{ $trip->shipment_reference ?? 'N/A' }}"
                                        {{ old('trip_id') == $trip->id ? 'selected' : '' }}>
                                    {{ $trip->trip_number }} - 
                                    {{ $trip->vehicle->vehicle_number ?? 'N/A' }} - 
                                    {{ $trip->driver->name ?? 'N/A' }} -
                                    Completed: {{ $trip->end_date ? \Carbon\Carbon::parse($trip->end_date)->format('M d, Y') : 'N/A' }}
                                </option>
                                @endforeach
                            </select>
                            @error('trip_id')
                                <div class="create-form-error">
                                    <i class="bi bi-exclamation-circle"></i>
                                    {{ $message }}
                                </div>
                            @enderror
                        </div>

                        {{-- Trip Info Display --}}
                        <div id="trip-info" class="alert alert-info d-none mt-3">
                            <h6 class="fw-bold mb-2">{{ __('trip_details') }}:</h6>
                            <div class="row g-2">
                                <div class="col-md-6">
                                    <small class="text-muted">{{ __('trip_number') }}:</small><br>
                                    <strong id="info-trip-number">-</strong>
                                </div>
                                <div class="col-md-6">
                                    <small class="text-muted">{{ __('vehicle') }}:</small><br>
                                    <strong id="info-vehicle">-</strong>
                                </div>
                                <div class="col-md-6">
                                    <small class="text-muted">{{ __('vehicle_type') }}:</small><br>
                                    <strong id="info-vehicle-type">-</strong>
                                </div>
                                <div class="col-md-6">
                                    <small class="text-muted">{{ __('driver') }}:</small><br>
                                    <strong id="info-driver">-</strong>
                                </div>
                                <div class="col-md-6">
                                    <small class="text-muted">{{ __('shipment') }}:</small><br>
                                    <strong id="info-shipment">-</strong>
                                </div>
                                <div class="col-md-6">
                                    <small class="text-muted">{{ __('base_price') }} ({{ __('vehicle_type') }}):</small><br>
                                    <strong class="text-success" id="info-base-price">$0.00</strong>
                                </div>
                                <div class="col-md-6">
                                    <small class="text-muted">{{ __('pickup_location') }}:</small><br>
                                    <strong id="info-pickup">-</strong>
                                </div>
                                <div class="col-md-6">
                                    <small class="text-muted">{{ __('drop_location') }}:</small><br>
                                    <strong id="info-drop">-</strong>
                                </div>
                                <div class="col-md-12">
                                    <small class="text-muted">{{ __('completed_at') }}:</small><br>
                                    <strong id="info-end-date">-</strong>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                {{-- Cost Details Card --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-header-icon success">
                            <i class="bi bi-cash-stack"></i>
                        </div>
                        <div class="create-card-header-content">
                            <h5 class="create-card-title">{{ __('cost') }} {{ __('details') }}</h5>
                            <p class="create-card-subtitle">{{ __('Enter all trip related costs') }}</p>
                        </div>
                    </div>
                    <div class="create-card-body">
                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-cash"></i>
                                    {{ __('base_cost') }} (USD)
                                    <span class="create-form-required">*</span>
                                </label>
                                <div class="create-input-group">
                                    <span class="create-input-addon create-input-addon-left">$</span>
                                    <input type="number" 
                                           class="create-input @error('base_cost') is-invalid @enderror" 
                                           id="base_cost" 
                                           name="base_cost" 
                                           step="0.01" 
                                           min="0"
                                           value="{{ old('base_cost', '0.00') }}"
                                           required
                                           onchange="calculateTotal()">
                                </div>
                                @error('base_cost')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                                <p class="create-form-hint">
                                    <i class="bi bi-lightning-fill text-success"></i>
                                    Auto-populated from vehicle type (editable)
                                </p>
                            </div>

                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-signpost-split"></i>
                                    {{ __('toll_cost') }} (USD)
                                    <span class="create-form-required">*</span>
                                </label>
                                <div class="create-input-group">
                                    <span class="create-input-addon create-input-addon-left">$</span>
                                    <input type="number" 
                                           class="create-input @error('toll_cost') is-invalid @enderror" 
                                           id="toll_cost" 
                                           name="toll_cost" 
                                           step="0.01" 
                                           min="0"
                                           value="{{ old('toll_cost', '0.00') }}"
                                           required
                                           onchange="calculateTotal()">
                                </div>
                                @error('toll_cost')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                                <p class="create-form-hint">
                                    <i class="bi bi-lightbulb"></i>
                                    Toll charges during the trip
                                </p>
                            </div>
                        </div>

                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-person-badge"></i>
                                    {{ __('driver_allowance') }} (USD)
                                    <span class="create-form-required">*</span>
                                </label>
                                <div class="create-input-group">
                                    <span class="create-input-addon create-input-addon-left">$</span>
                                    <input type="number" 
                                           class="create-input @error('driver_allowance') is-invalid @enderror" 
                                           id="driver_allowance" 
                                           name="driver_allowance" 
                                           step="0.01" 
                                           min="0"
                                           value="{{ old('driver_allowance', '0.00') }}"
                                           required
                                           onchange="calculateTotal()">
                                </div>
                                @error('driver_allowance')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                                <p class="create-form-hint">
                                    <i class="bi bi-lightbulb"></i>
                                    Daily allowance for driver
                                </p>
                            </div>

                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-fuel-pump"></i>
                                    {{ __('fuel_cost') }} (USD)
                                    <span class="create-form-optional">({{ __('Optional') }})</span>
                                </label>
                                <div class="create-input-group">
                                    <span class="create-input-addon create-input-addon-left">$</span>
                                    <input type="number" 
                                           class="create-input @error('fuel_cost') is-invalid @enderror" 
                                           id="fuel_cost" 
                                           name="fuel_cost" 
                                           step="0.01" 
                                           min="0"
                                           value="{{ old('fuel_cost', '0.00') }}"
                                           onchange="calculateTotal()">
                                </div>
                                @error('fuel_cost')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                                <p class="create-form-hint">
                                    <i class="bi bi-lightbulb"></i>
                                    Fuel expenses
                                </p>
                            </div>
                        </div>

                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-three-dots"></i>
                                    {{ __('other_costs') }} (USD)
                                    <span class="create-form-optional">({{ __('Optional') }})</span>
                                </label>
                                <div class="create-input-group">
                                    <span class="create-input-addon create-input-addon-left">$</span>
                                    <input type="number" 
                                           class="create-input @error('other_costs') is-invalid @enderror" 
                                           id="other_costs" 
                                           name="other_costs" 
                                           step="0.01" 
                                           min="0"
                                           value="{{ old('other_costs', '0.00') }}"
                                           onchange="calculateTotal()">
                                </div>
                                @error('other_costs')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                                <p class="create-form-hint">
                                    <i class="bi bi-lightbulb"></i>
                                    Miscellaneous expenses
                                </p>
                            </div>

                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-calculator"></i>
                                    {{ __('total_cost') }} (USD)
                                </label>
                                <div class="create-input-group">
                                    <span class="create-input-addon create-input-addon-left bg-success text-white">$</span>
                                    <input type="text" 
                                           class="create-input fw-bold text-success" 
                                           id="total_cost_display" 
                                           value="0.00"
                                           readonly>
                                </div>
                                <p class="create-form-hint">
                                    <i class="bi bi-info-circle"></i>
                                    Auto-calculated total
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                {{-- Notes Card --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-header-icon info">
                            <i class="bi bi-journal-text"></i>
                        </div>
                        <div class="create-card-header-content">
                            <h5 class="create-card-title">{{ __('notes') }}</h5>
                            <p class="create-card-subtitle">{{ __('Additional remarks') }}</p>
                        </div>
                    </div>
                    <div class="create-card-body">
                        <div class="create-form-group">
                            <label class="create-form-label">
                                <i class="bi bi-pencil"></i>
                                {{ __('notes') }}
                                <span class="create-form-optional">({{ __('Optional') }})</span>
                            </label>
                            <textarea class="create-textarea @error('notes') is-invalid @enderror" 
                                      id="notes" 
                                      name="notes" 
                                      rows="3"
                                      placeholder="Additional cost notes or remarks...">{{ old('notes') }}</textarea>
                            @error('notes')
                                <div class="create-form-error">
                                    <i class="bi bi-exclamation-circle"></i>
                                    {{ $message }}
                                </div>
                            @enderror
                        </div>
                    </div>

                    {{-- Form Actions --}}
                    <div class="create-card-actions">
                        <div class="create-card-actions-left">
                            <a href="{{ route('admin.costs.index') }}" class="create-btn create-btn-secondary">
                                <i class="bi bi-x-circle"></i>
                                {{ __('cancel') }}
                            </a>
                        </div>
                        <div class="create-card-actions-right">
                            <button type="submit" class="create-btn create-btn-primary" {{ $trips->isEmpty() ? 'disabled' : '' }}>
                                <i class="bi bi-check-circle"></i>
                                {{ __('save') }} {{ __('costs') }}
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            {{-- Sidebar --}}
            <div class="create-form-sidebar">
                {{-- Tips Card --}}
                <div class="create-sidebar-card">
                    <div class="create-sidebar-header">
                        <div class="create-sidebar-icon info">
                            <i class="bi bi-lightbulb"></i>
                        </div>
                        <h5 class="create-sidebar-title">{{ __('Quick Tips') }}</h5>
                    </div>
                    <div class="create-sidebar-body">
                        <ul class="create-tips-list">
                            <li>
                                <i class="bi bi-check-circle-fill"></i>
                                <span>Base cost is auto-filled from vehicle type</span>
                            </li>
                            <li>
                                <i class="bi bi-check-circle-fill"></i>
                                <span>All required costs must be entered</span>
                            </li>
                            <li>
                                <i class="bi bi-check-circle-fill"></i>
                                <span>Total cost is calculated automatically</span>
                            </li>
                            <li>
                                <i class="bi bi-check-circle-fill"></i>
                                <span>Only delivered trips can have costs</span>
                            </li>
                        </ul>
                    </div>
                </div>

                {{-- Cost Categories Card --}}
                <div class="create-sidebar-card">
                    <div class="create-sidebar-header">
                        <div class="create-sidebar-icon warning">
                            <i class="bi bi-tags"></i>
                        </div>
                        <h5 class="create-sidebar-title">{{ __('Cost Categories') }}</h5>
                    </div>
                    <div class="create-sidebar-body">
                        <ul class="create-tips-list">
                            <li>
                                <i class="bi bi-truck text-primary"></i>
                                <span><strong>Base Cost:</strong> Transportation charges</span>
                            </li>
                            <li>
                                <i class="bi bi-signpost-split text-warning"></i>
                                <span><strong>Toll Cost:</strong> Highway & road tolls</span>
                            </li>
                            <li>
                                <i class="bi bi-person-badge text-info"></i>
                                <span><strong>Driver Allowance:</strong> Daily driver payment</span>
                            </li>
                            <li>
                                <i class="bi bi-fuel-pump text-danger"></i>
                                <span><strong>Fuel Cost:</strong> Fuel expenses</span>
                            </li>
                            <li>
                                <i class="bi bi-three-dots text-secondary"></i>
                                <span><strong>Other Costs:</strong> Miscellaneous</span>
                            </li>
                        </ul>
                    </div>
                </div>

                {{-- Quick Actions Card --}}
                <div class="create-sidebar-card">
                    <div class="create-sidebar-header">
                        <div class="create-sidebar-icon primary">
                            <i class="bi bi-lightning"></i>
                        </div>
                        <h5 class="create-sidebar-title">{{ __('Quick Actions') }}</h5>
                    </div>
                    <div class="create-sidebar-body">
                        <div class="create-quick-actions">
                            <a href="{{ route('admin.costs.index') }}" class="create-quick-action-btn">
                                <i class="bi bi-list-ul"></i>
                                <span>{{ __('View All') }} {{ __('costs') }}</span>
                            </a>
                            <a href="{{ route('admin.trips.index') }}" class="create-quick-action-btn">
                                <i class="bi bi-geo-alt"></i>
                                <span>{{ __('Manage') }} {{ __('trips') }}</span>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </form>
</div>

<script>
    // Calculate total cost
    function calculateTotal() {
        const baseCost = parseFloat(document.getElementById('base_cost').value) || 0;
        const tollCost = parseFloat(document.getElementById('toll_cost').value) || 0;
        const driverAllowance = parseFloat(document.getElementById('driver_allowance').value) || 0;
        const fuelCost = parseFloat(document.getElementById('fuel_cost').value) || 0;
        const otherCosts = parseFloat(document.getElementById('other_costs').value) || 0;

        const total = baseCost + tollCost + driverAllowance + fuelCost + otherCosts;
        document.getElementById('total_cost_display').value = total.toFixed(2);
    }

    // Update trip info display and auto-populate base cost
    function updateTripInfo() {
        const select = document.getElementById('trip_id');
        const selectedOption = select.options[select.selectedIndex];
        const infoDiv = document.getElementById('trip-info');

        if (selectedOption.value) {
            // Update trip details display
            document.getElementById('info-trip-number').textContent = selectedOption.dataset.tripNumber || '-';
            document.getElementById('info-vehicle').textContent = selectedOption.dataset.vehicle || '-';
            document.getElementById('info-vehicle-type').textContent = selectedOption.dataset.vehicleType || '-';
            document.getElementById('info-driver').textContent = selectedOption.dataset.driver || '-';
            document.getElementById('info-shipment').textContent = selectedOption.dataset.shipment || '-';
            document.getElementById('info-pickup').textContent = selectedOption.dataset.pickup || '-';
            document.getElementById('info-drop').textContent = selectedOption.dataset.drop || '-';
            document.getElementById('info-end-date').textContent = selectedOption.dataset.endDate || '-';
            
            // Get base price from vehicle type
            const basePrice = parseFloat(selectedOption.dataset.basePrice) || 0;
            document.getElementById('info-base-price').textContent = '$' + basePrice.toFixed(2);
            
            // AUTO-POPULATE base cost from vehicle type base price
            document.getElementById('base_cost').value = basePrice.toFixed(2);
            
            // Highlight base cost field to show it was auto-populated
            const baseCostField = document.getElementById('base_cost');
            baseCostField.classList.add('border-success');
            setTimeout(() => {
                baseCostField.classList.remove('border-success');
            }, 2000);
            
            // Recalculate total with new base cost
            calculateTotal();
            
            infoDiv.classList.remove('d-none');
        } else {
            infoDiv.classList.add('d-none');
            // Reset base cost if no trip selected
            document.getElementById('base_cost').value = '0.00';
            calculateTotal();
        }
    }

    // Initialize on page load
    document.addEventListener('DOMContentLoaded', function() {
        calculateTotal();
        updateTripInfo();
        
        // Initialize Bootstrap tooltips
        var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    });
</script>
@endsection