@extends('layouts.app')

@section('title', __('deliveries') . ' - ' . __('app_name'))

@push('styles')
<link rel="stylesheet" href="{{ asset('css/adminlistpage.css') }}">
@endpush

@section('content')
<div class="container-fluid">
    {{-- Page Header --}}
    <div class="list-page-header">
        <div class="list-page-header-left">
            <h1 class="list-page-title">
                <i class="bi bi-clipboard-check"></i>
                {{ __('deliveries') }}
            </h1>
            <p class="list-page-subtitle">{{ __('Manage and track all delivery records') }}</p>
        </div>
        <div class="list-page-header-actions">
            @if(auth()->user()->hasPermission('deliveries.create'))
            <a href="{{ route('admin.deliveries.create') }}" class="btn-create">
                <i class="bi bi-plus-lg"></i>
                {{ __('add_delivery') }}
            </a>
            @endif
        </div>
    </div>

    {{-- Flash Messages --}}
    @if(session('success'))
        <div class="alert alert-success alert-dismissible fade show mb-3" role="alert">
            <i class="bi bi-check-circle me-2"></i>{{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    @if(session('error'))
        <div class="alert alert-danger alert-dismissible fade show mb-3" role="alert">
            <i class="bi bi-exclamation-triangle me-2"></i>{{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    {{-- Stats Bar --}}
    <div class="list-stats-bar">
        <div class="list-stat-card gradient-purple">
            <div class="list-stat-icon">
                <i class="bi bi-clipboard-check"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $stats['total'] }}</div>
                <div class="list-stat-label">{{ __('total_deliveries') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-box-seam"></i>
                {{ __('All Time') }}
            </div>
        </div>

        <div class="list-stat-card gradient-blue">
            <div class="list-stat-icon">
                <i class="bi bi-calendar-check"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $stats['this_month'] }}</div>
                <div class="list-stat-label">{{ __('this_month') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-calendar"></i>
                {{ now()->format('F Y') }}
            </div>
        </div>

        <div class="list-stat-card gradient-green">
            <div class="list-stat-icon">
                <i class="bi bi-check-circle"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $stats['completed'] }}</div>
                <div class="list-stat-label">{{ __('completed') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-check2-all"></i>
                {{ __('Successful') }}
            </div>
        </div>

        <div class="list-stat-card gradient-orange">
            <div class="list-stat-icon">
                <i class="bi bi-clock"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $stats['pending'] }}</div>
                <div class="list-stat-label">{{ __('pending') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-hourglass-split"></i>
                {{ __('Awaiting') }}
            </div>
        </div>
    </div>

    {{-- Filter Bar --}}
    <div class="list-filter-bar">
        <div class="list-filter-header">
            <div class="list-filter-title">
                <i class="bi bi-funnel"></i>
                {{ __('Filters') }}
            </div>
            @if(request()->hasAny(['search', 'status', 'from_date', 'to_date']))
                <a href="{{ route('admin.deliveries.index') }}" class="list-filter-clear">
                    <i class="bi bi-x-circle"></i>
                    {{ __('Clear Filters') }}
                </a>
            @endif
        </div>
        <form action="{{ route('admin.deliveries.index') }}" method="GET">
            <div class="list-filter-row">
                <div class="list-filter-group search-group">
                    <label>{{ __('search') }}</label>
                    <div class="list-search-box">
                        <i class="bi bi-search"></i>
                        <input type="text" name="search" class="form-control" 
                               placeholder="{{ __('Search customer, phone, trip...') }}"
                               value="{{ request('search') }}">
                    </div>
                </div>
                <div class="list-filter-group">
                    <label>{{ __('status') }}</label>
                    <select name="status" class="form-select">
                        <option value="">{{ __('all_statuses') }}</option>
                        <option value="pending" {{ request('status') === 'pending' ? 'selected' : '' }}>{{ __('pending') }}</option>
                        <option value="partial" {{ request('status') === 'partial' ? 'selected' : '' }}>{{ __('partial') }}</option>
                        <option value="completed" {{ request('status') === 'completed' ? 'selected' : '' }}>{{ __('completed') }}</option>
                        <option value="failed" {{ request('status') === 'failed' ? 'selected' : '' }}>{{ __('failed') }}</option>
                    </select>
                </div>
                <div class="list-filter-group">
                    <label>{{ __('from_date') }}</label>
                    <input type="date" name="from_date" class="form-control" 
                           value="{{ request('from_date') }}">
                </div>
                <div class="list-filter-group">
                    <label>{{ __('to_date') }}</label>
                    <input type="date" name="to_date" class="form-control" 
                           value="{{ request('to_date') }}">
                </div>
                <div class="list-filter-group">
                    <label>&nbsp;</label>
                    <button type="submit" class="btn-filter">
                        <i class="bi bi-funnel"></i>
                        {{ __('filter') }}
                    </button>
                </div>
            </div>
        </form>
    </div>

    {{-- Data Table --}}
    <div class="list-table-container">
        <div class="list-table-header">
            <h5 class="list-table-title">
                <i class="bi bi-table"></i>
                {{ __('delivery_list') }}
                <span class="list-table-count">({{ $deliveries->total() }} {{ __('records') }})</span>
            </h5>
        </div>

        @php
            $statusClasses = [
                'pending' => 'pending',
                'partial' => 'warning',
                'completed' => 'success',
                'failed' => 'danger'
            ];
            $statusIcons = [
                'pending' => 'clock',
                'partial' => 'exclamation-circle',
                'completed' => 'check-circle',
                'failed' => 'x-circle'
            ];
        @endphp

        <table class="list-table">
            <thead>
                <tr>
                    <th>{{ __('trip_number') }}</th>
                    <th>{{ __('customer_name') }}</th>
                    <th>{{ __('phone') }}</th>
                    <th>{{ __('vehicle') }}</th>
                    <th>{{ __('driver') }}</th>
                    <th>{{ __('delivered_at') }}</th>
                    <th>{{ __('status') }}</th>
                    <th>{{ __('actions') }}</th>
                </tr>
            </thead>
            <tbody>
                @forelse($deliveries as $delivery)
                <tr>
                    <td>
                        <span class="table-cell-id">
                            <a href="{{ route('admin.trips.show', $delivery->trip) }}">
                                {{ $delivery->trip->trip_number ?? 'N/A' }}
                            </a>
                        </span>
                    </td>
                    <td>
                        <div class="table-cell-main">
                            <strong>{{ $delivery->customer_name }}</strong>
                            @if($delivery->customer_email)
                                <span class="table-cell-sub">{{ $delivery->customer_email }}</span>
                            @endif
                        </div>
                    </td>
                    <td>
                        @if($delivery->customer_phone)
                            <span class="table-cell-phone">
                                <i class="bi bi-telephone"></i>
                                {{ $delivery->customer_phone }}
                            </span>
                        @else
                            <span class="text-muted">-</span>
                        @endif
                    </td>
                    <td>
                        <div class="table-cell-vehicle">
                            <i class="bi bi-truck"></i>
                            <span>{{ $delivery->trip->vehicle->vehicle_number ?? 'N/A' }}</span>
                        </div>
                    </td>
                    <td>
                        <div class="table-cell-user">
                            <div class="table-cell-user-avatar">
                                <i class="bi bi-person"></i>
                            </div>
                            <span>{{ $delivery->trip->driver->name ?? 'N/A' }}</span>
                        </div>
                    </td>
                    <td class="table-cell-date">
                        <i class="bi bi-calendar"></i>
                        {{ \Carbon\Carbon::parse($delivery->delivered_at)->format('d M Y, h:i A') }}
                    </td>
                    <td>
                        <span class="table-badge table-badge-{{ $statusClasses[$delivery->delivery_status] ?? 'pending' }}">
                            <i class="bi bi-{{ $statusIcons[$delivery->delivery_status] ?? 'circle' }}"></i>
                            {{ __(ucfirst($delivery->delivery_status)) }}
                        </span>
                    </td>
                    <td>
                        <div class="table-actions">
                            @if(auth()->user()->hasPermission('deliveries.view'))
                            <a href="{{ route('admin.deliveries.show', $delivery) }}" 
                               class="btn-table-action btn-table-view" 
                               title="{{ __('view') }}">
                                <i class="bi bi-eye"></i>
                            </a>
                            @endif

                            @if(auth()->user()->hasPermission('deliveries.edit'))
                            <a href="{{ route('admin.deliveries.edit', $delivery) }}" 
                               class="btn-table-action btn-table-edit" 
                               title="{{ __('edit') }}">
                                <i class="bi bi-pencil"></i>
                            </a>
                            @endif

                            @if(auth()->user()->hasPermission('deliveries.delete'))
                            <form action="{{ route('admin.deliveries.destroy', $delivery) }}" 
                                  method="POST" 
                                  class="d-inline"
                                  onsubmit="return confirm('{{ __('Are you sure you want to delete this delivery?') }}')">
                                @csrf
                                @method('DELETE')
                                <button type="submit" 
                                        class="btn-table-action btn-table-delete" 
                                        title="{{ __('delete') }}">
                                    <i class="bi bi-trash"></i>
                                </button>
                            </form>
                            @endif
                        </div>
                    </td>
                </tr>
                @empty
                <tr>
                    <td colspan="8">
                        <div class="list-empty-state">
                            <div class="list-empty-state-icon">
                                <i class="bi bi-clipboard-x"></i>
                            </div>
                            <h5 class="list-empty-state-title">{{ __('No Deliveries Found') }}</h5>
                            <p class="list-empty-state-text">{{ __('There are no deliveries matching your criteria. Create a new delivery to get started.') }}</p>
                            @if(auth()->user()->hasPermission('deliveries.create'))
                            <a href="{{ route('admin.deliveries.create') }}" class="btn-create">
                                <i class="bi bi-plus-lg"></i>
                                {{ __('add_delivery') }}
                            </a>
                            @endif
                        </div>
                    </td>
                </tr>
                @endforelse
            </tbody>
        </table>

        {{-- Pagination --}}
        @if($deliveries->hasPages())
        <div class="list-pagination">
            <div class="list-pagination-info">
                {{ __('Showing') }} {{ $deliveries->firstItem() }} {{ __('to') }} {{ $deliveries->lastItem() }} {{ __('of') }} {{ $deliveries->total() }} {{ __('entries') }}
            </div>
            <div class="list-pagination-nav">
                {{ $deliveries->links() }}
            </div>
        </div>
        @endif
    </div>
</div>
@endsection