@extends('layouts.app')

@section('title', __('Create Trip') . ' - ' . __('app_name'))

@push('styles')
<link rel="stylesheet" href="{{ asset('css/admincreatepage.css') }}">
@endpush

@section('content')
<div class="create-page-container">
    {{-- Page Header --}}
    <div class="create-page-header">
        <div class="create-page-header-content">
            <div class="create-page-breadcrumb">
                <a href="{{ route('admin.dashboard') }}">{{ __('Dashboard') }}</a>
                <i class="bi bi-chevron-right"></i>
                <a href="{{ route('admin.trips.index') }}">{{ __('Trips') }}</a>
                <i class="bi bi-chevron-right"></i>
                <span>{{ __('Create') }}</span>
            </div>
            <h1 class="create-page-title">
                <div class="create-page-title-icon">
                    <i class="bi bi-plus-lg"></i>
                </div>
                {{ __('Create New Trip') }}
            </h1>
            <p class="create-page-subtitle">{{ __('Assign a vehicle and driver to create a new trip') }}</p>
        </div>
        <div class="create-page-actions">
            <a href="{{ route('admin.trips.index') }}" class="create-btn create-btn-secondary">
                <i class="bi bi-arrow-left"></i>
                {{ __('Back to List') }}
            </a>
        </div>
    </div>

    {{-- Validation Errors --}}
    @if($errors->any())
        <div class="create-alert danger">
            <div class="create-alert-icon">
                <i class="bi bi-exclamation-triangle"></i>
            </div>
            <div class="create-alert-content">
                <div class="create-alert-title">{{ __('Please fix the following errors') }}</div>
                <div class="create-alert-text">
                    <ul>
                        @foreach($errors->all() as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                </div>
            </div>
        </div>
    @endif

    <form action="{{ route('admin.trips.store') }}" method="POST">
        @csrf

        <div class="create-form-layout">
            {{-- Main Form --}}
            <div class="create-form-main">
                {{-- Shipment Assignment Card --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-header-icon primary">
                            <i class="bi bi-box-seam"></i>
                        </div>
                        <div class="create-card-header-content">
                            <h5 class="create-card-title">{{ __('Shipment Assignment') }}</h5>
                            <p class="create-card-subtitle">{{ __('Optionally link this trip to a pending shipment') }}</p>
                        </div>
                    </div>
                    <div class="create-card-body">
                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-box-seam"></i>
                                    {{ __('Link to Shipment') }}
                                    <span class="create-form-optional">({{ __('Optional') }})</span>
                                </label>
                                <div class="create-input-icon-wrapper">
                                    <select name="shipment_id" 
                                            id="shipment_id"
                                            class="create-select @error('shipment_id') is-invalid @enderror">
                                        <option value="">{{ __('No Shipment (Manual Trip)') }}</option>
                                        @foreach($shipments as $shipment)
                                            <option value="{{ $shipment->id }}" 
                                                    data-vehicle-type="{{ $shipment->vehicle_type_id }}"
                                                    {{ old('shipment_id', request('shipment_id')) == $shipment->id ? 'selected' : '' }}>
                                                {{ $shipment->shipment_number }} - {{ $shipment->vehicleType->name }} 
                                                @if($shipment->cargo_weight)({{ $shipment->cargo_weight }}T)@endif
                                            </option>
                                        @endforeach
                                    </select>
                                </div>
                                @error('shipment_id')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                                <p class="create-form-hint">
                                    <i class="bi bi-lightbulb"></i>
                                    {{ __('Selecting a shipment will filter available vehicles by matching vehicle type') }}
                                </p>
                            </div>

                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-hash"></i>
                                    {{ __('Shipment Reference') }}
                                    <span class="create-form-optional">({{ __('Optional') }})</span>
                                </label>
                                <div class="create-input-icon-wrapper">
                                    <input type="text" 
                                           name="shipment_reference" 
                                           id="shipment_reference"
                                           class="create-input @error('shipment_reference') is-invalid @enderror"
                                           value="{{ old('shipment_reference') }}"
                                           placeholder="{{ __('Enter manual reference number') }}">
                                    <i class="bi bi-hash create-input-icon"></i>
                                </div>
                                @error('shipment_reference')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                                <p class="create-form-hint">
                                    <i class="bi bi-lightbulb"></i>
                                    {{ __('Optional reference for manual tracking') }}
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                {{-- Vehicle & Driver Card --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-header-icon info">
                            <i class="bi bi-truck"></i>
                        </div>
                        <div class="create-card-header-content">
                            <h5 class="create-card-title">{{ __('Vehicle & Driver Assignment') }}</h5>
                            <p class="create-card-subtitle">{{ __('Select the vehicle and driver for this trip') }}</p>
                        </div>
                    </div>
                    <div class="create-card-body">
                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-truck-front"></i>
                                    {{ __('Vehicle') }}
                                    <span class="create-form-required">*</span>
                                </label>
                                <select name="vehicle_id" 
                                        id="vehicle_id"
                                        class="create-select @error('vehicle_id') is-invalid @enderror"
                                        required>
                                    <option value="">{{ __('Select Vehicle') }}</option>
                                    @foreach($vehicles as $vehicle)
                                        @php
                                            $isInUse = in_array($vehicle->id, $vehiclesInUse);
                                            $tripNumber = $vehicleTrips[$vehicle->id] ?? '';
                                            $ownerDriverId = $vehicleOwners[$vehicle->id] ?? null;
                                        @endphp
                                        <option value="{{ $vehicle->id }}" 
                                                data-vehicle-type="{{ $vehicle->vehicle_type_id }}"
                                                data-owner-driver="{{ $ownerDriverId }}"
                                                {{ $isInUse ? 'disabled' : '' }}
                                                {{ old('vehicle_id') == $vehicle->id ? 'selected' : '' }}>
                                            {{ $vehicle->vehicle_number }} - {{ $vehicle->vehicleType->name ?? 'N/A' }}
                                            @if($isInUse) ({{ __('In Use') }}: {{ $tripNumber }}) @endif
                                            @if($ownerDriverId) [{{ __('Own Vehicle') }}] @endif
                                        </option>
                                    @endforeach
                                </select>
                                @error('vehicle_id')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>

                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-person"></i>
                                    {{ __('Driver') }}
                                    <span class="create-form-required">*</span>
                                </label>
                                <select name="driver_id" 
                                        id="driver_id"
                                        class="create-select @error('driver_id') is-invalid @enderror"
                                        required>
                                    <option value="">{{ __('Select Driver') }}</option>
                                    @foreach($drivers as $driver)
                                        @php
                                            $isOnTrip = in_array($driver->id, $driversOnTrip);
                                            $tripNumber = $driverTrips[$driver->id] ?? '';
                                            $ownVehicleId = $driverOwnVehicles[$driver->id] ?? null;
                                        @endphp
                                        <option value="{{ $driver->id }}" 
                                                data-driver-type="{{ $driver->driver_type }}"
                                                data-own-vehicle="{{ $ownVehicleId }}"
                                                {{ $isOnTrip ? 'disabled' : '' }}
                                                {{ old('driver_id') == $driver->id ? 'selected' : '' }}>
                                            {{ $driver->name }} ({{ $driver->mobile }})
                                            @if($driver->driver_type === 'own_vehicle') [{{ __('Own Vehicle') }}] @endif
                                            @if($isOnTrip) ({{ __('On Trip') }}: {{ $tripNumber }}) @endif
                                        </option>
                                    @endforeach
                                </select>
                                @error('driver_id')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>
                        </div>

                        {{-- Driver-Vehicle Compatibility Alert --}}
                        <div id="compatibility-alert" class="create-alert info" style="display: none;">
                            <div class="create-alert-icon">
                                <i class="bi bi-info-circle"></i>
                            </div>
                            <div class="create-alert-content">
                                <div class="create-alert-title">{{ __('Driver-Vehicle Relationship') }}</div>
                                <div class="create-alert-text" id="compatibility-message"></div>
                            </div>
                        </div>
                    </div>
                </div>

                {{-- Location Details Card --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-header-icon success">
                            <i class="bi bi-geo-alt"></i>
                        </div>
                        <div class="create-card-header-content">
                            <h5 class="create-card-title">{{ __('Location Details') }}</h5>
                            <p class="create-card-subtitle">{{ __('Specify pickup and delivery locations') }}</p>
                        </div>
                    </div>
                    <div class="create-card-body">
                        {{-- Location Type Toggle --}}
                        <div class="create-form-group">
                            <label class="create-form-label">
                                <i class="bi bi-pin-map"></i>
                                {{ __('Delivery Type') }}
                                <span class="create-form-required">*</span>
                            </label>
                            <div class="create-toggle-group">
                                <label class="create-toggle-option">
                                    <input type="radio" name="has_multiple_locations" value="0" 
                                           {{ old('has_multiple_locations', '0') == '0' ? 'checked' : '' }}>
                                    <span class="create-toggle-label">
                                        <i class="bi bi-geo"></i>
                                        {{ __('Single Drop Location') }}
                                    </span>
                                </label>
                                <label class="create-toggle-option">
                                    <input type="radio" name="has_multiple_locations" value="1" 
                                           {{ old('has_multiple_locations') == '1' ? 'checked' : '' }}>
                                    <span class="create-toggle-label">
                                        <i class="bi bi-pin-map"></i>
                                        {{ __('Multiple Drop Locations') }}
                                    </span>
                                </label>
                            </div>
                            <p class="create-form-hint">
                                <i class="bi bi-lightbulb"></i>
                                {{ __('For multiple locations, delivery addresses can be added after trip creation') }}
                            </p>
                        </div>

                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-geo-alt text-success"></i>
                                    {{ __('Pickup Location') }}
                                    <span class="create-form-required">*</span>
                                </label>
                                <textarea name="pickup_location" 
                                          id="pickup_location"
                                          class="create-textarea @error('pickup_location') is-invalid @enderror"
                                          rows="3"
                                          placeholder="{{ __('Enter complete pickup address') }}"
                                          required>{{ old('pickup_location') }}</textarea>
                                @error('pickup_location')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>

                            <div class="create-form-group" id="drop_location_group">
                                <label class="create-form-label">
                                    <i class="bi bi-geo-alt-fill text-danger"></i>
                                    {{ __('Drop Location') }}
                                    <span class="create-form-required" id="drop_required_asterisk">*</span>
                                </label>
                                <textarea name="drop_location" 
                                          id="drop_location"
                                          class="create-textarea @error('drop_location') is-invalid @enderror"
                                          rows="3"
                                          placeholder="{{ __('Enter complete delivery address') }}">{{ old('drop_location') }}</textarea>
                                @error('drop_location')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>

                {{-- Schedule & Instructions Card --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-header-icon warning">
                            <i class="bi bi-calendar-event"></i>
                        </div>
                        <div class="create-card-header-content">
                            <h5 class="create-card-title">{{ __('Schedule & Instructions') }}</h5>
                            <p class="create-card-subtitle">{{ __('Set trip dates and additional instructions') }}</p>
                        </div>
                    </div>
                    <div class="create-card-body">
                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-calendar-plus"></i>
                                    {{ __('Start Date') }}
                                    <span class="create-form-optional">({{ __('Optional') }})</span>
                                </label>
                                <div class="create-input-icon-wrapper">
                                    <input type="datetime-local" 
                                           name="start_date" 
                                           id="start_date"
                                           class="create-input @error('start_date') is-invalid @enderror"
                                           value="{{ old('start_date') }}">
                                    <i class="bi bi-calendar-plus create-input-icon"></i>
                                </div>
                                @error('start_date')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>

                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-calendar-check"></i>
                                    {{ __('Expected End Date') }}
                                    <span class="create-form-optional">({{ __('Optional') }})</span>
                                </label>
                                <div class="create-input-icon-wrapper">
                                    <input type="datetime-local" 
                                           name="end_date" 
                                           id="end_date"
                                           class="create-input @error('end_date') is-invalid @enderror"
                                           value="{{ old('end_date') }}">
                                    <i class="bi bi-calendar-check create-input-icon"></i>
                                </div>
                                @error('end_date')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>
                        </div>

                        <div class="create-form-group">
                            <label class="create-form-label">
                                <i class="bi bi-card-text"></i>
                                {{ __('Trip Instructions') }}
                                <span class="create-form-optional">({{ __('Optional') }})</span>
                            </label>
                            <textarea name="trip_instructions" 
                                      id="trip_instructions"
                                      class="create-textarea @error('trip_instructions') is-invalid @enderror"
                                      rows="4"
                                      placeholder="{{ __('Enter special instructions, notes, or requirements for this trip...') }}">{{ old('trip_instructions') }}</textarea>
                            @error('trip_instructions')
                                <div class="create-form-error">
                                    <i class="bi bi-exclamation-circle"></i>
                                    {{ $message }}
                                </div>
                            @enderror
                        </div>

                        {{-- Form Actions --}}
                        <div class="create-card-actions">
                            <div class="create-card-actions-left">
                                <a href="{{ route('admin.trips.index') }}" class="create-btn create-btn-secondary">
                                    <i class="bi bi-x-lg"></i>
                                    {{ __('Cancel') }}
                                </a>
                            </div>
                            <div class="create-card-actions-right">
                                <button type="submit" class="create-btn create-btn-primary">
                                    <i class="bi bi-save"></i>
                                    {{ __('Create Trip') }}
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            {{-- Sidebar --}}
            <div class="create-form-sidebar">
                {{-- Tips Card --}}
                <div class="create-sidebar-card">
                    <div class="create-sidebar-header">
                        <div class="create-sidebar-icon info">
                            <i class="bi bi-lightbulb"></i>
                        </div>
                        <h5 class="create-sidebar-title">{{ __('Quick Tips') }}</h5>
                    </div>
                    <div class="create-sidebar-body">
                        <ul class="create-tips-list">
                            <li>
                                <i class="bi bi-check-circle-fill"></i>
                                <span>{{ __('Link to a shipment for automatic vehicle type filtering') }}</span>
                            </li>
                            <li>
                                <i class="bi bi-check-circle-fill"></i>
                                <span>{{ __('Own-vehicle drivers can only be assigned to their specific vehicle') }}</span>
                            </li>
                            <li>
                                <i class="bi bi-check-circle-fill"></i>
                                <span>{{ __('Vehicles currently in use on other trips are disabled') }}</span>
                            </li>
                            <li>
                                <i class="bi bi-check-circle-fill"></i>
                                <span>{{ __('For multiple deliveries, add delivery addresses after trip creation') }}</span>
                            </li>
                        </ul>
                    </div>
                </div>

                {{-- Initial Status Card --}}
                <div class="create-sidebar-card">
                    <div class="create-sidebar-header">
                        <div class="create-sidebar-icon success">
                            <i class="bi bi-check-circle"></i>
                        </div>
                        <h5 class="create-sidebar-title">{{ __('Initial Status') }}</h5>
                    </div>
                    <div class="create-sidebar-body">
                        <div class="create-status-preview">
                            <span class="create-status-badge active">
                                <i class="bi bi-check-circle"></i>
                                {{ __('Assigned') }}
                            </span>
                            <p class="create-status-description">
                                {{ __('New trips are created with assigned status. The driver will be notified immediately.') }}
                            </p>
                        </div>
                    </div>
                </div>

                {{-- Quick Actions Card --}}
                <div class="create-sidebar-card">
                    <div class="create-sidebar-header">
                        <div class="create-sidebar-icon primary">
                            <i class="bi bi-lightning"></i>
                        </div>
                        <h5 class="create-sidebar-title">{{ __('Quick Actions') }}</h5>
                    </div>
                    <div class="create-sidebar-body">
                        <div class="create-quick-actions">
                            <a href="{{ route('admin.trips.index') }}" class="create-quick-action-btn">
                                <i class="bi bi-list-ul"></i>
                                {{ __('View All Trips') }}
                            </a>
                            <a href="{{ route('admin.vehicles.index') }}" class="create-quick-action-btn">
                                <i class="bi bi-truck"></i>
                                {{ __('Manage Vehicles') }}
                            </a>
                            <a href="{{ route('admin.drivers.index') }}" class="create-quick-action-btn">
                                <i class="bi bi-people"></i>
                                {{ __('Manage Drivers') }}
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </form>
</div>
@endsection

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const shipmentSelect = document.getElementById('shipment_id');
    const vehicleSelect = document.getElementById('vehicle_id');
    const driverSelect = document.getElementById('driver_id');
    const hasMultipleLocations = document.querySelectorAll('input[name="has_multiple_locations"]');
    const dropLocationGroup = document.getElementById('drop_location_group');
    const dropRequiredAsterisk = document.getElementById('drop_required_asterisk');
    const dropLocationInput = document.getElementById('drop_location');
    const compatibilityAlert = document.getElementById('compatibility-alert');
    const compatibilityMessage = document.getElementById('compatibility-message');

    // Vehicle owners map (vehicle_id => driver_id)
    const vehicleOwners = @json($vehicleOwners);
    // Driver own vehicles map (driver_id => vehicle_id)
    const driverOwnVehicles = @json($driverOwnVehicles);

    // Handle multiple locations toggle
    function toggleDropLocation() {
        const isMultiple = document.querySelector('input[name="has_multiple_locations"]:checked').value === '1';
        if (isMultiple) {
            dropLocationGroup.style.opacity = '0.5';
            dropLocationInput.removeAttribute('required');
            dropRequiredAsterisk.style.display = 'none';
            dropLocationInput.placeholder = '{{ __("Multiple locations - add after trip creation") }}';
        } else {
            dropLocationGroup.style.opacity = '1';
            dropLocationInput.setAttribute('required', 'required');
            dropRequiredAsterisk.style.display = 'inline';
            dropLocationInput.placeholder = '{{ __("Enter complete delivery address") }}';
        }
    }

    hasMultipleLocations.forEach(function(radio) {
        radio.addEventListener('change', toggleDropLocation);
    });

    // Initial toggle state
    toggleDropLocation();

    // Handle vehicle-driver compatibility
    function checkCompatibility() {
        const vehicleId = vehicleSelect.value;
        const driverId = driverSelect.value;

        if (!vehicleId || !driverId) {
            compatibilityAlert.style.display = 'none';
            return;
        }

        const ownerDriverId = vehicleOwners[vehicleId];
        const driverOwnVehicleId = driverOwnVehicles[driverId];

        // Check if vehicle has an owner
        if (ownerDriverId) {
            if (ownerDriverId == driverId) {
                compatibilityAlert.className = 'create-alert success';
                compatibilityMessage.innerHTML = '<i class="bi bi-check-circle me-1"></i> {{ __("Perfect match! This driver owns this vehicle.") }}';
                compatibilityAlert.style.display = 'flex';
            } else {
                compatibilityAlert.className = 'create-alert danger';
                compatibilityMessage.innerHTML = '<i class="bi bi-exclamation-triangle me-1"></i> {{ __("Warning: This vehicle belongs to another driver. Please select a different vehicle or driver.") }}';
                compatibilityAlert.style.display = 'flex';
            }
        } else if (driverOwnVehicleId) {
            // Driver has own vehicle but different vehicle selected
            if (driverOwnVehicleId != vehicleId) {
                compatibilityAlert.className = 'create-alert warning';
                compatibilityMessage.innerHTML = '<i class="bi bi-exclamation-circle me-1"></i> {{ __("This driver has their own vehicle. Consider selecting their assigned vehicle instead.") }}';
                compatibilityAlert.style.display = 'flex';
            } else {
                compatibilityAlert.style.display = 'none';
            }
        } else {
            compatibilityAlert.style.display = 'none';
        }
    }

    vehicleSelect.addEventListener('change', checkCompatibility);
    driverSelect.addEventListener('change', checkCompatibility);

    // Filter vehicles based on shipment vehicle type
    shipmentSelect.addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        const vehicleTypeId = selectedOption.dataset.vehicleType;

        Array.from(vehicleSelect.options).forEach(function(option) {
            if (option.value === '') return;
            
            const optionVehicleType = option.dataset.vehicleType;
            if (vehicleTypeId && optionVehicleType !== vehicleTypeId) {
                option.style.display = 'none';
            } else {
                option.style.display = '';
            }
        });

        // Reset vehicle selection if current selection is hidden
        if (vehicleSelect.selectedOptions[0]?.style.display === 'none') {
            vehicleSelect.value = '';
        }
    });
});
</script>
@endpush