@extends('layouts.app')

@section('title', __('Edit Trip') . ' - ' . __('app_name'))

@push('styles')
<link rel="stylesheet" href="{{ asset('css/admincreatepage.css') }}">
@endpush

@section('content')
<div class="create-page-container">
    {{-- Page Header --}}
    <div class="create-page-header">
        <div class="create-page-header-content">
            <div class="create-page-breadcrumb">
                <a href="{{ route('admin.dashboard') }}">{{ __('Dashboard') }}</a>
                <i class="bi bi-chevron-right"></i>
                <a href="{{ route('admin.trips.index') }}">{{ __('Trips') }}</a>
                <i class="bi bi-chevron-right"></i>
                <a href="{{ route('admin.trips.show', $trip) }}">{{ $trip->trip_number }}</a>
                <i class="bi bi-chevron-right"></i>
                <span>{{ __('Edit') }}</span>
            </div>
            <h1 class="create-page-title">
                <div class="create-page-title-icon">
                    <i class="bi bi-pencil"></i>
                </div>
                {{ __('Edit Trip') }}
            </h1>
            <p class="create-page-subtitle">{{ __('Update the trip details below') }}</p>
        </div>
        <div class="create-page-actions">
            <a href="{{ route('admin.trips.show', $trip) }}" class="create-btn create-btn-secondary">
                <i class="bi bi-arrow-left"></i>
                {{ __('Back to Details') }}
            </a>
        </div>
    </div>

    {{-- Validation Errors --}}
    @if($errors->any())
        <div class="create-alert danger">
            <div class="create-alert-icon">
                <i class="bi bi-exclamation-triangle"></i>
            </div>
            <div class="create-alert-content">
                <div class="create-alert-title">{{ __('Please fix the following errors') }}</div>
                <div class="create-alert-text">
                    <ul>
                        @foreach($errors->all() as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                </div>
            </div>
        </div>
    @endif

    <form action="{{ route('admin.trips.update', $trip) }}" method="POST">
        @csrf
        @method('PUT')

        <div class="create-form-layout">
            {{-- Main Form --}}
            <div class="create-form-main">
                {{-- Shipment Assignment Card --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-header-icon primary">
                            <i class="bi bi-box-seam"></i>
                        </div>
                        <div class="create-card-header-content">
                            <h5 class="create-card-title">{{ __('Shipment Assignment') }}</h5>
                            <p class="create-card-subtitle">{{ __('Optionally link this trip to a pending shipment') }}</p>
                        </div>
                    </div>
                    <div class="create-card-body">
                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-box-seam"></i>
                                    {{ __('Link to Shipment') }}
                                    <span class="create-form-optional">({{ __('Optional') }})</span>
                                </label>
                                <select name="shipment_id" 
                                        id="shipment_id"
                                        class="create-select @error('shipment_id') is-invalid @enderror">
                                    <option value="">{{ __('No Shipment (Manual Trip)') }}</option>
                                    @foreach($shipments as $shipment)
                                        <option value="{{ $shipment->id }}" 
                                                data-vehicle-type="{{ $shipment->vehicle_type_id }}"
                                                {{ old('shipment_id', $trip->shipment_id) == $shipment->id ? 'selected' : '' }}>
                                            {{ $shipment->shipment_number }} - {{ $shipment->vehicleType->name }} 
                                            @if($shipment->cargo_weight)({{ $shipment->cargo_weight }}T)@endif
                                        </option>
                                    @endforeach
                                </select>
                                @error('shipment_id')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                                <p class="create-form-hint">
                                    <i class="bi bi-lightbulb"></i>
                                    {{ __('Selecting a shipment will filter available vehicles by matching vehicle type') }}
                                </p>
                            </div>

                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-hash"></i>
                                    {{ __('Shipment Reference') }}
                                    <span class="create-form-optional">({{ __('Optional') }})</span>
                                </label>
                                <div class="create-input-icon-wrapper">
                                    <input type="text" 
                                           name="shipment_reference" 
                                           id="shipment_reference"
                                           class="create-input @error('shipment_reference') is-invalid @enderror"
                                           value="{{ old('shipment_reference', $trip->shipment_reference) }}"
                                           placeholder="{{ __('Enter manual reference number') }}">
                                    <i class="bi bi-hash create-input-icon"></i>
                                </div>
                                @error('shipment_reference')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                                <p class="create-form-hint">
                                    <i class="bi bi-lightbulb"></i>
                                    {{ __('Optional reference for manual tracking') }}
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                {{-- Vehicle & Driver Card --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-header-icon info">
                            <i class="bi bi-truck"></i>
                        </div>
                        <div class="create-card-header-content">
                            <h5 class="create-card-title">{{ __('Vehicle & Driver Assignment') }}</h5>
                            <p class="create-card-subtitle">{{ __('Update the vehicle and driver for this trip') }}</p>
                        </div>
                    </div>
                    <div class="create-card-body">
                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-truck-front"></i>
                                    {{ __('Vehicle') }}
                                    <span class="create-form-required">*</span>
                                </label>
                                <select name="vehicle_id" 
                                        id="vehicle_id"
                                        class="create-select @error('vehicle_id') is-invalid @enderror"
                                        required>
                                    <option value="">{{ __('Select Vehicle') }}</option>
                                    @foreach($vehicles as $vehicle)
                                        @php
                                            $isInUse = in_array($vehicle->id, $vehiclesInUse);
                                            $tripNumber = $vehicleTrips[$vehicle->id] ?? '';
                                            $ownerDriverId = $vehicleOwners[$vehicle->id] ?? null;
                                        @endphp
                                        <option value="{{ $vehicle->id }}" 
                                                data-vehicle-type="{{ $vehicle->vehicle_type_id }}"
                                                data-owner-driver="{{ $ownerDriverId }}"
                                                {{ $isInUse ? 'disabled' : '' }}
                                                {{ old('vehicle_id', $trip->vehicle_id) == $vehicle->id ? 'selected' : '' }}>
                                            {{ $vehicle->vehicle_number }} - {{ $vehicle->vehicleType->name ?? 'N/A' }}
                                            @if($isInUse) ({{ __('In Use') }}: {{ $tripNumber }}) @endif
                                            @if($ownerDriverId) [{{ __('Own Vehicle') }}] @endif
                                        </option>
                                    @endforeach
                                </select>
                                @error('vehicle_id')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>

                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-person"></i>
                                    {{ __('Driver') }}
                                    <span class="create-form-required">*</span>
                                </label>
                                <select name="driver_id" 
                                        id="driver_id"
                                        class="create-select @error('driver_id') is-invalid @enderror"
                                        required>
                                    <option value="">{{ __('Select Driver') }}</option>
                                    @foreach($drivers as $driver)
                                        @php
                                            $isOnTrip = in_array($driver->id, $driversOnTrip);
                                            $tripNumber = $driverTrips[$driver->id] ?? '';
                                            $ownVehicleId = $driverOwnVehicles[$driver->id] ?? null;
                                        @endphp
                                        <option value="{{ $driver->id }}" 
                                                data-driver-type="{{ $driver->driver_type }}"
                                                data-own-vehicle="{{ $ownVehicleId }}"
                                                {{ $isOnTrip ? 'disabled' : '' }}
                                                {{ old('driver_id', $trip->driver_id) == $driver->id ? 'selected' : '' }}>
                                            {{ $driver->name }} ({{ $driver->mobile }})
                                            @if($driver->driver_type === 'own_vehicle') [{{ __('Own Vehicle') }}] @endif
                                            @if($isOnTrip) ({{ __('On Trip') }}: {{ $tripNumber }}) @endif
                                        </option>
                                    @endforeach
                                </select>
                                @error('driver_id')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>
                        </div>

                        {{-- Driver-Vehicle Compatibility Alert --}}
                        <div id="compatibility-alert" class="create-alert info" style="display: none;">
                            <div class="create-alert-icon">
                                <i class="bi bi-info-circle"></i>
                            </div>
                            <div class="create-alert-content">
                                <div class="create-alert-title">{{ __('Driver-Vehicle Relationship') }}</div>
                                <div class="create-alert-text" id="compatibility-message"></div>
                            </div>
                        </div>
                    </div>
                </div>

                {{-- Location Details Card --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-header-icon success">
                            <i class="bi bi-geo-alt"></i>
                        </div>
                        <div class="create-card-header-content">
                            <h5 class="create-card-title">{{ __('Location Details') }}</h5>
                            <p class="create-card-subtitle">{{ __('Update pickup and delivery locations') }}</p>
                        </div>
                    </div>
                    <div class="create-card-body">
                        {{-- Location Type Toggle --}}
                        <div class="create-form-group">
                            <label class="create-form-label">
                                <i class="bi bi-pin-map"></i>
                                {{ __('Delivery Type') }}
                                <span class="create-form-required">*</span>
                            </label>
                            <div class="create-toggle-group">
                                <label class="create-toggle-option">
                                    <input type="radio" name="has_multiple_locations" value="0" 
                                           {{ old('has_multiple_locations', $trip->has_multiple_locations ? '1' : '0') == '0' ? 'checked' : '' }}>
                                    <span class="create-toggle-label">
                                        <i class="bi bi-geo"></i>
                                        {{ __('Single Drop Location') }}
                                    </span>
                                </label>
                                <label class="create-toggle-option">
                                    <input type="radio" name="has_multiple_locations" value="1" 
                                           {{ old('has_multiple_locations', $trip->has_multiple_locations ? '1' : '0') == '1' ? 'checked' : '' }}>
                                    <span class="create-toggle-label">
                                        <i class="bi bi-pin-map"></i>
                                        {{ __('Multiple Drop Locations') }}
                                    </span>
                                </label>
                            </div>
                            <p class="create-form-hint">
                                <i class="bi bi-lightbulb"></i>
                                {{ __('For multiple locations, manage delivery addresses in the deliveries section') }}
                            </p>
                        </div>

                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-geo-alt text-success"></i>
                                    {{ __('Pickup Location') }}
                                    <span class="create-form-required">*</span>
                                </label>
                                <textarea name="pickup_location" 
                                          id="pickup_location"
                                          class="create-textarea @error('pickup_location') is-invalid @enderror"
                                          rows="3"
                                          placeholder="{{ __('Enter complete pickup address') }}"
                                          required>{{ old('pickup_location', $trip->pickup_location) }}</textarea>
                                @error('pickup_location')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>

                            <div class="create-form-group" id="drop_location_group">
                                <label class="create-form-label">
                                    <i class="bi bi-geo-alt-fill text-danger"></i>
                                    {{ __('Drop Location') }}
                                    <span class="create-form-required" id="drop_required_asterisk">*</span>
                                </label>
                                <textarea name="drop_location" 
                                          id="drop_location"
                                          class="create-textarea @error('drop_location') is-invalid @enderror"
                                          rows="3"
                                          placeholder="{{ __('Enter complete delivery address') }}">{{ old('drop_location', $trip->drop_location) }}</textarea>
                                @error('drop_location')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>

                {{-- Schedule & Instructions Card --}}
                <div class="create-card">
                    <div class="create-card-header">
                        <div class="create-card-header-icon warning">
                            <i class="bi bi-calendar-event"></i>
                        </div>
                        <div class="create-card-header-content">
                            <h5 class="create-card-title">{{ __('Schedule & Instructions') }}</h5>
                            <p class="create-card-subtitle">{{ __('Update trip dates and instructions') }}</p>
                        </div>
                    </div>
                    <div class="create-card-body">
                        <div class="create-form-row">
                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-calendar-plus"></i>
                                    {{ __('Start Date') }}
                                    <span class="create-form-optional">({{ __('Optional') }})</span>
                                </label>
                                <div class="create-input-icon-wrapper">
                                    <input type="datetime-local" 
                                           name="start_date" 
                                           id="start_date"
                                           class="create-input @error('start_date') is-invalid @enderror"
                                           value="{{ old('start_date', $trip->start_date ? $trip->start_date->format('Y-m-d\TH:i') : '') }}">
                                    <i class="bi bi-calendar-plus create-input-icon"></i>
                                </div>
                                @error('start_date')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>

                            <div class="create-form-group">
                                <label class="create-form-label">
                                    <i class="bi bi-calendar-check"></i>
                                    {{ __('Expected End Date') }}
                                    <span class="create-form-optional">({{ __('Optional') }})</span>
                                </label>
                                <div class="create-input-icon-wrapper">
                                    <input type="datetime-local" 
                                           name="end_date" 
                                           id="end_date"
                                           class="create-input @error('end_date') is-invalid @enderror"
                                           value="{{ old('end_date', $trip->end_date ? $trip->end_date->format('Y-m-d\TH:i') : '') }}">
                                    <i class="bi bi-calendar-check create-input-icon"></i>
                                </div>
                                @error('end_date')
                                    <div class="create-form-error">
                                        <i class="bi bi-exclamation-circle"></i>
                                        {{ $message }}
                                    </div>
                                @enderror
                            </div>
                        </div>

                        {{-- Status Selection --}}
                        <div class="create-form-group">
                            <label class="create-form-label">
                                <i class="bi bi-flag"></i>
                                {{ __('Trip Status') }}
                                <span class="create-form-required">*</span>
                            </label>
                            <select name="status" 
                                    id="status"
                                    class="create-select @error('status') is-invalid @enderror"
                                    required>
                                <option value="pending" {{ old('status', $trip->status) === 'pending' ? 'selected' : '' }}>{{ __('Pending') }}</option>
                                <option value="assigned" {{ old('status', $trip->status) === 'assigned' ? 'selected' : '' }}>{{ __('Assigned') }}</option>
                                <option value="in_transit" {{ old('status', $trip->status) === 'in_transit' ? 'selected' : '' }}>{{ __('In Transit') }}</option>
                                <option value="delivered" {{ old('status', $trip->status) === 'delivered' ? 'selected' : '' }}>{{ __('Delivered') }}</option>
                                <option value="cancelled" {{ old('status', $trip->status) === 'cancelled' ? 'selected' : '' }}>{{ __('Cancelled') }}</option>
                            </select>
                            @error('status')
                                <div class="create-form-error">
                                    <i class="bi bi-exclamation-circle"></i>
                                    {{ $message }}
                                </div>
                            @enderror
                            <p class="create-form-hint">
                                <i class="bi bi-lightbulb"></i>
                                {{ __('Changing status may trigger notifications to the driver') }}
                            </p>
                        </div>

                        <div class="create-form-group">
                            <label class="create-form-label">
                                <i class="bi bi-card-text"></i>
                                {{ __('Trip Instructions') }}
                                <span class="create-form-optional">({{ __('Optional') }})</span>
                            </label>
                            <textarea name="trip_instructions" 
                                      id="trip_instructions"
                                      class="create-textarea @error('trip_instructions') is-invalid @enderror"
                                      rows="4"
                                      placeholder="{{ __('Enter special instructions, notes, or requirements for this trip...') }}">{{ old('trip_instructions', $trip->trip_instructions) }}</textarea>
                            @error('trip_instructions')
                                <div class="create-form-error">
                                    <i class="bi bi-exclamation-circle"></i>
                                    {{ $message }}
                                </div>
                            @enderror
                        </div>

                        {{-- Form Actions --}}
                        <div class="create-card-actions">
                            <div class="create-card-actions-left">
                                <a href="{{ route('admin.trips.show', $trip) }}" class="create-btn create-btn-secondary">
                                    <i class="bi bi-x-lg"></i>
                                    {{ __('Cancel') }}
                                </a>
                            </div>
                            <div class="create-card-actions-right">
                                <button type="submit" class="create-btn create-btn-primary">
                                    <i class="bi bi-save"></i>
                                    {{ __('Update Trip') }}
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            {{-- Sidebar --}}
            <div class="create-form-sidebar">
                {{-- Current Status Card --}}
                <div class="create-sidebar-card">
                    <div class="create-sidebar-header">
                        @php
                            $statusConfig = [
                                'pending' => ['icon' => 'clock', 'color' => 'warning'],
                                'assigned' => ['icon' => 'check-circle', 'color' => 'info'],
                                'in_transit' => ['icon' => 'truck', 'color' => 'primary'],
                                'delivered' => ['icon' => 'check-circle-fill', 'color' => 'success'],
                                'cancelled' => ['icon' => 'x-circle', 'color' => 'danger'],
                            ];
                            $config = $statusConfig[$trip->status] ?? ['icon' => 'question-circle', 'color' => 'secondary'];
                        @endphp
                        <div class="create-sidebar-icon {{ $config['color'] }}">
                            <i class="bi bi-{{ $config['icon'] }}"></i>
                        </div>
                        <h5 class="create-sidebar-title">{{ __('Current Status') }}</h5>
                    </div>
                    <div class="create-sidebar-body">
                        <div class="create-status-preview">
                            <span class="create-status-badge {{ $trip->status === 'delivered' ? 'active' : ($trip->status === 'pending' ? 'pending' : 'active') }}">
                                <i class="bi bi-{{ $config['icon'] }}"></i>
                                {{ __(ucfirst(str_replace('_', ' ', $trip->status))) }}
                            </span>
                            <p class="create-status-description">
                                {{ __('Created') }}: {{ $trip->created_at->format('d M Y, h:i A') }}
                            </p>
                        </div>
                    </div>
                </div>

                {{-- Trip Details Card --}}
                <div class="create-sidebar-card">
                    <div class="create-sidebar-header">
                        <div class="create-sidebar-icon primary">
                            <i class="bi bi-info-circle"></i>
                        </div>
                        <h5 class="create-sidebar-title">{{ __('Trip Details') }}</h5>
                    </div>
                    <div class="create-sidebar-body">
                        <ul class="create-tips-list">
                            <li>
                                <i class="bi bi-hash text-primary"></i>
                                <span><strong>{{ __('Trip #') }}:</strong> {{ $trip->trip_number }}</span>
                            </li>
                            <li>
                                <i class="bi bi-truck text-info"></i>
                                <span><strong>{{ __('Vehicle') }}:</strong> {{ $trip->vehicle->vehicle_number ?? '-' }}</span>
                            </li>
                            <li>
                                <i class="bi bi-person text-success"></i>
                                <span><strong>{{ __('Driver') }}:</strong> {{ $trip->driver->name ?? '-' }}</span>
                            </li>
                            @if($trip->creator)
                            <li>
                                <i class="bi bi-person-check text-warning"></i>
                                <span><strong>{{ __('Created By') }}:</strong> {{ $trip->creator->name }}</span>
                            </li>
                            @endif
                            @if($trip->updated_at)
                            <li>
                                <i class="bi bi-clock-history text-secondary"></i>
                                <span><strong>{{ __('Last Updated') }}:</strong> {{ $trip->updated_at->diffForHumans() }}</span>
                            </li>
                            @endif
                        </ul>
                    </div>
                </div>

                {{-- Quick Actions Card --}}
                <div class="create-sidebar-card">
                    <div class="create-sidebar-header">
                        <div class="create-sidebar-icon success">
                            <i class="bi bi-lightning"></i>
                        </div>
                        <h5 class="create-sidebar-title">{{ __('Quick Actions') }}</h5>
                    </div>
                    <div class="create-sidebar-body">
                        <div class="create-quick-actions">
                            <a href="{{ route('admin.trips.show', $trip) }}" class="create-quick-action-btn">
                                <i class="bi bi-eye"></i>
                                {{ __('View Details') }}
                            </a>
                            <a href="{{ route('admin.trips.index') }}" class="create-quick-action-btn">
                                <i class="bi bi-list-ul"></i>
                                {{ __('View All Trips') }}
                            </a>
                            @if($trip->shipment)
                            <a href="{{ route('admin.shipments.show', $trip->shipment) }}" class="create-quick-action-btn">
                                <i class="bi bi-box-seam"></i>
                                {{ __('View Shipment') }}
                            </a>
                            @endif
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </form>
</div>
@endsection

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const vehicleSelect = document.getElementById('vehicle_id');
    const driverSelect = document.getElementById('driver_id');
    const hasMultipleLocations = document.querySelectorAll('input[name="has_multiple_locations"]');
    const dropLocationGroup = document.getElementById('drop_location_group');
    const dropRequiredAsterisk = document.getElementById('drop_required_asterisk');
    const dropLocationInput = document.getElementById('drop_location');
    const compatibilityAlert = document.getElementById('compatibility-alert');
    const compatibilityMessage = document.getElementById('compatibility-message');

    // Vehicle owners map (vehicle_id => driver_id)
    const vehicleOwners = @json($vehicleOwners);
    // Driver own vehicles map (driver_id => vehicle_id)
    const driverOwnVehicles = @json($driverOwnVehicles);

    // Handle multiple locations toggle
    function toggleDropLocation() {
        const isMultiple = document.querySelector('input[name="has_multiple_locations"]:checked').value === '1';
        if (isMultiple) {
            dropLocationGroup.style.opacity = '0.5';
            dropLocationInput.removeAttribute('required');
            dropRequiredAsterisk.style.display = 'none';
            dropLocationInput.placeholder = '{{ __("Multiple locations - manage in deliveries") }}';
        } else {
            dropLocationGroup.style.opacity = '1';
            dropLocationInput.setAttribute('required', 'required');
            dropRequiredAsterisk.style.display = 'inline';
            dropLocationInput.placeholder = '{{ __("Enter complete delivery address") }}';
        }
    }

    hasMultipleLocations.forEach(function(radio) {
        radio.addEventListener('change', toggleDropLocation);
    });

    // Initial toggle state
    toggleDropLocation();

    // Handle vehicle-driver compatibility
    function checkCompatibility() {
        const vehicleId = vehicleSelect.value;
        const driverId = driverSelect.value;

        if (!vehicleId || !driverId) {
            compatibilityAlert.style.display = 'none';
            return;
        }

        const ownerDriverId = vehicleOwners[vehicleId];
        const driverOwnVehicleId = driverOwnVehicles[driverId];

        if (ownerDriverId) {
            if (ownerDriverId == driverId) {
                compatibilityAlert.className = 'create-alert success';
                compatibilityMessage.innerHTML = '<i class="bi bi-check-circle me-1"></i> {{ __("Perfect match! This driver owns this vehicle.") }}';
                compatibilityAlert.style.display = 'flex';
            } else {
                compatibilityAlert.className = 'create-alert danger';
                compatibilityMessage.innerHTML = '<i class="bi bi-exclamation-triangle me-1"></i> {{ __("Warning: This vehicle belongs to another driver.") }}';
                compatibilityAlert.style.display = 'flex';
            }
        } else if (driverOwnVehicleId && driverOwnVehicleId != vehicleId) {
            compatibilityAlert.className = 'create-alert warning';
            compatibilityMessage.innerHTML = '<i class="bi bi-exclamation-circle me-1"></i> {{ __("This driver has their own vehicle. Consider selecting their assigned vehicle instead.") }}';
            compatibilityAlert.style.display = 'flex';
        } else {
            compatibilityAlert.style.display = 'none';
        }
    }

    vehicleSelect.addEventListener('change', checkCompatibility);
    driverSelect.addEventListener('change', checkCompatibility);

    // Initial check on page load
    checkCompatibility();
});
</script>
@endpush