@extends('layouts.app')

@section('title', __('Trips') . ' - ' . __('app_name'))

@push('styles')
<link rel="stylesheet" href="{{ asset('css/adminlistpage.css') }}">
@endpush

@section('content')
<div class="container-fluid">
    {{-- Page Header --}}
    <div class="list-page-header">
        <div class="list-page-header-left">
            <h1 class="list-page-title">
                <i class="bi bi-geo-alt"></i>
                {{ __('Trips') }}
            </h1>
            <p class="list-page-subtitle">{{ __('Manage and track all your trips and routes') }}</p>
        </div>
        <div class="list-page-header-actions">
            <a href="{{ route('admin.trips.create') }}" class="btn-create">
                <i class="bi bi-plus-lg"></i>
                {{ __('New Trip') }}
            </a>
        </div>
    </div>

    {{-- Flash Messages --}}
    @if(session('success'))
        <div class="alert alert-success alert-dismissible fade show mb-3" role="alert">
            <i class="bi bi-check-circle me-2"></i>{{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    @if(session('error'))
        <div class="alert alert-danger alert-dismissible fade show mb-3" role="alert">
            <i class="bi bi-exclamation-triangle me-2"></i>{{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    {{-- Stats Bar --}}
    <div class="list-stats-bar">
        <div class="list-stat-card gradient-blue">
            <div class="list-stat-icon">
                <i class="bi bi-geo-alt"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $trips->total() }}</div>
                <div class="list-stat-label">{{ __('Total Trips') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-graph-up-arrow"></i>
                {{ __('All time') }}
            </div>
        </div>
        <div class="list-stat-card gradient-purple">
            <div class="list-stat-icon">
                <i class="bi bi-check-circle"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $trips->where('status', 'assigned')->count() }}</div>
                <div class="list-stat-label">{{ __('Assigned') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-clipboard-check"></i>
                {{ __('Ready') }}
            </div>
        </div>
        <div class="list-stat-card gradient-cyan">
            <div class="list-stat-icon">
                <i class="bi bi-truck"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $trips->where('status', 'in_transit')->count() }}</div>
                <div class="list-stat-label">{{ __('In Transit') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-geo-alt"></i>
                {{ __('On route') }}
            </div>
        </div>
        <div class="list-stat-card gradient-green">
            <div class="list-stat-icon">
                <i class="bi bi-check-all"></i>
            </div>
            <div class="list-stat-content">
                <div class="list-stat-value">{{ $trips->where('status', 'delivered')->count() }}</div>
                <div class="list-stat-label">{{ __('Delivered') }}</div>
            </div>
            <div class="list-stat-footer">
                <i class="bi bi-check2-all"></i>
                {{ __('Completed') }}
            </div>
        </div>
    </div>

    {{-- Filter Bar --}}
    <div class="list-filter-bar">
        <div class="list-filter-header">
            <div class="list-filter-title">
                <i class="bi bi-funnel"></i>
                {{ __('Filters') }}
            </div>
            @if(request()->hasAny(['search', 'status']))
                <a href="{{ route('admin.trips.index') }}" class="list-filter-clear">
                    <i class="bi bi-x-circle"></i>
                    {{ __('Clear Filters') }}
                </a>
            @endif
        </div>
        <form action="{{ route('admin.trips.index') }}" method="GET">
            <div class="list-filter-row">
                <div class="list-filter-group search-group">
                    <label>{{ __('Search') }}</label>
                    <div class="list-search-box">
                        <i class="bi bi-search"></i>
                        <input type="text" name="search" class="form-control" 
                               placeholder="{{ __('Search by trip number, reference...') }}"
                               value="{{ request('search') }}">
                    </div>
                </div>
                <div class="list-filter-group">
                    <label>{{ __('Status') }}</label>
                    <select name="status" class="form-select">
                        <option value="">{{ __('All Statuses') }}</option>
                        <option value="pending" {{ request('status') === 'pending' ? 'selected' : '' }}>{{ __('Pending') }}</option>
                        <option value="assigned" {{ request('status') === 'assigned' ? 'selected' : '' }}>{{ __('Assigned') }}</option>
                        <option value="in_transit" {{ request('status') === 'in_transit' ? 'selected' : '' }}>{{ __('In Transit') }}</option>
                        <option value="delivered" {{ request('status') === 'delivered' ? 'selected' : '' }}>{{ __('Delivered') }}</option>
                        <option value="cancelled" {{ request('status') === 'cancelled' ? 'selected' : '' }}>{{ __('Cancelled') }}</option>
                    </select>
                </div>
                <div class="list-filter-buttons">
                    <button type="submit" class="btn-filter btn-filter-primary">
                        <i class="bi bi-search"></i>
                        {{ __('Apply') }}
                    </button>
                </div>
            </div>
        </form>
    </div>

    {{-- Data Table --}}
    <div class="list-table-container">
        <div class="list-table-header">
            <h5 class="list-table-title">
                {{ __('Trips List') }}
                <span class="list-table-count">({{ $trips->total() }} {{ __('records') }})</span>
            </h5>
        </div>

        @if($trips->count() > 0)
            <div class="table-responsive">
                <table class="list-table">
                    <thead>
                        <tr>
                            <th>{{ __('Trip #') }}</th>
                            <th>{{ __('Vehicle') }}</th>
                            <th>{{ __('Driver') }}</th>
                            <th>{{ __('Pickup Location') }}</th>
                            <th>{{ __('Drop Location') }}</th>
                            <th>{{ __('Status') }}</th>
                            <th>{{ __('Created At') }}</th>
                            <th>{{ __('Actions') }}</th>
                        </tr>
                    </thead>
                    <tbody>
                        @php
                            $statusClasses = [
                                'pending' => 'pending',
                                'assigned' => 'assigned',
                                'in_transit' => 'in-transit',
                                'delivered' => 'delivered',
                                'cancelled' => 'cancelled'
                            ];
                            $statusIcons = [
                                'pending' => 'clock',
                                'assigned' => 'check-circle',
                                'in_transit' => 'truck',
                                'delivered' => 'check-all',
                                'cancelled' => 'x-circle'
                            ];
                        @endphp
                        @foreach($trips as $trip)
                            <tr>
                                <td class="table-cell-id">
                                    <a href="{{ route('admin.trips.show', $trip) }}">
                                        {{ $trip->trip_number }}
                                    </a>
                                </td>
                                <td>
                                    @if($trip->vehicle)
                                        <div class="table-cell-vehicle">
                                            <div class="table-cell-vehicle-icon">
                                                <i class="bi bi-truck"></i>
                                            </div>
                                            <div class="table-cell-vehicle-info">
                                                <span class="vehicle-number">{{ $trip->vehicle->vehicle_number }}</span>
                                                <span class="vehicle-type">{{ $trip->vehicle->vehicleType->name ?? '-' }}</span>
                                            </div>
                                        </div>
                                    @else
                                        <span class="text-muted">-</span>
                                    @endif
                                </td>
                                <td>
                                    @if($trip->driver)
                                        <div class="table-cell-user">
                                            <div class="table-cell-user-avatar">
                                                <i class="bi bi-person"></i>
                                            </div>
                                            <span>{{ $trip->driver->name }}</span>
                                        </div>
                                    @else
                                        <span class="text-muted">-</span>
                                    @endif
                                </td>
                                <td>
                                    <span class="table-cell-location" title="{{ $trip->pickup_location }}">
                                        <i class="bi bi-geo-alt text-success"></i>
                                        {{ Str::limit($trip->pickup_location, 30) }}
                                    </span>
                                </td>
                                <td>
                                    @if($trip->has_multiple_locations)
                                        <span class="table-badge table-badge-info">
                                            <i class="bi bi-pin-map"></i>
                                            {{ __('Multiple') }}
                                        </span>
                                    @elseif($trip->drop_location)
                                        <span class="table-cell-location" title="{{ $trip->drop_location }}">
                                            <i class="bi bi-geo-alt-fill text-danger"></i>
                                            {{ Str::limit($trip->drop_location, 30) }}
                                        </span>
                                    @else
                                        <span class="text-muted">-</span>
                                    @endif
                                </td>
                                <td>
                                    <span class="table-badge table-badge-{{ $statusClasses[$trip->status] ?? 'pending' }}">
                                        <i class="bi bi-{{ $statusIcons[$trip->status] ?? 'circle' }}"></i>
                                        {{ __(ucfirst(str_replace('_', ' ', $trip->status))) }}
                                    </span>
                                </td>
                                <td class="table-cell-date">
                                    <i class="bi bi-calendar"></i>
                                    {{ $trip->created_at->format('d M Y, h:i A') }}
                                </td>
                                <td>
                                    <div class="table-actions">
                                        <a href="{{ route('admin.trips.show', $trip) }}" 
                                           class="btn-table-action btn-table-view" 
                                           title="{{ __('View') }}">
                                            <i class="bi bi-eye"></i>
                                        </a>
                                        @if($trip->status === 'pending')
                                            <a href="{{ route('admin.trips.edit', $trip) }}" 
                                               class="btn-table-action btn-table-edit" 
                                               title="{{ __('Edit') }}">
                                                <i class="bi bi-pencil"></i>
                                            </a>
                                            <form action="{{ route('admin.trips.destroy', $trip) }}" 
                                                  method="POST" 
                                                  class="d-inline"
                                                  onsubmit="return confirm('{{ __('Are you sure you want to delete this trip?') }}')">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" 
                                                        class="btn-table-action btn-table-delete" 
                                                        title="{{ __('Delete') }}">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </form>
                                        @endif
                                        @if($trip->status === 'cancelled')
                                            <form action="{{ route('admin.trips.destroy', $trip) }}" 
                                                  method="POST" 
                                                  class="d-inline"
                                                  onsubmit="return confirm('{{ __('Are you sure you want to delete this cancelled trip?') }}')">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" 
                                                        class="btn-table-action btn-table-delete" 
                                                        title="{{ __('Delete') }}">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </form>
                                        @endif
                                    </div>
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>

            {{-- Pagination --}}
            @if($trips->hasPages())
            <div class="list-pagination">
                <div class="list-pagination-info">
                    {{ __('Showing') }} {{ $trips->firstItem() }} {{ __('to') }} {{ $trips->lastItem() }} {{ __('of') }} {{ $trips->total() }} {{ __('results') }}
                </div>
                <nav class="list-pagination-nav">
                    {{ $trips->links() }}
                </nav>
            </div>
            @endif
        @else
            {{-- Empty State --}}
            <div class="list-empty-state">
                <div class="list-empty-state-icon">
                    <i class="bi bi-geo-alt"></i>
                </div>
                <h4 class="list-empty-state-title">{{ __('No Trips Found') }}</h4>
                <p class="list-empty-state-text">
                    {{ __('There are no trips matching your criteria. Create a new trip to get started.') }}
                </p>
                <a href="{{ route('admin.trips.create') }}" class="btn-create">
                    <i class="bi bi-plus-lg"></i>
                    {{ __('Create First Trip') }}
                </a>
            </div>
        @endif
    </div>
</div>
@endsection